/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.bi.platform.commons.crypto.internal.keystore;

import com.ibm.bi.platform.commons.crypto.CAMCryptoException;
import com.ibm.bi.platform.commons.crypto.CAMCryptoRuntimeException;
import com.ibm.bi.platform.commons.crypto.CamCryptoMessageKeys;
import com.ibm.bi.platform.commons.crypto.internal.keystore.CAMKeyStoreJKS;
import com.ibm.bi.platform.commons.crypto.internal.keystore.CAMKeyStorePKCS12;
import com.ibm.bi.platform.commons.crypto.internal.utils.CertificatesUtils;
import com.ibm.bi.platform.commons.crypto.internal.utils.GenericUtils;
import com.ibm.bi.platform.commons.crypto.internal.utils.JcaFactory;
import com.ibm.bi.platform.commons.crypto.internal.utils.Validator;
import com.ibm.bi.platform.commons.messages.Message;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CAMKeyStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(CAMKeyStore.class);
    final String keyStorePersistenceFilePath;
    final char[] password;
    final List<Validator<PrivateKey>> privateKeyValidators = new ArrayList<Validator<PrivateKey>>();
    final List<Validator<Certificate[]>> certChainValidators = new ArrayList<Validator<Certificate[]>>();
    final List<Validator<KeyPair>> keyPairValidators = new ArrayList<Validator<KeyPair>>();
    final List<Validator<Certificate>> certificateValidators = new ArrayList<Validator<Certificate>>();
    private KeyStore keyStore;

    protected CAMKeyStore(String keyStorePersistenceFilePath, char[] password) throws CAMCryptoException {
        this.keyStorePersistenceFilePath = keyStorePersistenceFilePath;
        this.password = password;
    }

    public static CAMKeyStore newInstance(String keyStoreType, String keyStoreFilePath, char[] password) throws CAMCryptoException {
        switch (keyStoreType.toLowerCase()) {
            case "pkcs12": {
                return new CAMKeyStorePKCS12(keyStoreFilePath, password);
            }
            case "jks": {
                return new CAMKeyStoreJKS(keyStoreFilePath, password);
            }
        }
        throw new IllegalArgumentException("Key store type " + keyStoreType + " not supported");
    }

    protected void init(String type) throws CAMCryptoException {
        KeyStore keyStore = null;
        FileInputStream keystoreSource = null;
        try {
            keyStore = JcaFactory.getInstance(KeyStore.class, type);
            File persistenceFile = new File(this.keyStorePersistenceFilePath);
            if (persistenceFile.exists()) {
                keystoreSource = FileUtils.openInputStream(persistenceFile);
            }
            keyStore.load(keystoreSource, CAMKeyStore.passwordCopy(this.password));
            this.keyStore = keyStore;
        }
        catch (JcaFactory.CouldNotGetInstanceException | IOException | NoSuchAlgorithmException | CertificateException e) {
            try {
                throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_keystore, this.keyStorePersistenceFilePath));
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(keystoreSource);
                throw throwable;
            }
        }
        IOUtils.closeQuietly(keystoreSource);
        assert (this.keyStore != null);
        LOGGER.info("{} keystore loaded successfully from {}", (Object)type, (Object)this.keyStorePersistenceFilePath);
    }

    private static char[] passwordCopy(char[] password) {
        return Arrays.copyOf(password, password.length);
    }

    public void addPrivateKeyWithCertChain(String alias, PrivateKey privateKey, Certificate[] certChain) throws CAMCryptoException {
        Objects.requireNonNull(alias);
        Objects.requireNonNull(privateKey);
        Objects.requireNonNull(certChain);
        if (certChain.length == 0) {
            throw new CAMCryptoException(CamCryptoMessageKeys.certificate_chain_should_not_be_empty.buildMessage());
        }
        this.assertAliasIsLowerCase(alias);
        Certificate targetCert = certChain[0];
        Certificate[] chainWithoutTheAnchor = this.chainWithoutTheAnchorIfNotOnlyTarget(certChain);
        this.validatePrivateKey(privateKey);
        this.validateKeyPair(privateKey, targetCert);
        this.validateCertChain(chainWithoutTheAnchor);
        try {
            this.keyStore.deleteEntry(alias);
            this.keyStore.setKeyEntry(alias, privateKey, CAMKeyStore.passwordCopy(this.password), certChain);
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while adding private key", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_keystore, this.keyStorePersistenceFilePath));
        }
    }

    private void assertAliasIsLowerCase(String alias) {
        if (!alias.toLowerCase().equals(alias)) {
            throw new IllegalArgumentException("The alias should be lower case");
        }
    }

    private Certificate[] chainWithoutTheAnchorIfNotOnlyTarget(Certificate[] certChain) {
        if (certChain.length == 1) {
            return certChain;
        }
        return (Certificate[])Arrays.stream(certChain).limit(certChain.length - 1).toArray(Certificate[]::new);
    }

    private void validatePrivateKey(PrivateKey privateKey) throws CAMCryptoException {
        Validator.ValidatorResult validatorResult = Validator.fromConjunction(this.privateKeyValidators).validate(privateKey);
        if (!validatorResult.isValid) {
            throw new CAMCryptoException(this.firstMessage(validatorResult.messages).orElse(new Message(CamCryptoMessageKeys.private_key_not_valid)));
        }
    }

    private void validateKeyPair(PrivateKey privateKey, Certificate targetCert) throws CAMCryptoException {
        KeyPair keyPair = new KeyPair(targetCert.getPublicKey(), privateKey);
        Validator.ValidatorResult validatorResult = Validator.fromConjunction(this.keyPairValidators).validate(keyPair);
        if (!validatorResult.isValid) {
            LOGGER.debug("Private key don't correspond to target cert {}", (Object)CertificatesUtils.certToString(targetCert));
            throw new CAMCryptoException(this.firstMessage(validatorResult.messages).orElse(new Message(CamCryptoMessageKeys.private_key_do_not_match_public_key)));
        }
    }

    private void validateCertChain(Certificate[] certChain) throws CAMCryptoException {
        Validator.ValidatorResult validatorResult = Validator.fromConjunction(this.certChainValidators).validate(certChain);
        if (!validatorResult.isValid) {
            LOGGER.debug("Cert chain is not valid {}", (Object)CertificatesUtils.certChainToString(certChain));
            throw new CAMCryptoException(this.firstMessage(validatorResult.messages).orElse(new Message(CamCryptoMessageKeys.cert_chain_not_valid)));
        }
    }

    private Optional<Message> firstMessage(List<Message> messages) {
        if (messages != null) {
            return messages.stream().findFirst();
        }
        return Optional.empty();
    }

    public void addTrustedCertificate(String alias, Certificate certificate) throws CAMCryptoException {
        this.assertAliasIsLowerCase(alias);
        this.validateCertificate(certificate);
        try {
            this.keyStore.setCertificateEntry(alias, certificate);
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while adding a trusted certificate", e);
            throw new CAMCryptoException(e, CamCryptoMessageKeys.could_not_add_trusted_certificate.buildMessage());
        }
    }

    private void validateCertificate(Certificate certificate) throws CAMCryptoException {
        Validator.ValidatorResult validatorResult = Validator.fromConjunction(this.certificateValidators).validate(certificate);
        if (!validatorResult.isValid) {
            LOGGER.debug("Certificate {} is not valid", (Object)CertificatesUtils.certToString(certificate));
            throw new CAMCryptoException(this.firstMessage(validatorResult.messages).orElse(CamCryptoMessageKeys.certificate_not_valid.buildMessage(GenericUtils.supplierToOptional(() -> ((X509Certificate)certificate).getSubjectX500Principal().getName()).orElse("_"))));
        }
    }

    public void deleteTrustedCertificate(Certificate certificate) throws CAMCryptoException {
        try {
            Enumeration<String> aliases = this.keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String currentAlias = aliases.nextElement();
                if (!this.keyStore.isCertificateEntry(currentAlias) || !this.keyStore.getCertificate(currentAlias).equals(certificate)) continue;
                this.keyStore.deleteEntry(currentAlias);
                return;
            }
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while deleting a trusted certificate", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_delete_trusted_certificate));
        }
    }

    public PrivateKey getPrivateKey(String alias) throws CAMCryptoException {
        try {
            return (PrivateKey)this.keyStore.getKey(alias, CAMKeyStore.passwordCopy(this.password));
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting private key", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_keystore_general));
        }
        catch (UnrecoverableKeyException e) {
            LOGGER.error("Exception occurred while getting private key", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_private_key_wrong_password));
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.error("Exception occurred while getting private key", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_private_key));
        }
    }

    public Certificate[] getCertificateChain(String alias) throws CAMCryptoException {
        try {
            return this.keyStore.getCertificateChain(alias);
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting certificate chain", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_certificates));
        }
    }

    public Set<X509Certificate> getTrustedCertificates() throws CAMCryptoException {
        HashSet<X509Certificate> trustedCertificates = null;
        try {
            trustedCertificates = new HashSet<X509Certificate>();
            Enumeration<String> aliases = this.keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String currentAlias = aliases.nextElement();
                if (!this.keyStore.isCertificateEntry(currentAlias)) continue;
                trustedCertificates.add((X509Certificate)this.keyStore.getCertificate(currentAlias));
            }
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting trusted certificates", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_certificates));
        }
        return trustedCertificates;
    }

    public Certificate getTrustedCertificate(String alias) throws CAMCryptoException {
        try {
            if (this.keyStore.isCertificateEntry(alias)) {
                return this.keyStore.getCertificate(alias);
            }
            return null;
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting trusted certificate", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_certificates));
        }
    }

    public Certificate getTargetCertificate(String alias) throws CAMCryptoException {
        try {
            if (!this.keyStore.isCertificateEntry(alias)) {
                return this.keyStore.getCertificate(alias);
            }
            return null;
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting target certificate", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_certificates));
        }
    }

    public void persist() throws CAMCryptoException {
        FileOutputStream persistenceStream = null;
        try {
            File persistenceFile = new File(this.keyStorePersistenceFilePath);
            if (persistenceFile.exists()) {
                persistenceFile.delete();
                persistenceFile.createNewFile();
            }
            persistenceStream = FileUtils.openOutputStream(persistenceFile);
            this.keyStore.store(persistenceStream, CAMKeyStore.passwordCopy(this.password));
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            try {
                LOGGER.error("Exception occurred while persisting the keystore", e);
                throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_store_keystore, this.keyStorePersistenceFilePath));
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(persistenceStream);
                throw throwable;
            }
        }
        IOUtils.closeQuietly(persistenceStream);
    }

    public void clear() throws CAMCryptoException {
        this.aliases().stream().forEach(alias -> {
            try {
                this.keyStore.deleteEntry((String)alias);
            }
            catch (KeyStoreException e) {
                throw new CAMCryptoRuntimeException(new CAMCryptoException(e, CamCryptoMessageKeys.could_not_clear_keystore.buildMessage()));
            }
        });
    }

    public List<String> aliases() throws CAMCryptoException {
        try {
            return Collections.list(this.keyStore.aliases());
        }
        catch (KeyStoreException e) {
            LOGGER.error("Exception occurred while getting the aliases", e);
            throw new CAMCryptoException(e, new Message(CamCryptoMessageKeys.could_not_load_keystore_general));
        }
    }

    public void addPrivateKeyValidators(Validator<PrivateKey> ... privateKeyValidators) {
        this.privateKeyValidators.addAll(Arrays.asList(privateKeyValidators));
    }

    public void addCertChainValidators(Validator<Certificate[]> ... certChainValidators) {
        this.certChainValidators.addAll(Arrays.asList(certChainValidators));
    }

    public void addCertificateValidators(Validator<Certificate> ... CertificatesValidators) {
        this.certificateValidators.addAll(Arrays.asList(CertificatesValidators));
    }

    public void addKeyPairValidators(Validator<KeyPair> ... keyPairValidators) {
        this.keyPairValidators.addAll(Arrays.asList(keyPairValidators));
    }
}

