/*
 * Decompiled with CFR 0.152.
 */
package edu.stanford.nlp.coref.statistical;

import edu.stanford.nlp.coref.CorefProperties;
import edu.stanford.nlp.coref.CorefRules;
import edu.stanford.nlp.coref.CorefUtils;
import edu.stanford.nlp.coref.data.CorefCluster;
import edu.stanford.nlp.coref.data.Dictionaries;
import edu.stanford.nlp.coref.data.Document;
import edu.stanford.nlp.coref.data.Mention;
import edu.stanford.nlp.coref.md.RuleBasedCorefMentionFinder;
import edu.stanford.nlp.coref.statistical.CompressedFeatureVector;
import edu.stanford.nlp.coref.statistical.Compressor;
import edu.stanford.nlp.coref.statistical.DocumentExamples;
import edu.stanford.nlp.coref.statistical.Example;
import edu.stanford.nlp.coref.statistical.StatisticalCorefTrainer;
import edu.stanford.nlp.io.IOUtils;
import edu.stanford.nlp.ling.CoreAnnotations;
import edu.stanford.nlp.ling.CoreLabel;
import edu.stanford.nlp.ling.IndexedWord;
import edu.stanford.nlp.semgraph.SemanticGraphEdge;
import edu.stanford.nlp.stats.ClassicCounter;
import edu.stanford.nlp.stats.Counter;
import edu.stanford.nlp.trees.Tree;
import edu.stanford.nlp.util.Pair;
import edu.stanford.nlp.util.StringUtils;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Set;

public class FeatureExtractor {
    private static final int MIN_WORD_COUNT = 20;
    private static final int BIN_EXACT = 10;
    private static final double BIN_EXPONENT = 1.5;
    private static final Map<Integer, String> SINGLETON_FEATURES = new HashMap<Integer, String>();
    private final Dictionaries dictionaries;
    private final Set<String> vocabulary;
    private final Compressor<String> compressor;
    private final boolean useConstituencyParse;
    private final boolean useDocSource;
    private static final Set<String> PROPERS;

    public FeatureExtractor(Properties props, Dictionaries dictionaries, Compressor<String> compressor) {
        this(props, dictionaries, compressor, StatisticalCorefTrainer.wordCountsFile);
    }

    public FeatureExtractor(Properties props, Dictionaries dictionaries, Compressor<String> compressor, String wordCountsPath) {
        this(props, dictionaries, compressor, FeatureExtractor.loadVocabulary(wordCountsPath));
    }

    public FeatureExtractor(Properties props, Dictionaries dictionaries, Compressor<String> compressor, Set<String> vocabulary) {
        this.dictionaries = dictionaries;
        this.compressor = compressor;
        this.vocabulary = vocabulary;
        this.useDocSource = CorefProperties.conll(props);
        this.useConstituencyParse = CorefProperties.useConstituencyParse(props);
    }

    private static Set<String> loadVocabulary(String wordCountsPath) {
        HashSet<String> vocabulary = new HashSet<String>();
        try {
            Counter counts = (Counter)IOUtils.readObjectFromURLOrClasspathOrFileSystem(wordCountsPath);
            for (Map.Entry e : counts.entrySet()) {
                if (!(e.getValue() > 20.0)) continue;
                vocabulary.add((String)e.getKey());
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Error loading word counts", e);
        }
        return vocabulary;
    }

    public DocumentExamples extract(int id, Document document, Map<Pair<Integer, Integer>, Boolean> labeledPairs) {
        return this.extract(id, document, labeledPairs, this.compressor);
    }

    public DocumentExamples extract(int id, Document document, Map<Pair<Integer, Integer>, Boolean> labeledPairs, Compressor<String> compressor) {
        List<Mention> mentionsList = CorefUtils.getSortedMentions(document);
        HashMap<Integer, List<Mention>> mentionsByHeadIndex = new HashMap<Integer, List<Mention>>();
        for (Mention m : mentionsList) {
            ArrayList<Mention> withIndex = (ArrayList<Mention>)mentionsByHeadIndex.get(m.headIndex);
            if (withIndex == null) {
                withIndex = new ArrayList<Mention>();
                mentionsByHeadIndex.put(m.headIndex, withIndex);
            }
            withIndex.add(m);
        }
        Map<Integer, Mention> mentions = document.predictedMentionsByID;
        ArrayList<Example> examples = new ArrayList<Example>();
        HashSet<Integer> mentionsToExtract = new HashSet<Integer>();
        for (Map.Entry<Pair<Integer, Integer>, Boolean> pair : labeledPairs.entrySet()) {
            Mention m1 = mentions.get(pair.getKey().first);
            Mention m2 = mentions.get(pair.getKey().second);
            mentionsToExtract.add(m1.mentionID);
            mentionsToExtract.add(m2.mentionID);
            CompressedFeatureVector features = compressor.compress(this.getFeatures(document, m1, m2));
            examples.add(new Example(id, m1, m2, pair.getValue() != false ? 1.0 : 0.0, features));
        }
        HashMap<Integer, CompressedFeatureVector> mentionFeatures = new HashMap<Integer, CompressedFeatureVector>();
        Iterator iterator = mentionsToExtract.iterator();
        while (iterator.hasNext()) {
            int mentionID = (Integer)iterator.next();
            mentionFeatures.put(mentionID, compressor.compress(this.getFeatures(document, document.predictedMentionsByID.get(mentionID), mentionsByHeadIndex)));
        }
        return new DocumentExamples(id, examples, mentionFeatures);
    }

    private Counter<String> getFeatures(Document doc, Mention m, Map<Integer, List<Mention>> mentionsByHeadIndex) {
        ClassicCounter<String> features = new ClassicCounter<String>();
        features.incrementCount("mention-type=" + (Object)((Object)m.mentionType));
        features.incrementCount("gender=" + (Object)((Object)m.gender));
        features.incrementCount("person-fine=" + (Object)((Object)m.person));
        features.incrementCount("head-ne-type=" + m.nerString);
        ArrayList<String> singletonFeatures = m.getSingletonFeatures(this.dictionaries);
        for (Map.Entry<Integer, String> e : SINGLETON_FEATURES.entrySet()) {
            if (e.getKey() >= singletonFeatures.size()) continue;
            features.incrementCount(e.getValue() + "=" + (String)singletonFeatures.get(e.getKey()));
        }
        FeatureExtractor.addNumeric(features, "mention-length", m.spanToString().length());
        FeatureExtractor.addNumeric(features, "mention-words", m.originalSpan.size());
        FeatureExtractor.addNumeric(features, "sentence-words", m.sentenceWords.size());
        features.incrementCount("sentence-words=" + FeatureExtractor.bin(m.sentenceWords.size()));
        features.incrementCount("mention-position", (double)m.mentionNum / (double)doc.predictedMentions.size());
        features.incrementCount("sentence-position", (double)m.sentNum / (double)doc.numSentences);
        CoreLabel firstWord = FeatureExtractor.firstWord(m);
        CoreLabel lastWord = FeatureExtractor.lastWord(m);
        CoreLabel headWord = FeatureExtractor.headWord(m);
        CoreLabel prevWord = FeatureExtractor.prevWord(m);
        CoreLabel nextWord = FeatureExtractor.nextWord(m);
        CoreLabel prevprevWord = FeatureExtractor.prevprevWord(m);
        CoreLabel nextnextWord = FeatureExtractor.nextnextWord(m);
        String headPOS = FeatureExtractor.getPOS(headWord);
        String firstPOS = FeatureExtractor.getPOS(firstWord);
        String lastPOS = FeatureExtractor.getPOS(lastWord);
        String prevPOS = FeatureExtractor.getPOS(prevWord);
        String nextPOS = FeatureExtractor.getPOS(nextWord);
        String prevprevPOS = FeatureExtractor.getPOS(prevprevWord);
        String nextnextPOS = FeatureExtractor.getPOS(nextnextWord);
        features.incrementCount("first-word=" + this.wordIndicator(firstWord, firstPOS));
        features.incrementCount("last-word=" + this.wordIndicator(lastWord, lastPOS));
        features.incrementCount("head-word=" + this.wordIndicator(headWord, headPOS));
        features.incrementCount("next-word=" + this.wordIndicator(nextWord, nextPOS));
        features.incrementCount("prev-word=" + this.wordIndicator(prevWord, prevPOS));
        features.incrementCount("next-bigram=" + this.wordIndicator(nextWord, nextnextWord, nextPOS + "_" + nextnextPOS));
        features.incrementCount("prev-bigram=" + this.wordIndicator(prevprevWord, prevWord, prevprevPOS + "_" + prevPOS));
        features.incrementCount("next-pos=" + nextPOS);
        features.incrementCount("prev-pos=" + prevPOS);
        features.incrementCount("first-pos=" + firstPOS);
        features.incrementCount("last-pos=" + lastPOS);
        features.incrementCount("next-pos-bigram=" + nextPOS + "_" + nextnextPOS);
        features.incrementCount("prev-pos-bigram=" + prevprevPOS + "_" + prevPOS);
        this.addDependencyFeatures(features, "parent", FeatureExtractor.getDependencyParent(m), true);
        FeatureExtractor.addFeature(features, "ends-with-head", m.headIndex == m.endIndex - 1);
        FeatureExtractor.addFeature(features, "is-generic", m.originalSpan.size() == 1 && firstPOS.equals("NNS"));
        IndexedWord w = m.headIndexedWord;
        String depPath = "";
        int depth = 0;
        while (w != null) {
            SemanticGraphEdge e = FeatureExtractor.getDependencyParent(m, w);
            if (++depth <= 3 && e != null) {
                depPath = depPath + (depPath.isEmpty() ? "" : "_") + e.getRelation().toString();
                features.incrementCount("dep-path=" + depPath);
                w = e.getSource();
                continue;
            }
            w = null;
        }
        if (this.useConstituencyParse) {
            int fullEmbeddingLevel = this.headEmbeddingLevel(m.contextParseTree, m.headIndex);
            int mentionEmbeddingLevel = this.headEmbeddingLevel(m.mentionSubTree, m.headIndex - m.startIndex);
            if (fullEmbeddingLevel != -1 && mentionEmbeddingLevel != -1) {
                features.incrementCount("mention-embedding-level=" + FeatureExtractor.bin(fullEmbeddingLevel - mentionEmbeddingLevel));
                features.incrementCount("head-embedding-level=" + FeatureExtractor.bin(mentionEmbeddingLevel));
            } else {
                features.incrementCount("undetermined-embedding-level");
            }
            features.incrementCount("num-embedded-nps=" + FeatureExtractor.bin(this.numEmbeddedNps(m.mentionSubTree)));
            String syntaxPath = "";
            Tree tree = m.contextParseTree;
            Tree head = ((Tree)tree.getLeaves().get(m.headIndex)).ancestor(1, tree);
            depth = 0;
            for (Tree node : tree.pathNodeToNode(head, tree)) {
                syntaxPath = syntaxPath + node.value() + "-";
                features.incrementCount("syntax-path=" + syntaxPath);
                if (++depth < 4 && !node.value().equals("S")) continue;
                break;
            }
        }
        FeatureExtractor.addFeature(features, "contained-in-other-mention", mentionsByHeadIndex.get(m.headIndex).stream().anyMatch(m2 -> m != m2 && m.insideIn((Mention)m2)));
        FeatureExtractor.addFeature(features, "contains-other-mention", mentionsByHeadIndex.get(m.headIndex).stream().anyMatch(m2 -> m != m2 && m2.insideIn(m)));
        FeatureExtractor.addFeature(features, "bare-plural", m.originalSpan.size() == 1 && headPOS.equals("NNS"));
        FeatureExtractor.addFeature(features, "quantifier-start", this.dictionaries.quantifiers.contains(firstWord.word().toLowerCase()));
        FeatureExtractor.addFeature(features, "negative-start", firstWord.word().toLowerCase().matches("none|no|nothing|not"));
        FeatureExtractor.addFeature(features, "partitive", RuleBasedCorefMentionFinder.partitiveRule(m, m.sentenceWords, this.dictionaries));
        FeatureExtractor.addFeature(features, "adjectival-demonym", this.dictionaries.isAdjectivalDemonym(m.spanToString()));
        if (doc.docType != Document.DocType.ARTICLE && m.person == Dictionaries.Person.YOU && nextWord != null && nextWord.word().equalsIgnoreCase("know")) {
            features.incrementCount("generic-you");
        }
        return features;
    }

    private Counter<String> getFeatures(Document doc, Mention m1, Mention m2) {
        assert (m1.appearEarlierThan(m2));
        ClassicCounter<String> features = new ClassicCounter<String>();
        features.incrementCount("bias");
        if (this.useDocSource) {
            features.incrementCount("doc-type=" + (Object)((Object)doc.docType));
            if (doc.docInfo != null && doc.docInfo.containsKey("DOC_ID")) {
                features.incrementCount("doc-source=" + doc.docInfo.get("DOC_ID").split("/")[1]);
            }
        }
        ArrayList<String> singletonFeatures1 = m1.getSingletonFeatures(this.dictionaries);
        ArrayList<String> singletonFeatures2 = m2.getSingletonFeatures(this.dictionaries);
        for (Map.Entry<Integer, String> e : SINGLETON_FEATURES.entrySet()) {
            if (e.getKey() >= singletonFeatures1.size() || e.getKey() >= singletonFeatures2.size()) continue;
            features.incrementCount(e.getValue() + "=" + (String)singletonFeatures1.get(e.getKey()) + "_" + (String)singletonFeatures2.get(e.getKey()));
        }
        SemanticGraphEdge p1 = FeatureExtractor.getDependencyParent(m1);
        SemanticGraphEdge p2 = FeatureExtractor.getDependencyParent(m2);
        features.incrementCount("dep-relations=" + (p1 == null ? "null" : p1.getRelation()) + "_" + (p2 == null ? "null" : p2.getRelation()));
        features.incrementCount("roles=" + FeatureExtractor.getRole(m1) + "_" + FeatureExtractor.getRole(m2));
        CoreLabel headCL1 = FeatureExtractor.headWord(m1);
        CoreLabel headCL2 = FeatureExtractor.headWord(m2);
        String headPOS1 = FeatureExtractor.getPOS(headCL1);
        String headPOS2 = FeatureExtractor.getPOS(headCL2);
        features.incrementCount("head-pos-s=" + headPOS1 + "_" + headPOS2);
        features.incrementCount("head-words=" + this.wordIndicator("h_" + headCL1.word().toLowerCase() + "_" + headCL2.word().toLowerCase(), headPOS1 + "_" + headPOS2));
        FeatureExtractor.addFeature(features, "animacies-agree", m2.animaciesAgree(m1));
        FeatureExtractor.addFeature(features, "attributes-agree", m2.attributesAgree(m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "entity-types-agree", m2.entityTypesAgree(m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "numbers-agree", m2.numbersAgree(m1));
        FeatureExtractor.addFeature(features, "genders-agree", m2.gendersAgree(m1));
        FeatureExtractor.addFeature(features, "ner-strings-equal", m1.nerString.equals(m2.nerString));
        FeatureExtractor.addFeature(features, "antecedent-head-in-anaphor", FeatureExtractor.headContainedIn(m1, m2));
        FeatureExtractor.addFeature(features, "anaphor-head-in-antecedent", FeatureExtractor.headContainedIn(m2, m1));
        if (m1.mentionType != Dictionaries.MentionType.PRONOMINAL && m2.mentionType != Dictionaries.MentionType.PRONOMINAL) {
            FeatureExtractor.addFeature(features, "antecedent-in-anaphor", m2.spanToString().toLowerCase().contains(m1.spanToString().toLowerCase()));
            FeatureExtractor.addFeature(features, "anaphor-in-antecedent", m1.spanToString().toLowerCase().contains(m2.spanToString().toLowerCase()));
            FeatureExtractor.addFeature(features, "heads-equal", m1.headString.equalsIgnoreCase(m2.headString));
            FeatureExtractor.addFeature(features, "heads-agree", m2.headsAgree(m1));
            FeatureExtractor.addFeature(features, "exact-match", m1.toString().trim().toLowerCase().equals(m2.toString().trim().toLowerCase()));
            FeatureExtractor.addFeature(features, "partial-match", FeatureExtractor.relaxedStringMatch(m1, m2));
            double editDistance = (double)StringUtils.editDistance(m1.spanToString(), m2.spanToString()) / (double)(m1.spanToString().length() + m2.spanToString().length());
            features.incrementCount("edit-distance", editDistance);
            features.incrementCount("edit-distance=" + (double)((int)(editDistance * 10.0)) / 10.0);
            double headEditDistance = (double)StringUtils.editDistance(m1.headString, m2.headString) / (double)(m1.headString.length() + m2.headString.length());
            features.incrementCount("head-edit-distance", headEditDistance);
            features.incrementCount("head-edit-distance=" + (double)((int)(headEditDistance * 10.0)) / 10.0);
        }
        FeatureExtractor.addNumeric(features, "mention-distance", m2.mentionNum - m1.mentionNum);
        FeatureExtractor.addNumeric(features, "sentence-distance", m2.sentNum - m1.sentNum);
        if (m2.sentNum == m1.sentNum) {
            FeatureExtractor.addNumeric(features, "word-distance", m2.startIndex - m1.endIndex);
            if (m1.endIndex > m2.startIndex) {
                features.incrementCount("spans-intersect");
            }
        }
        HashSet<Mention> ms1 = new HashSet<Mention>();
        ms1.add(m1);
        HashSet<Mention> ms2 = new HashSet<Mention>();
        ms2.add(m2);
        Random r = new Random();
        CorefCluster c1 = new CorefCluster(20000 + r.nextInt(10000), ms1);
        CorefCluster c2 = new CorefCluster(10000 + r.nextInt(10000), ms2);
        String s2 = m2.lowercaseNormalizedSpanString();
        String s1 = m1.lowercaseNormalizedSpanString();
        FeatureExtractor.addFeature(features, "mention-speaker-PER0", ((String)m2.headWord.get(CoreAnnotations.SpeakerAnnotation.class)).equalsIgnoreCase("PER0"));
        FeatureExtractor.addFeature(features, "antecedent-is-anaphor-speaker", CorefRules.antecedentIsMentionSpeaker(doc, m2, m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "same-speaker", CorefRules.entitySameSpeaker(doc, m2, m1));
        FeatureExtractor.addFeature(features, "person-disagree-same-speaker", CorefRules.entityPersonDisagree(doc, m2, m1, this.dictionaries) && CorefRules.entitySameSpeaker(doc, m2, m1));
        FeatureExtractor.addFeature(features, "antecedent-matches-anaphor-speaker", CorefRules.antecedentMatchesMentionSpeakerAnnotation(m2, m1, doc));
        FeatureExtractor.addFeature(features, "discourse-you-PER0", m2.person == Dictionaries.Person.YOU && doc.docType == Document.DocType.ARTICLE && ((String)m2.headWord.get(CoreAnnotations.SpeakerAnnotation.class)).equals("PER0"));
        FeatureExtractor.addFeature(features, "speaker-match-i-i", m2.number == Dictionaries.Number.SINGULAR && this.dictionaries.firstPersonPronouns.contains(s1) && m1.number == Dictionaries.Number.SINGULAR && this.dictionaries.firstPersonPronouns.contains(s2) && CorefRules.entitySameSpeaker(doc, m2, m1));
        FeatureExtractor.addFeature(features, "speaker-match-speaker-i", m2.number == Dictionaries.Number.SINGULAR && this.dictionaries.firstPersonPronouns.contains(s2) && CorefRules.antecedentIsMentionSpeaker(doc, m2, m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "speaker-match-i-speaker", m1.number == Dictionaries.Number.SINGULAR && this.dictionaries.firstPersonPronouns.contains(s1) && CorefRules.antecedentIsMentionSpeaker(doc, m1, m2, this.dictionaries));
        FeatureExtractor.addFeature(features, "speaker-match-you-you", this.dictionaries.secondPersonPronouns.contains(s1) && this.dictionaries.secondPersonPronouns.contains(s2) && CorefRules.entitySameSpeaker(doc, m2, m1));
        FeatureExtractor.addFeature(features, "discourse-between-two-person", (m2.person == Dictionaries.Person.I && m1.person == Dictionaries.Person.YOU || m2.person == Dictionaries.Person.YOU && m1.person == Dictionaries.Person.I) && (Integer)m2.headWord.get(CoreAnnotations.UtteranceAnnotation.class) - (Integer)m1.headWord.get(CoreAnnotations.UtteranceAnnotation.class) == 1 && doc.docType == Document.DocType.CONVERSATION);
        FeatureExtractor.addFeature(features, "incompatible-not-match", m1.person != Dictionaries.Person.I && m2.person != Dictionaries.Person.I && (CorefRules.antecedentIsMentionSpeaker(doc, m1, m2, this.dictionaries) || CorefRules.antecedentIsMentionSpeaker(doc, m2, m1, this.dictionaries)));
        int utteranceDist = Math.abs((Integer)m1.headWord.get(CoreAnnotations.UtteranceAnnotation.class) - (Integer)m2.headWord.get(CoreAnnotations.UtteranceAnnotation.class));
        if (doc.docType != Document.DocType.ARTICLE && utteranceDist == 1 && !CorefRules.entitySameSpeaker(doc, m2, m1)) {
            FeatureExtractor.addFeature(features, "speaker-mismatch-i-i", m1.person == Dictionaries.Person.I && m2.person == Dictionaries.Person.I);
            FeatureExtractor.addFeature(features, "speaker-mismatch-you-you", m1.person == Dictionaries.Person.YOU && m2.person == Dictionaries.Person.YOU);
            FeatureExtractor.addFeature(features, "speaker-mismatch-we-we", m1.person == Dictionaries.Person.WE && m2.person == Dictionaries.Person.WE);
        }
        String firstWord1 = FeatureExtractor.firstWord(m1).word().toLowerCase();
        FeatureExtractor.addFeature(features, "indefinite-article-np", m1.appositions == null && m1.predicateNominatives == null && (firstWord1.equals("a") || firstWord1.equals("an")));
        FeatureExtractor.addFeature(features, "far-this", m2.lowercaseNormalizedSpanString().equals("this") && Math.abs(m2.sentNum - m1.sentNum) > 3);
        FeatureExtractor.addFeature(features, "per0-you-in-article", m2.person == Dictionaries.Person.YOU && doc.docType == Document.DocType.ARTICLE && ((String)m2.headWord.get(CoreAnnotations.SpeakerAnnotation.class)).equals("PER0"));
        FeatureExtractor.addFeature(features, "inside-in", m2.insideIn(m1) || m1.insideIn(m2));
        FeatureExtractor.addFeature(features, "indefinite-determiners", this.dictionaries.indefinitePronouns.contains(m1.originalSpan.get(0).lemma()) || this.dictionaries.indefinitePronouns.contains(m2.originalSpan.get(0).lemma()));
        FeatureExtractor.addFeature(features, "entity-attributes-agree", CorefRules.entityAttributesAgree(c2, c1));
        FeatureExtractor.addFeature(features, "entity-token-distance", CorefRules.entityTokenDistance(m2, m1));
        FeatureExtractor.addFeature(features, "i-within-i", CorefRules.entityIWithinI(m2, m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "exact-string-match", CorefRules.entityExactStringMatch(c2, c1, this.dictionaries, doc.roleSet));
        FeatureExtractor.addFeature(features, "entity-relaxed-heads-agree", CorefRules.entityRelaxedHeadsAgreeBetweenMentions(c2, c1, m2, m1));
        FeatureExtractor.addFeature(features, "is-acronym", CorefRules.entityIsAcronym(doc, c2, c1));
        FeatureExtractor.addFeature(features, "demonym", m2.isDemonym(m1, this.dictionaries));
        FeatureExtractor.addFeature(features, "incompatible-modifier", CorefRules.entityHaveIncompatibleModifier(m2, m1));
        FeatureExtractor.addFeature(features, "head-lemma-match", m1.headWord.lemma().equals(m2.headWord.lemma()));
        FeatureExtractor.addFeature(features, "words-included", CorefRules.entityWordsIncluded(c2, c1, m2, m1));
        FeatureExtractor.addFeature(features, "extra-proper-noun", CorefRules.entityHaveExtraProperNoun(m2, m1, new HashSet<String>()));
        FeatureExtractor.addFeature(features, "number-in-later-mentions", CorefRules.entityNumberInLaterMention(m2, m1));
        FeatureExtractor.addFeature(features, "sentence-context-incompatible", CorefRules.sentenceContextIncompatible(m2, m1, this.dictionaries));
        if (this.useConstituencyParse) {
            if (m1.sentNum == m2.sentNum) {
                int clauseCount = 0;
                Tree tree = m2.contextParseTree;
                Tree current = m2.mentionSubTree;
                do {
                    if (!(current = current.ancestor(1, tree)).label().value().startsWith("S")) continue;
                    ++clauseCount;
                } while (!current.dominates(m1.mentionSubTree) && !current.label().value().equals("ROOT") && current.ancestor(1, tree) != null);
                features.incrementCount("clause-count", clauseCount);
                features.incrementCount("clause-count=" + FeatureExtractor.bin(clauseCount));
            }
            if (RuleBasedCorefMentionFinder.isPleonastic(m2, m2.contextParseTree) || RuleBasedCorefMentionFinder.isPleonastic(m1, m1.contextParseTree)) {
                features.incrementCount("pleonastic-it");
            }
            if (this.maximalNp(m1.mentionSubTree) == this.maximalNp(m2.mentionSubTree)) {
                features.incrementCount("same-maximal-np");
            }
            boolean m1Embedded = this.headEmbeddingLevel(m1.mentionSubTree, m1.headIndex - m1.startIndex) > 1;
            boolean m2Embedded = this.headEmbeddingLevel(m2.mentionSubTree, m2.headIndex - m2.startIndex) > 1;
            features.incrementCount("embedding=" + m1Embedded + "_" + m2Embedded);
        }
        return features;
    }

    private static void addNumeric(Counter<String> features, String key, int value) {
        features.incrementCount(key + "=" + FeatureExtractor.bin(value));
        features.incrementCount(key, value);
    }

    public static boolean relaxedStringMatch(Mention m1, Mention m2) {
        Set<String> propers = FeatureExtractor.getPropers(m1);
        propers.retainAll(FeatureExtractor.getPropers(m2));
        return !propers.isEmpty();
    }

    private static Set<String> getPropers(Mention m) {
        HashSet<String> propers = new HashSet<String>();
        for (int i = m.startIndex; i < m.endIndex; ++i) {
            CoreLabel cl = m.sentenceWords.get(i);
            String POS = (String)cl.get(CoreAnnotations.PartOfSpeechAnnotation.class);
            String word = cl.word().toLowerCase();
            if (!PROPERS.contains(POS)) continue;
            propers.add(word);
        }
        return propers;
    }

    private static void addFeature(Counter<String> features, String name, boolean value) {
        if (value) {
            features.incrementCount(name);
        }
    }

    private static String bin(int value) {
        return FeatureExtractor.bin(value, 10, 1.5, Integer.MAX_VALUE);
    }

    private static String bin(int value, int binExact, double binExponent, int cap) {
        if (value < 0) {
            return "-" + FeatureExtractor.bin(-value);
        }
        if (value > cap) {
            return cap + "+";
        }
        String bin = String.valueOf(value);
        if (value > binExact) {
            double start = Math.pow(binExponent, (int)(Math.log(value) / Math.log(binExponent)));
            bin = (int)start + "-" + (int)(start * binExponent);
        }
        return bin;
    }

    private static String getRole(Mention m) {
        if (m.isSubject) {
            return "subject";
        }
        if (m.isDirectObject) {
            return "direct-object";
        }
        if (m.isIndirectObject) {
            return "indirect-object";
        }
        if (m.isPrepositionObject) {
            return "preposition-object";
        }
        return "unknown";
    }

    private static SemanticGraphEdge getDependencyParent(Mention m) {
        return FeatureExtractor.getDependencyParent(m, m.headIndexedWord);
    }

    private static SemanticGraphEdge getDependencyParent(Mention m, IndexedWord w) {
        Iterator<SemanticGraphEdge> iterator = m.enhancedDependency.incomingEdgeIterator(w);
        return iterator.hasNext() ? iterator.next() : null;
    }

    private void addDependencyFeatures(Counter<String> features, String prefix, SemanticGraphEdge e, boolean addWord) {
        if (e == null) {
            features.incrementCount("no-" + prefix);
            return;
        }
        IndexedWord parent = e.getSource();
        String parentPOS = parent.tag();
        String parentWord = parent.word();
        String parentRelation = e.getRelation().toString();
        if (addWord) {
            features.incrementCount(prefix + "-word=" + this.wordIndicator(parentWord, parentPOS));
        }
        features.incrementCount(prefix + "-POS=" + parentPOS);
        features.incrementCount(prefix + "-relation=" + parentRelation);
    }

    public Tree maximalNp(Tree mentionSubTree) {
        Tree maximalSubtree = mentionSubTree;
        for (Tree subtree : mentionSubTree.postOrderNodeList()) {
            String label;
            if (subtree.isLeaf() || subtree.isPreTerminal() || !(label = (String)((CoreLabel)subtree.label()).get(CoreAnnotations.ValueAnnotation.class)).equals("NP")) continue;
            maximalSubtree = subtree;
        }
        return maximalSubtree;
    }

    private int numEmbeddedNps(Tree mentionSubTree) {
        int embeddedNps = 0;
        for (Tree subtree : mentionSubTree.postOrderNodeList()) {
            String label;
            if (subtree.isLeaf() || subtree.isPreTerminal() || !(label = (String)((CoreLabel)subtree.label()).get(CoreAnnotations.ValueAnnotation.class)).equals("NP")) continue;
            ++embeddedNps;
        }
        return embeddedNps;
    }

    private int headEmbeddingLevel(Tree tree, int headIndex) {
        int embeddingLevel = 0;
        try {
            Tree subtree = (Tree)tree.getLeaves().get(headIndex);
            while (subtree != null) {
                String label = (String)((CoreLabel)subtree.label()).get(CoreAnnotations.ValueAnnotation.class);
                subtree = subtree.ancestor(1, tree);
                if (!label.equals("NP")) continue;
                ++embeddingLevel;
            }
        }
        catch (Exception e) {
            return -1;
        }
        return embeddingLevel;
    }

    private static boolean headContainedIn(Mention m1, Mention m2) {
        String head = m1.headString;
        for (CoreLabel cl : m2.originalSpan) {
            if (!head.equals(cl.word().toLowerCase())) continue;
            return true;
        }
        return false;
    }

    private String wordIndicator(CoreLabel cl1, CoreLabel cl2, String POS) {
        String w1 = cl1 == null ? "NONE" : cl1.word().toLowerCase();
        String w2 = cl2 == null ? "NONE" : cl2.word().toLowerCase();
        return this.wordIndicator(w1 + "_" + w2, POS);
    }

    private String wordIndicator(CoreLabel cl, String POS) {
        if (cl == null) {
            return "NONE";
        }
        return this.wordIndicator(cl.word().toLowerCase(), POS);
    }

    private String wordIndicator(String word, String POS) {
        if (word == null) {
            return "NONE";
        }
        return this.vocabulary.contains(word) ? word : POS;
    }

    private static String getPOS(CoreLabel cl) {
        return cl == null ? "NONE" : (String)cl.get(CoreAnnotations.PartOfSpeechAnnotation.class);
    }

    private static CoreLabel firstWord(Mention m) {
        return m.originalSpan.get(0);
    }

    private static CoreLabel headWord(Mention m) {
        return m.headWord;
    }

    private static CoreLabel lastWord(Mention m) {
        return m.originalSpan.get(m.originalSpan.size() - 1);
    }

    private static CoreLabel nextnextWord(Mention m) {
        return m.endIndex + 1 < m.sentenceWords.size() ? m.sentenceWords.get(m.endIndex + 1) : null;
    }

    private static CoreLabel nextWord(Mention m) {
        return m.endIndex < m.sentenceWords.size() ? m.sentenceWords.get(m.endIndex) : null;
    }

    private static CoreLabel prevWord(Mention m) {
        return m.startIndex > 0 ? m.sentenceWords.get(m.startIndex - 1) : null;
    }

    private static CoreLabel prevprevWord(Mention m) {
        return m.startIndex > 1 ? m.sentenceWords.get(m.startIndex - 2) : null;
    }

    static {
        SINGLETON_FEATURES.put(2, "animacy");
        SINGLETON_FEATURES.put(3, "person-coarse");
        SINGLETON_FEATURES.put(4, "number");
        SINGLETON_FEATURES.put(5, "position");
        SINGLETON_FEATURES.put(6, "relation");
        SINGLETON_FEATURES.put(7, "quantification");
        SINGLETON_FEATURES.put(8, "modifiers");
        SINGLETON_FEATURES.put(9, "negation");
        SINGLETON_FEATURES.put(10, "modal");
        SINGLETON_FEATURES.put(11, "attitude");
        SINGLETON_FEATURES.put(12, "coordination");
        PROPERS = new HashSet<String>();
        PROPERS.add("NN");
        PROPERS.add("NNS");
        PROPERS.add("NNP");
        PROPERS.add("NNPS");
    }
}

