/*
 * IBM Confidential
 * OCO Source Materials
 * IBM Cognos Products: XQE
 * (C) Copyright IBM Corp. 2009, 2015
 * The source code for this program is not published or otherwise divested of its trade secrets,
 * irrespective of what has been deposited with the U.S. Copyright Office.
 */
package udf.samples;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.NumberFormat;
import java.util.Locale;

import com.cognos.xqe.function.udf.sdk.Table;
import com.cognos.xqe.util.LocaleConverter;
import com.cognos.xqeapi.jdbc.DQJDBCDriver;

/**
 * Table function to enumerate locale information.
 */
public class Locales {

	/**
	 * Constructor.
	 */
	public Locales() {
	}

	/**
	 * Table function - Enumerate locale information.
	 * @return ResultSet
	 */
	public static ResultSet enumerateLocales() {
		return new LocalesTable();
	}

	/**
	 * Table function - Enumerate locale information for given language.
	 * @param language language of interest
	 * @return ResultSet
	 */
	public static ResultSet enumerateLocales(String language) {
		return new LocalesTable(language);
	}

	/**
	 * Release resources.
	 */
	public void release() {
	}

	/**
	 * Iterator class.
	 */
	private static class LocalesTable extends Table {

		/**
		 * Current row index.
		 */
		private int index;

		/**
		 * All available locales.
		 */
		private Locale[] locales;

		/**
		 * Language parameter (optional).
		 */
		private String language;

		/**
		 * Result row.
		 */
		private Object[] result = new Object[5];

		/**
		 * Constructor.
		 */
		public LocalesTable() {
			locales = NumberFormat.getAvailableLocales();
		}

		/**
		 * Constructor.
		 * @param pLanguage language
		 */
		public LocalesTable(String pLanguage) {
			language = pLanguage;
			locales = NumberFormat.getAvailableLocales();
		}

		/**
		 * Move to next row.
		 * @return <i>true</i> if more data, <i>false</i> otherwise
		 */
		public boolean next() {
			Locale locale;

			do {
				if (index == locales.length) {
					return false;
				}

				locale = locales[index++];
			} while (!matchesCriteria(locale));

			setValue(0, locale.getDisplayCountry());
			setValue(1, locale.getDisplayLanguage());
			setValue(2, locale.getCountry());
			setValue(3, LocaleConverter.getLanguage(locale));
			setValue(4, NumberFormat.getInstance(locale).getCurrency().toString());

			return true;
		}

		/**
		 * Determine if criteria matched.
		 */
		private boolean matchesCriteria(Locale locale) {
			boolean result;

			if(locale.getDisplayCountry().length() == 0) {
				result = false;
			} else if(language != null && !locale.getDisplayLanguage().equals(language)) {
				result = false;
			} else {
				result = true;
			}

			return result;
		}

		/**
		 * Get designated column's value.
		 * @param index column index
		 * @return column object
		 */
		public String getString(int index) {
			return (String) result[index - 1];
		}

		/**
		 * Get designated column's value.
		 * @param index column index
		 * @return column object
		 */
		public Object getObject(int index) {
			return result[index - 1];
		}

		/**
		 * @return
		 * @throws SQLException
		 * @see java.sql.ResultSet#wasNull()
		 */
		public boolean wasNull() throws SQLException {
			return false;
		}

		/**
		 * Set column value.
		 * @param i - column index
		 * @param value - column value
		 */
		private void setValue(int i, String value) {
			if (value.length() == 0) {
				value = null;
			}

			result[i] = value;
		}

		/**
		 * {@inheritDoc}
		 */
		public <T> T getObject(int columnIndex, Class<T> type) throws SQLException {
			throw new SQLException(DQJDBCDriver.METHOD_NOT_SUPPORTED);
		}

		/**
		 * {@inheritDoc}
		 */
		public <T> T getObject(String columnLabel, Class<T> type) throws SQLException {
			throw new SQLException(DQJDBCDriver.METHOD_NOT_SUPPORTED);
		}
	}
}
