/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.neo.dataimport.storage.connection;

import com.ibm.neo.dataimport.api.WAStorageException;
import com.ibm.neo.dataimport.storage.connection.DefaultConnection;
import com.ibm.neo.dataimport.storage.connection.IConnectionFactory;
import com.ibm.neo.dataimport.storage.connection.IConnectionPool;
import com.ibm.neo.dataimport.storage.connection.IConnectionSelector;
import com.ibm.neo.dataimport.storage.connection.IExpirationPolicy;
import com.ibm.neo.dataimport.storage.connection.IPooledConnection;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultConnectionPool
implements IConnectionPool {
    private final String mPoolName;
    private final IConnectionFactory mFactory;
    private final IConnectionSelector mSelector;
    private final IExpirationPolicy mDefaultExpirationPolicy;
    private final int mBorrowTimeoutSeconds;
    private final AtomicBoolean mShutdown = new AtomicBoolean(false);
    private final ConcurrentLinkedQueue<DefaultConnection> mConnections = new ConcurrentLinkedQueue();
    private final Semaphore mBorrowPermits;
    private final AtomicLong mCreationCount = new AtomicLong(0L);
    private final AtomicLong mReuseCount = new AtomicLong(0L);
    private final AtomicLong mInvalidCount = new AtomicLong(0L);
    private final AtomicLong mExpiredCount = new AtomicLong(0L);
    private final ScheduledFuture<?> mMaintenanceFuture;
    private final Logger mLogger = LoggerFactory.getLogger(this.getClass());

    public DefaultConnectionPool(String poolName, IConnectionFactory factory, IConnectionSelector selector, IExpirationPolicy defaultExpirationPolicy, ScheduledExecutorService scheduler) {
        this(poolName, factory, selector, defaultExpirationPolicy, 64, 60, scheduler, 30);
    }

    public DefaultConnectionPool(String poolName, IConnectionFactory factory, IConnectionSelector selector, IExpirationPolicy defaultExpirationPolicy, int maxInUse, int borrowTimeoutSeconds, ScheduledExecutorService scheduler, int maintenanceIntervalSeconds) {
        this.mPoolName = poolName;
        this.mFactory = factory;
        this.mSelector = selector;
        this.mDefaultExpirationPolicy = defaultExpirationPolicy;
        this.mBorrowTimeoutSeconds = borrowTimeoutSeconds;
        this.mBorrowPermits = new Semaphore(maxInUse, false);
        this.mMaintenanceFuture = scheduler.scheduleWithFixedDelay(new Runnable(){

            @Override
            public void run() {
                DefaultConnectionPool.this.doMaintenance();
            }
        }, maintenanceIntervalSeconds, maintenanceIntervalSeconds, TimeUnit.SECONDS);
        this.mLogger.info("Initialized connection pool '{}'", (Object)this.mPoolName);
    }

    @Override
    public String getPoolName() {
        return this.mPoolName;
    }

    @Override
    public IConnectionFactory getFactory() {
        return this.mFactory;
    }

    @Override
    public IConnectionSelector getSelector() {
        return this.mSelector;
    }

    @Override
    public IExpirationPolicy getDefaultExpirationPolicy() {
        return this.mDefaultExpirationPolicy;
    }

    @Override
    public boolean isShutdown() {
        return this.mShutdown.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public IPooledConnection borrow(Map<String, Object> parameters) throws WAStorageException {
        this.assertNotShutdown();
        this.mLogger.debug("borrow() called on pool '{}' with parameters {}", (Object)this.mPoolName, parameters);
        try {
            if (!this.mBorrowPermits.tryAcquire(this.mBorrowTimeoutSeconds, TimeUnit.SECONDS)) {
                this.mLogger.error("A permit could not be acquired to borrow a connection from pool '{}' before the timeout ({} seconds) elapsed.", (Object)this.mPoolName, (Object)this.mBorrowTimeoutSeconds);
                throw new WAStorageException.ConnectionBorrowTimeout();
            }
        }
        catch (InterruptedException ex) {
            this.mLogger.error("An interrupted occured while acquiring a permit to borrow a connection from pool '{}'", (Object)this.mPoolName);
            throw new WAStorageException.ConnectionBorrowInterrupted((Throwable)ex);
        }
        boolean borrowed = false;
        try {
            DefaultConnection conn = this.selectFromPool(parameters);
            if (null == conn) {
                this.mLogger.debug("borrow() is creating new connection");
                conn = new DefaultConnection(this.mFactory.create(parameters), parameters, this);
                if (!conn.isInUse()) {
                    throw new IllegalStateException("Expected connection to be in use");
                }
                this.mConnections.add(conn);
                this.mCreationCount.incrementAndGet();
            } else {
                this.mReuseCount.incrementAndGet();
            }
            this.mSelector.prepareForBorrow(conn);
            this.mLogger.debug("borrow() is returning connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
            borrowed = true;
            DefaultConnection defaultConnection = conn;
            return defaultConnection;
        }
        finally {
            if (!borrowed) {
                this.mBorrowPermits.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doMaintenance() {
        if (this.isShutdown()) {
            return;
        }
        this.mLogger.debug("doMaintenance() called on pool '{}'", (Object)this.mPoolName);
        LinkedList<DefaultConnection> invalidConnections = new LinkedList<DefaultConnection>();
        LinkedList<DefaultConnection> expiredConnections = new LinkedList<DefaultConnection>();
        for (DefaultConnection conn : this.mConnections) {
            IExpirationPolicy policy = conn.getExpirationPolicy();
            if (null == policy) {
                policy = this.getDefaultExpirationPolicy();
            }
            if (null != policy && policy.isExpired(conn)) {
                if (conn.casInUse(false, true)) {
                    this.mLogger.info("Marking expired connection #{} for removal", (Object)Integer.toHexString(System.identityHashCode(conn)));
                    expiredConnections.add(conn);
                    continue;
                }
                this.mLogger.debug("Expired connection #{} is still in use", (Object)Integer.toHexString(System.identityHashCode(conn)));
            }
            boolean wasValidated = false;
            boolean isValid = true;
            if (conn.casInUse(false, true)) {
                try {
                    wasValidated = true;
                    isValid = conn.validate();
                }
                catch (Exception ex) {
                    this.mLogger.warn("An exception was encountered while validating connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
                    isValid = false;
                }
                finally {
                    conn.casInUse(true, false);
                }
            }
            if (!wasValidated || isValid) continue;
            if (conn.casInUse(false, true)) {
                this.mLogger.info("Marking invalid connection #{} for removal", (Object)Integer.toHexString(System.identityHashCode(conn)));
                invalidConnections.add(conn);
                continue;
            }
            this.mLogger.debug("Invalid connection #{} is still in use", (Object)Integer.toHexString(System.identityHashCode(conn)));
        }
        for (DefaultConnection conn : invalidConnections) {
            this.mLogger.info("Removing invalid connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
            this.mConnections.remove(conn);
            this.mInvalidCount.incrementAndGet();
            if (conn.isClosed()) continue;
            try {
                conn.close();
            }
            catch (Exception ex) {
                this.mLogger.warn("An exception was encountered while closing connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
            }
        }
        for (DefaultConnection conn : expiredConnections) {
            this.mLogger.info("Removing expired connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
            this.mConnections.remove(conn);
            this.mExpiredCount.incrementAndGet();
            if (conn.isClosed()) continue;
            try {
                conn.close();
            }
            catch (Exception ex) {
                this.mLogger.warn("An exception was encountered while closing connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
            }
        }
    }

    private DefaultConnection selectFromPool(Map<String, Object> parameters) {
        LinkedList<DefaultConnection> invalidConnections = new LinkedList<DefaultConnection>();
        DefaultConnection selected = null;
        for (DefaultConnection conn : this.mConnections) {
            String connHash = Integer.toHexString(System.identityHashCode(conn));
            this.mLogger.debug("Attempting to select connection #{}", (Object)connHash);
            if (conn.isInUse()) {
                this.mLogger.debug("Selection failed: connection #{} is in use", (Object)connHash);
                continue;
            }
            if (!this.mSelector.canSelect(conn, parameters)) {
                this.mLogger.debug("Selection failed: connection #{} is not compatible", (Object)connHash);
                continue;
            }
            if (conn.casInUse(false, true)) {
                if (conn.validate()) {
                    selected = conn;
                    this.mLogger.debug("Selection succeeded with connection #{}", (Object)connHash);
                    break;
                }
                this.mLogger.debug("Selection failed: connection #{} is invalid", (Object)connHash);
                invalidConnections.add(conn);
                continue;
            }
            this.mLogger.debug("Selection failed: connection #{} was in use", (Object)connHash);
        }
        for (DefaultConnection conn : invalidConnections) {
            this.mLogger.info("Removing invalid connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
            this.mConnections.remove(conn);
            this.mInvalidCount.incrementAndGet();
            if (conn.isClosed()) continue;
            try {
                conn.close();
            }
            catch (Exception ex) {
                this.mLogger.warn("An exception was encountered while closing connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
            }
        }
        return selected;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void returnToPool(DefaultConnection conn) {
        this.assertNotShutdown();
        if (this != conn.getConnectionPool()) {
            throw new IllegalArgumentException("Connection does not belong to this pool.");
        }
        this.mLogger.debug("Returning connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
        boolean isValid = false;
        try {
            if (conn.validate()) {
                try {
                    this.mSelector.prepareForReturn(conn);
                    if (!conn.casInUse(true, false)) {
                        throw new IllegalArgumentException("Connection was not borrowed.");
                    }
                    isValid = true;
                }
                catch (Exception ex) {
                    this.mLogger.warn("An exception was encountered while preparing connection #{} to be returned: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
                }
            }
            if (!isValid) {
                this.mLogger.warn("Removing invalid connection #{}", (Object)Integer.toHexString(System.identityHashCode(conn)));
                this.mConnections.remove(conn);
                this.mInvalidCount.incrementAndGet();
                if (!conn.isClosed()) {
                    try {
                        conn.close();
                    }
                    catch (Exception ex) {
                        this.mLogger.warn("An exception was encountered while closing connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
                    }
                }
            }
        }
        finally {
            this.mBorrowPermits.release();
        }
    }

    @Override
    public void shutdown() {
        if (this.mShutdown.compareAndSet(false, true)) {
            this.mLogger.info("Shutting down connection pool '{}'", (Object)this.mPoolName);
            this.mMaintenanceFuture.cancel(false);
            for (DefaultConnection conn : this.mConnections) {
                try {
                    if (conn.isClosed()) continue;
                    if (!conn.isInUse()) {
                        conn.casInUse(false, true);
                    }
                    conn.close();
                }
                catch (Throwable ex) {
                    this.mLogger.warn("An exception was encountered while closing connection #{}: {}", (Object)Integer.toHexString(System.identityHashCode(conn)), (Object)ex.getMessage());
                }
            }
            this.mConnections.clear();
        }
    }

    @Override
    public int getActiveInUseCount() {
        int count = 0;
        for (IPooledConnection iPooledConnection : this.mConnections) {
            if (!iPooledConnection.isInUse()) continue;
            ++count;
        }
        return count;
    }

    @Override
    public int getActiveIdleCount() {
        int count = 0;
        for (IPooledConnection iPooledConnection : this.mConnections) {
            if (iPooledConnection.isInUse()) continue;
            ++count;
        }
        return count;
    }

    @Override
    public int getActiveCount() {
        return this.mConnections.size();
    }

    @Override
    public long getCreationCount() {
        return this.mCreationCount.get();
    }

    @Override
    public long getReuseCount() {
        return this.mReuseCount.get();
    }

    @Override
    public long getInvalidCount() {
        return this.mInvalidCount.get();
    }

    @Override
    public long getExpiredCount() {
        return this.mExpiredCount.get();
    }

    private void assertNotShutdown() {
        if (this.mShutdown.get()) {
            throw new IllegalStateException("Pool was shutdown");
        }
    }
}

