/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.neo.dataimport.cdf.sheet.impl;

import com.ibm.neo.blobstore.BlobStoreException;
import com.ibm.neo.blobstore.IBlob;
import com.ibm.neo.blobstore.IBlobContainer;
import com.ibm.neo.blobstore.IBlobStore;
import com.ibm.neo.dataimport.cdf.sheet.CDFSheetFactory;
import com.ibm.neo.dataimport.cdf.sheet.ICDFSheet;
import com.ibm.neo.dataimport.cdf.sheet.ICDFSheetLibrary;
import com.ibm.neo.dataimport.cdf.sheet.NoSuchSheetException;
import com.ibm.neo.io.ArchiveUtils;
import com.ibm.neo.util.TTLCache;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang.NullArgumentException;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobSheetLibrary
implements ICDFSheetLibrary {
    private static final String SHEET_CONTAINER_NAME = "import.sheets-fs";
    private static final String ARCHIVE_EXTENSION = ".tar";
    private static final String KEY_PARTITIONED = "partitioned";
    private static final String KEY_LAST_UPDATE_DATE = "last-update-date";
    private static final Logger LOGGER = LoggerFactory.getLogger(BlobSheetLibrary.class);
    private final File mCacheDir;
    private final IBlobStore mBlobStore;
    private final TTLCache<String, ICDFSheet> mSheetCache = new TTLCache(1L, TimeUnit.HOURS);

    public BlobSheetLibrary(File cacheDir, IBlobStore blobStore) {
        if (null == cacheDir) {
            throw new NullArgumentException("cacheDir");
        }
        if (null == blobStore) {
            throw new NullArgumentException("blobStore");
        }
        this.mCacheDir = cacheDir;
        this.mBlobStore = blobStore;
    }

    @Override
    public ICDFSheetLibrary.IManifestEntry[] getManifest() throws IOException {
        ArrayList<ManifestEntry> entries = new ArrayList<ManifestEntry>();
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            for (IBlob blob : container) {
                entries.add(new ManifestEntry(blob));
            }
        }
        catch (BlobStoreException ex) {
            throw new IOException("Failed to load manifest entries: " + ex.getMessage(), ex);
        }
        return entries.toArray(new ICDFSheetLibrary.IManifestEntry[entries.size()]);
    }

    @Override
    public ICDFSheetLibrary.IManifestEntry getManifestEntry(String sheetId) throws IOException, NoSuchSheetException {
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            IBlob blob = BlobSheetLibrary.findBlobForSheetId(container, sheetId, true);
            if (null == blob) {
                throw new NoSuchSheetException(sheetId);
            }
            return new ManifestEntry(blob);
        }
        catch (BlobStoreException ex) {
            throw new IOException("Failed to load manifest entry from repository: " + ex.getMessage(), ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ICDFSheet load(String sheetId) throws IOException, NoSuchSheetException {
        LOGGER.trace("Loading sheet ({})", (Object)sheetId);
        ICDFSheet sheet = (ICDFSheet)this.mSheetCache.get((Object)sheetId);
        if (null != sheet && !sheet.isDestroyed()) {
            LOGGER.trace("Found sheet ({}) in cache", (Object)sheetId);
            return sheet;
        }
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            IBlob blob = BlobSheetLibrary.findBlobForSheetId(container, sheetId, true);
            if (null == blob) {
                LOGGER.error("Failed to find sheet ({}) in blob store", (Object)sheetId);
                throw new NoSuchSheetException(sheetId);
            }
            LOGGER.trace("Found sheet ({}) in blob store with blob key: {}", (Object)sheetId, (Object)blob.getKey());
            ManifestEntry entry = new ManifestEntry(blob);
            sheet = entry.isPartitioned() ? CDFSheetFactory.createPartitionedSheet(sheetId, this.mCacheDir) : CDFSheetFactory.createSheet(sheetId, this.mCacheDir);
            InputStream in = blob.open();
            boolean loadSuccess = false;
            try {
                LOGGER.trace("Expanding TAR for sheet ({}) into directory: {}", (Object)sheetId, (Object)sheet.getDataDirectory());
                ArchiveUtils.untarDirectory((InputStream)in, (File)sheet.getDataDirectory());
                sheet.loadMetadata();
                loadSuccess = true;
            }
            finally {
                in.close();
                if (!loadSuccess) {
                    LOGGER.error("Failed to expand TAR for sheet ({}); cleaning up", (Object)sheetId);
                    sheet.destroy();
                }
            }
        }
        catch (BlobStoreException ex) {
            LOGGER.error("Failed to load sheet ({}) from blob store: {}", (Object)sheetId, (Object)ex);
            throw new IOException("Failed to load sheet from repository: " + ex.getMessage(), ex);
        }
        this.mSheetCache.put((Object)sheetId, (Object)sheet);
        return sheet;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void store(ICDFSheet sheet) throws IOException {
        LOGGER.trace("Storing sheet ({})", (Object)sheet.getId());
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            IBlob blob = BlobSheetLibrary.findBlobForSheetId(container, sheet.getId(), false);
            if (null != blob) {
                LOGGER.warn("Removing existing blob ({}) for sheet ({})", (Object)blob.getKey(), (Object)sheet.getId());
                blob.remove();
            }
            File tempTar = File.createTempFile("tmp-sheet", "tar");
            try {
                LOGGER.trace("Packing TAR for sheet ({}) from directory: {}", (Object)sheet.getId(), (Object)sheet.getDataDirectory());
                ArchiveUtils.tarDirectory((File)tempTar, (File)sheet.getDataDirectory());
                try (FileInputStream in = new FileInputStream(tempTar);){
                    LOGGER.trace("Uploading TAR for sheet ({}) into blob store", (Object)sheet.getId());
                    String filename = BlobSheetLibrary.sheetId2Filename(sheet.getId());
                    blob = container.supportsCustomBlobKeys() ? container.create(filename, filename, "application/x-tar", (InputStream)in, true) : container.create(filename, "application/x-tar", (InputStream)in, true);
                    LOGGER.trace("TAR upload complete for sheet ({})", (Object)sheet.getId());
                    blob.setMetadataValue(KEY_LAST_UPDATE_DATE, (Object)new Date());
                    blob.setMetadataValue(KEY_PARTITIONED, (Object)sheet.isPartitioned());
                    blob.saveMetadata();
                }
            }
            finally {
                if (!tempTar.delete()) {
                    LOGGER.error("Failed delete temp tar directory: {}", (Object)tempTar.getAbsolutePath());
                }
            }
        }
        catch (BlobStoreException ex) {
            LOGGER.error("Failed to store sheet ({}) to blob store: {}", (Object)sheet.getId(), (Object)ex);
            throw new IOException("Failed to save sheet to repository: " + ex.getMessage(), ex);
        }
        this.mSheetCache.put((Object)sheet.getId(), (Object)sheet);
    }

    @Override
    public void remove(String sheetId) throws IOException, NoSuchSheetException {
        LOGGER.trace("Removing sheet ({})", (Object)sheetId);
        this.mSheetCache.invalidate((Object)sheetId);
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            IBlob blob = BlobSheetLibrary.findBlobForSheetId(container, sheetId, true);
            if (null == blob) {
                LOGGER.error("Failed to find sheet ({}) in blob store", (Object)sheetId);
                throw new NoSuchSheetException(sheetId);
            }
            LOGGER.warn("Removing blob ({}) for sheet ({})", (Object)blob.getKey(), (Object)sheetId);
            blob.remove();
        }
        catch (BlobStoreException ex) {
            LOGGER.error("Failed to remove sheet ({}) from blob store: {}", (Object)sheetId, (Object)ex);
            throw new IOException("Failed to remove sheet from repository: " + ex.getMessage(), ex);
        }
    }

    @Override
    public void clear() throws IOException {
        this.mSheetCache.clear();
        try {
            IBlobContainer container = this.mBlobStore.getContainer(SHEET_CONTAINER_NAME, true);
            for (IBlob blob : container) {
                blob.remove();
            }
        }
        catch (BlobStoreException ex) {
            throw new IOException("Failed to clear sheets from repository: " + ex.getMessage(), ex);
        }
    }

    @Override
    public void dispose() {
        this.mSheetCache.clear();
    }

    @Override
    public boolean isCached(String sheetId) {
        return this.mSheetCache.containsKey((Object)sheetId);
    }

    @Override
    public void uncache(String sheetId) throws IOException {
        this.mSheetCache.invalidate((Object)sheetId);
    }

    public void clearCache() {
        this.mSheetCache.clear();
    }

    private static IBlob findBlobForSheetId(IBlobContainer container, String sheetId, boolean legacyScan) throws BlobStoreException {
        String filename = BlobSheetLibrary.sheetId2Filename(sheetId);
        IBlob blob = null;
        try {
            List matches;
            if (container.supportsCustomBlobKeys() && null != (blob = container.get(filename))) {
                return blob;
            }
            if (legacyScan && null != (matches = container.findFilename(filename)) && !matches.isEmpty()) {
                blob = (IBlob)matches.get(0);
            }
        }
        catch (BlobStoreException.ContainerNotFound containerNotFound) {
            // empty catch block
        }
        return blob;
    }

    private static String sheetId2Filename(String sheetId) {
        return sheetId + ARCHIVE_EXTENSION;
    }

    private static final class ManifestEntry
    implements ICDFSheetLibrary.IManifestEntry {
        private final String mSheetId;
        private final String mArchiveName;
        private final boolean mPartitioned;
        private final Date mLastUpdateDate;
        private final long mSize;

        private ManifestEntry(IBlob blob) throws BlobStoreException {
            this.mArchiveName = blob.getFilename();
            this.mSheetId = blob.getFilename().substring(0, this.mArchiveName.length() - BlobSheetLibrary.ARCHIVE_EXTENSION.length());
            this.mPartitioned = Boolean.TRUE.equals(blob.getMetadataValue(BlobSheetLibrary.KEY_PARTITIONED));
            this.mLastUpdateDate = (Date)blob.getMetadataValue(BlobSheetLibrary.KEY_LAST_UPDATE_DATE);
            this.mSize = blob.getContentLength();
        }

        @Override
        public String getSheetId() {
            return this.mSheetId;
        }

        @Override
        public String getArchiveName() {
            return this.mArchiveName;
        }

        @Override
        public boolean isPartitioned() {
            return this.mPartitioned;
        }

        @Override
        public Date getLastUpdateDate() {
            return this.mLastUpdateDate;
        }

        @Override
        public long getSize() {
            return this.mSize;
        }

        public String toString() {
            return ToStringBuilder.reflectionToString((Object)this, (ToStringStyle)ToStringStyle.MULTI_LINE_STYLE);
        }
    }
}

