package com.cognos.cm.store.path;

/*
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * BI and PM: CM
 *
 * (C) Copyright IBM Corp. 2005
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 */

/**
 * Bare bones search path lexer.
 *
 * <p>Use yylex() to get next token. Tokens are represented by int constants
 * defined in sym.java. Special token sym.EOF denotes an end of the stream.
 *
 * <p>Use getText() to obtain string value of current token.
 */


%%


%class Lexer
%public
%int
%unicode
%table
%eofclose
%buffer 4096
%scanerror Lexer.LexerException
%eofval{
  return sym.EOF;
%eofval}

%{
	public class LexerException extends Exception {
		public LexerException(String msg) {
			super(msg);
		}
	}

  /**
   * Creates a new scanner
   *
   * @param   in  the String to read input from.
   */
	public Lexer(String in) {
		this(new java.io.StringReader(in));
	}

	private boolean      ignoreWhitespace_;
	private StringBuffer buffer_ = new StringBuffer(128);
	private int          sym_;

	/**
	 * Specifies handling of whitespace characters.
	 * @param val - true - whitespace characters will be ignored.
	 *    false - whitespace characters will be returned by yylex() as
	 *    WHITESPACE token.
	 */
	public boolean ignoreWhitespace(boolean val) {
		boolean ret = ignoreWhitespace_;
		ignoreWhitespace_ = val;
		return ret;
	}

	/**
	 * Return string value of current token.
	 */
	public String getText() {
		if (sym_ == sym.LITERAL)
			return buffer_.toString();
		return yytext();
	}
%}


WhitespaceChar   = [ \t\r\n]
WS               = {WhitespaceChar}*
Letter           = [A-Z]|[a-z]
NameChar         = {Letter}|[0-9]|"-"|"_"
Name             = ({Letter}|"_"){NameChar}*


%state SSTRING, DSTRING

%%

<YYINITIAL> {

"::"                             { return sym_ = sym.COLUMN2; }

"and"                            { return sym_ = sym.AND; }
"or"                             { return sym_ = sym.OR; }

"="                              { return sym_ = sym.EQ; }
"!="                             { return sym_ = sym.NE; }
">="                             { return sym_ = sym.GE; }
"<="                             { return sym_ = sym.LE; }
"<"                              { return sym_ = sym.LT; }
">"                              { return sym_ = sym.GT; }

"*"                              { return sym_ = sym.STAR; }
"//"                             { return sym_ = sym.SLASH2; }
"/"                              { return sym_ = sym.SLASH; }
"~"                              { return sym_ = sym.TILDE; }
"~~"                             { return sym_ = sym.TILDE2; }
"["                              { return sym_ = sym.LBRACKET; }
"]"                              { return sym_ = sym.RBRACKET; }
"("                              { return sym_ = sym.LPARA; }
")"                              { return sym_ = sym.RPARA; }
","                              { return sym_ = sym.COMMA; }
"@"                              { return sym_ = sym.AT; }
"|"                              { return sym_ = sym.PIPE; }

{Name}                           { return sym_ = sym.NAME; }
\"                               { yybegin(DSTRING); buffer_.setLength(0); }
"'"                              { yybegin(SSTRING); buffer_.setLength(0); }
{WS}                             { if (!ignoreWhitespace_) return sym_ = sym.WHITESPACE; }
.                                { throw new LexerException("Invalid character: '" + yytext() + "'"); }

}


<SSTRING> {
[^']+                            { buffer_.append(yytext()); }
"''"                             { buffer_.append('\''); }
"'"                              { yybegin(YYINITIAL);
                                   return sym_ = sym.LITERAL; }
}


<DSTRING> {
[^\"]+                           { buffer_.append(yytext()); }
"\"\""                           { buffer_.append('\"'); }
\"                               { yybegin(YYINITIAL);
                                   return sym_ = sym.LITERAL; }
}
