// IBM Confidential
// OCO Source Materials
// BI and PM: Mobile
// (C) Copyright IBM Corp. 2012
// The source code for this program is not published or otherwise divested of its trade secrets, irrespective of what has been deposited with the U.S. Copyright Office.

package com.cognos.mobile.server.validation;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import com.cognos.mobile.common.CMErrors;
import com.cognos.mobile.common.CMException;
import com.cognos.mobile.common.CMUrlHelper;
import com.cognos.mobile.vm.VM;
import com.cognos.pogo.pdk.BIBusEnvelope;
import com.cognos.pogo.util.NameValuePair;
import com.cognos.validator.api.IParameter;
import com.cognos.validator.api.IParameterUpdater;
import com.cognos.validator.api.ValidatorException;
import com.cognos.validator.api.VerifyException;


/**
 * wrapper class to use cognos validator to verify mobile running parameters;
 * @author wangj
 *
 */
public class MobileValidator {
	
	private static final Class CLASS = MobileValidator.class;
	private ICognosValidator c8Validator;
	private HashMapParameter m_hmParameters;


	private static HashMap<String, String> m_hmProxyPrefix = new HashMap<String,String>();

	static {
		// default proxy settings for mashups
		m_hmProxyPrefix.put("/mum/proxy/", "/mum/proxy/"); //$NON-NLS-1$ //$NON-NLS-2$
		m_hmProxyPrefix.put("/mum/cognosProxy/", "/mum/cognosProxy/"); //$NON-NLS-1$ //$NON-NLS-2$
		m_hmProxyPrefix.put("/wps/proxy/", "/wps/proxy/"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public MobileValidator(ICognosValidator c8Validator) {
		super();
		this.c8Validator = c8Validator;
	}


	private void buildInputData(String id, String value) {

		// the cv.gateway and cv.webcontent is currently passed in by Business
		// Space and will have the following
		// format /mum/proxy/http/localhost:9080/p2pd/servlet
		// we need to strip off the proxy information (/mum/proxy/) and replace
		// http/ with http:// so that
		// the domain can be validated correctly
		if ("cv.webcontent".equals(id) || "cv.gateway".equals(id)) {
			Iterator it = m_hmProxyPrefix.values().iterator();

			while (it.hasNext()) {
				String sProxy = (String) it.next();
				if (value.startsWith(sProxy)) {
					// remove the known proxy information
					value = value.substring(sProxy.length());

					// the URL we get from mashups has all the parts between the
					// / URL encoded.
					// So we need to split it, urldecode it and then put it back
					// together
					if (value.indexOf("/") != -1) { //$NON-NLS-1$
						String[] aProxyParts = value.split("/"); //$NON-NLS-1$
						String sDecodedValue = ""; //$NON-NLS-1$
						int iSize = aProxyParts.length;
						for (int i = 0; i < iSize; i++) {
							if (aProxyParts[i].length() > 0) {
								if (i > 0) {
									sDecodedValue += "/"; //$NON-NLS-1$
								}
								sDecodedValue += CMUrlHelper.decodeUrl(aProxyParts[i]);
							}
						}

						value = sDecodedValue.replaceFirst("/", "://"); //$NON-NLS-1$ //$NON-NLS-2$
					}
					break;

				}
			}
		}

		m_hmParameters.put(id, value);
	}

	private IParameterUpdater getInputData() {
		return m_hmParameters;
	}

	


	/**
	 * This method throws a CV-0003 Validation Error.
	 * 
	 * @param messageContext
	 * @param e
	 * @throws CMException
	 */
	public void throwValidationError(BIBusEnvelope busEnvelope, Exception e) throws CMException {
		VM.log(CLASS, VM.SEVERITY_ERROR,e.getMessage(),e);
		CMException cmException = new CMException(CMErrors.SERVER_VALIDATION, "parameter validation failed!", e);
		throw cmException;
	}

	public void validate(BIBusEnvelope busEnvelope) throws CMException {

		if (c8Validator != null) {
			m_hmParameters = new HashMapParameter();
			List formFieldVars = busEnvelope.getFormFieldValues();
			ListIterator iterator = formFieldVars.listIterator();
			while(iterator.hasNext())
			{
				NameValuePair formFieldVar = (NameValuePair) iterator.next();

				String formFieldName = formFieldVar.getName();
				String formFieldValue = formFieldVar.getValue();
				try
				{
					buildInputData(formFieldName, formFieldValue);
				}
				catch (Exception e)
				{
					VM.log(CLASS, VM.SEVERITY_ERROR, "failed to build input data:" //$NON-NLS-1$
							+ e.getMessage(), e);
				}
			}
		

			IParameterUpdater input_data = getInputData();

			try {
				c8Validator.setProperty(ICognosValidator.PROP_BOOL_IGNORE_UNKNOWN_PARAMETERS, true);
				c8Validator.validate(input_data);
			} catch (VerifyException e) {
				VM.log(CLASS,VM.SEVERITY_ERROR,e.getFailureDetails());
				throw new CMException(CMErrors.SERVER_VALIDATION, "parameter validation failed!", e);
			} catch (ValidatorException e) {
				throwValidationError(busEnvelope, e);
			}
		}
	}

	public static void addProxyPrefix(String sPrefix) {
		m_hmProxyPrefix.put(sPrefix, sPrefix);
	}

	public static HashMap<String,String> getProxyPrefix() {
		return m_hmProxyPrefix;
	}


	public static class HashMapParameter implements IParameterUpdater {
		static class Parameter implements IParameter {
			private String m_name = null;
			private String[] m_values = null;

			Parameter(Map.Entry<String,String> entry) {
				m_name = (String) entry.getKey();
				m_values = new String[1];
				m_values[0] = (String) entry.getValue();
			}

			public String name() {
				return m_name;
			}

			public String[] values() {
				return m_values;
			}

			public void setValues(String[] arg0) {
				// TODO Auto-generated method stub
			}
		}

		private Iterator<Map.Entry<String,String>> m_iterator = null;
		private HashMap<String,String> m_hashMap = new HashMap<String,String>();

		public void put(String name, String value) {
			m_hashMap.put(name, value);
		}

		public boolean hasNext() {
			if (m_iterator == null) {
				m_iterator = m_hashMap.entrySet().iterator();
			}
			return m_iterator.hasNext();
		}

		public IParameter next() {
			Map.Entry<String,String> entry = (Map.Entry<String,String>) m_iterator.next();
			return new Parameter(entry);
		}

		public void reset() {
			m_iterator = null;
		}
	}
}
