/*************************************************************************************************************************************************************
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * IBM Cognos Products: Cognos Analytics
 *
 * (C) Copyright IBM Corp. 2019
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *************************************************************************************************************************************************************/

package com.ibm.bi.rest.bridge.cm;

import java.util.NoSuchElementException;

import org.apache.log.Hierarchy;
import org.apache.log.Logger;
import org.dom4j.Element;

import com.cognos.developer.schemas.bibus._3.PropEnum;
import com.cognos.pogo.bibus.CommandExecutionException;
import com.cognos.pogo.contentmanager.CMClass;
import com.cognos.pogo.contentmanager.ContentManagerQueryCommand;
import com.cognos.pogo.pdk.BIBusEnvelope;
import com.ibm.bi.json.JsonObject;
import com.ibm.bi.rest.bridge.exception.RestBridgeRuntimeException;

/*
	Example: Runnable CM object returned by ContentManagerQueryCommand.

	<item xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:type="cm:jupyterNotebook">
		<objectClass xsi:type="cm:classEnumProp">
			<value xsi:type="cm:classEnum">jupyterNotebook</value>
		</objectClass>
		<storeID xsi:type="cm:guidProp">
			<value xsi:type="cm:guid">i23AED96ACF5B4056846E5C453C6FC94B</value>
		</storeID>
	</item>
*/

/**
 * CM query handler.
 * 
 * RestBridgeService queries CM to fetch the details of the runnable objects.
 */
public class CMQueryHandler {

	private static final Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor(CMQueryHandler.class.getName());
	
	// Incoming SOAP request
	private BIBusEnvelope requestEnvelope;
	
	public CMQueryHandler(BIBusEnvelope requestEnvelope) {
		this.requestEnvelope = requestEnvelope;
	}
	
	/**
	 * Get runnable object details as a JsonObject.
	 * 
	 * @param objectPath		Runnable object path in CM.
	 * @return a {@link JsonObject}
	 */
	public JsonObject getRunnableObject(String objectPath) {
		logMethodCall("getRunnableObject(String)");
		
		CMClass cmObject = this.getCMObject(objectPath);

		JsonObject runnableObject = new JsonObject()
			.set("path", objectPath)
			.set("id", this.getProperty(cmObject, "storeID"))
			.set("type", this.getProperty(cmObject, "objectClass"));

		if (logger.isDebugEnabled()) {
			logger.debug("runnableObject json: " + runnableObject);
		}
		
		return runnableObject;
	}
	
	/**
	 * Fetch an object from CM.
	 */
	private CMClass getCMObject(String objectPath) {
		logMethodCall("getCMObject(String)");

		ContentManagerQueryCommand queryCommand = new ContentManagerQueryCommand(objectPath);
		queryCommand.addQueryPropertyName(PropEnum._objectClass);
		queryCommand.addQueryPropertyName(PropEnum._storeID);
		this.requestEnvelope.copyBiBusHeaderTo(queryCommand.getRequestEnvelope());

		try {
			if (logger.isDebugEnabled()) {
				logger.debug("Calling CM to fetch the object: " + objectPath);
			}

			queryCommand.execute();
			CMClass cmObject = queryCommand.getFirstResultClass();

			if (logger.isDebugEnabled()) {
				logger.debug("Fetched CM object: " + cmObject.getItemElement().asXML());
			}

			return cmObject;
		} catch (CommandExecutionException ex) {
			throw new RestBridgeRuntimeException("CM query for the object '" + objectPath + "' has failed.", ex);
		} catch (NoSuchElementException ex) {
			throw new RestBridgeRuntimeException("CM object not found: " + objectPath, ex);
		}
	}

	/**
	 * Fetch property value from the given CM object.
	 */
	private String getProperty(CMClass cmObject, String propertyName) {
		logMethodCall("getProperty(CMClass, String)");
		Element propertyElement = cmObject.getPropertyAsElement(propertyName);
		Element value = propertyElement.element("value");
		if (value == null) {
			throw new NoSuchElementException("Missing value element for property '" + propertyName + "'");
		}
		return value.getText().trim();
	}
	
	private void logMethodCall(String methodName) {
		if (logger.isDebugEnabled()) {
			logger.debug("Calling " + methodName);
		}
	}
}
