/*************************************************************************************************************************************************************
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * IBM Cognos Products: Cognos Analytics
 *
 * (C) Copyright IBM Corp. 2019
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *************************************************************************************************************************************************************/

package com.ibm.bi.rest.bridge.config;

import java.io.InputStream;
import java.io.InputStreamReader;

import org.apache.log.Hierarchy;
import org.apache.log.Logger;

import com.ibm.bi.json.JsonObject;
import com.ibm.bi.json.JsonParser;
import com.ibm.bi.rest.bridge.exception.RestBridgeRuntimeException;

/**
 * This class is responsible for loading and parsing restBridgeConfig.json. 
 *
 * Each incoming SOAP request creates its own instance of this class.
 * This is slightly inefficient because it causes repeated loading of the same file.
 * The benefit is that we can update the config file without restarting CA.
 */
public final class RestBridgeConfiguration {

	private static final Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor(RestBridgeConfiguration.class.getName());

	private static final String configFile = "restBridgeConfig.json";
	
	private final JsonObject config;
	private final JsonObject classMap;
	
	public RestBridgeConfiguration() {
		this.config = load();

		this.classMap = this.config.getObject(ConfigKeys.CLASS_MAP);
		if (classMap == null || classMap.isEmpty()) {
			throw new RestBridgeRuntimeException("Missing classMap configuration");
		}
	}

	private JsonObject load() {
		logger.debug("Loading service configuration from restBridgeConfig.json");

		ClassLoader cl = RestBridgeConfiguration.class.getClassLoader();
		InputStream inputStream = cl.getResourceAsStream(configFile);
		if (inputStream == null) {
			throw new RestBridgeRuntimeException("Resource file not found: restBridgeConfig.json");
		}

		try (InputStreamReader reader = new InputStreamReader(inputStream)) {
			JsonParser parser = new JsonParser();
			JsonObject config = (JsonObject) parser.parse(reader);
			if (logger.isDebugEnabled()) {
				logger.debug("Loaded service configuration: " + config.toString());
			}
			return config;
		} catch (Exception ex) {
			throw new RestBridgeRuntimeException("Failed to load restBridgeConfig.json", ex);
		}
	}
	
	public JsonObject getClassConfig(String className) {
		if (className == null || className.isEmpty()) {
			throw new IllegalArgumentException("invalid className");
		}
		return classMap.getObject(className);
	}
	
	public int getInt(String key, int defaultValue) {
		Long value = config.getLong(key);
		return (value != null) ? value.intValue() : defaultValue;
	}

	public JsonObject getObject(String key) {
		return config.getObject(key);
	}
}
