/*************************************************************************************************************************************************************
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * IBM Cognos Products: Cognos Analytics
 *
 * (C) Copyright IBM Corp. 2019
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *************************************************************************************************************************************************************/

package com.ibm.bi.rest.bridge.methods;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.log.Hierarchy;
import org.apache.log.Logger;
import org.dom4j.Element;

import com.cognos.developer.schemas.bibus._3.AsynchReplyStatusEnum;
import com.cognos.pogo.async.AsyncContext;
import com.cognos.pogo.async.AsyncState;
import com.cognos.pogo.pdk.Fault;
import com.cognos.pogo.pdk.MessageContext;
import com.ibm.bi.json.JsonObject;
import com.ibm.bi.rest.RESTClient;
import com.ibm.bi.rest.bridge.cm.CMQueryHandler;
import com.ibm.bi.rest.bridge.config.ConfigKeys;
import com.ibm.bi.rest.bridge.config.RestBridgeConfiguration;
import com.ibm.bi.rest.bridge.exception.RestBridgeRuntimeException;
import com.ibm.bi.rest.bridge.utils.RESTClientBuilder;
import com.ibm.bi.rest.bridge.utils.SoapRequestHelper;
import com.ibm.bi.rest.bridge.utils.SoapResponseHelper;

/*
	An example of <run> method element in the incoming SOAP request

	<soapenv:Body>
		<ns2:run xmlns:ns2="http://developer.cognos.com/schemas/restBridgeService/1" soapenv:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">
			<objectPath xmlns:ns3="http://developer.cognos.com/schemas/bibus/3/" xsi:type="ns3:searchPathSingleObject">
				/content/folder[@name='notebooks']/jupyterNotebook[@name='My Basic Notebook']
			</objectPath>
			<parameterValues> 
				...
			</parameterValues>
			<options>
				...
			</options>
		</ns2:run>
	</soapenv:Body>
*/


/**
 * RunMethod class bridges BI Bus <run> method.
 */ 
public final class RunMethod extends BaseMethod implements IServiceMethod
{
	private static final Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor(RunMethod.class.getName());

	private static final int DEFAULT_RUN_TIMEOUT = 30; // seconds
	private static final int DEFAULT_WAIT_SLEEP = 30; // seconds

	private int runTimeout;
	private int waitSleep;

	// JSON representation of the runnable CM object
	private JsonObject runnableObject;

	// Bridge configuration for the class of the runnable object
	private JsonObject classBridgeConfig;

	// The ID of the run in progress (returned by the target service)
	private String runId;

	// Cancel flag is set in the main thread; checked in async thread.
	private boolean isCancelled = false;

	public RunMethod(HttpServletRequest httpServletRequest, RestBridgeConfiguration bridgeConfig, AsyncContext asyncContext, Element methodElement) {
		super(logger, httpServletRequest, bridgeConfig, asyncContext, methodElement);
		logMethodCall("RunMethod constructor");
		initializeRunnableObject();

		this.runTimeout = this.bridgeConfig.getInt(ConfigKeys.RUN_TIMEOUT, DEFAULT_RUN_TIMEOUT);
		this.waitSleep = this.bridgeConfig.getInt(ConfigKeys.WAIT_SLEEP, DEFAULT_WAIT_SLEEP);
		this.classBridgeConfig = this.getClassBridgeConfig();
	}

	private void initializeRunnableObject() {
		logMethodCall("initializeRunnableObject()");

		String objectPath = SoapRequestHelper.getObjectPath(this.methodElement);
		CMQueryHandler cmQuery = new CMQueryHandler(this.requestEnvelope);
		this.runnableObject = cmQuery.getRunnableObject(objectPath);
	}

	/**
	 * Returns class bridge configuration for the runnableObject type.
	 */
	private JsonObject getClassBridgeConfig() {
		logMethodCall("getClassBridgeConfig()");
		
		String runnableType = this.runnableObject.getString("type");
		JsonObject classBridgeConfig = this.bridgeConfig.getClassConfig(runnableType);

		if (classBridgeConfig == null) {
			String msg = "Missing bridge configuration for CM class: " + runnableType;
			logger.error(msg);
			logFailedRunnableObject();
			throw new RestBridgeRuntimeException(msg);
		}
		
		if (logger.isDebugEnabled()) {
			logger.debug("Bridge configuration for CM class " + runnableType + ": " + classBridgeConfig);
		}
		
		return classBridgeConfig;
	}

	/**
	 * @see IServiceMethod#cancel()
	 */
	@Override
	public void cancel() {
		logMethodCall("cancel()");
		this.isCancelled = true;
	}

	/**
	 * @see IServiceMethod#execute()
	 */
	@Override
	public void execute() {
		logMethodCall("execute()");
		try (RESTClient restClient = RESTClientBuilder.createClient(this.httpServletRequest, this.requestEnvelope)) {
			this.processingLoop(restClient);
		} catch (IOException ex) {
			logFailedRunnableObject();
			throw new RestBridgeRuntimeException(ex);
		}
	}
	
	/**
	 * The main processing loop for the run method.
	 */
	private void processingLoop(RESTClient restClient) throws IOException {
		logMethodCall("processingLoop(RESTClient)");

		// Call the target service to start the run
		int statusCode = this.restRun(restClient);

		// Keep polling the service for as long as we get 202 back
		while (statusCode == HttpStatus.SC_ACCEPTED) {

			if (logger.isDebugEnabled()) {
				logger.debug("processingLoop sleeping for " + this.waitSleep + " seconds");
			}

			try {
				Thread.sleep(this.waitSleep * 1000);
				logger.debug("processingLoop sleep ended");
			} catch (InterruptedException e) {
				logger.debug("processingLoop sleep interrupted");
			}

			if (this.isCancelled) {
				// Cancel arrived while the thread was sleeping
				logger.debug("Cancelling the processing loop (isCancelled=true)");
				this.restCancel(restClient);
				this.messageContext.setFault(new Fault("RunMethod.cancelled"));
				return;
			}

			AsyncState state = this.asyncContext.getState();
			if (state != AsyncState.EXECUTING) {
				// AsyncState changed while the thread was sleeping
				logger.debug("Cancelling the processing loop (asyncState='" + state + "')");
				this.restCancel(restClient);
				this.messageContext.setFault(new Fault("RunMethod.asyncState." + state));
				return;
			}

			// Call the target service to check the status
			statusCode = this.restWait(restClient);
		}

		if (logger.isDebugEnabled()) {
			logger.debug("Processing loop is complete (statusCode=" + statusCode + ")");
		}

		// Handle an unlikely case: cancel arrived in the middle of a REST call.
		if (this.isCancelled) {
			logger.debug("The request has been cancelled (isCancelled=true)");
			this.messageContext.setFault(new Fault("RunMethod.cancelled"));
			return;
		}

		// Handle an unlikely case: AsyncState changed in the middle of a REST call.
		AsyncState state = this.asyncContext.getState();
		if (state != AsyncState.EXECUTING) {
			logger.debug("Unexpected asyncState: " + state);
			this.messageContext.setFault(new Fault("RunMethod.asyncState." + state));
			return;
		}

		if (statusCode != HttpStatus.SC_OK) {
			// This is just a bit of defensive programming. We are not expected to get here ever!
			// restRun() and restWait() throw exceptions when RESTClient returns an unexpected status.
			String msg = "Unexpected HTTP status code at the end of processing loop: " + statusCode;
			logger.error(msg);
			logFailedRunnableObject();
			throw new IllegalStateException(msg);
		}

		// Success! Prepare SOAP response and return normally.
		// The response prepared here will be used if the run method completes *before* primaryWaitThreshold expiration.
		// If primaryWaitThreshold has already expired, pogo.async toolkit calls responseReadyImpl() and uses the response
		// prepared there.
		SoapResponseHelper builder = new SoapResponseHelper(this.messageContext);
		builder.prepareRunResponse(AsynchReplyStatusEnum.conversationComplete);
	}

	/**
	 * Make a "run" call to the target REST service.
	 */
	private int restRun(RESTClient restClient) throws IOException {
		logMethodCall("restRun(RESTClient)");
		
		String url = this.classBridgeConfig.getString("url");

		JsonObject payload = new JsonObject();
		payload.set("id", this.runnableObject.get("id"));
		payload.set("timeout", this.runTimeout);

		StringEntity entity = new StringEntity(payload.toString(), ContentType.APPLICATION_JSON);
		entity.setChunked(false);
		
		if (logger.isDebugEnabled()) {
			logger.debug("Calling RESTClient.createResource on URL: " + url);
		}

		int statusCode = restClient.createResource(url, entity);

		if (logger.isDebugEnabled()) {
			logger.debug("RESTClient.createResource returned HTTP statusCode: " + statusCode);
		}

		if (statusCode == HttpStatus.SC_OK) {
			// Success! Run request completed within the runTimeout.
			return statusCode;
		}

		if (statusCode == HttpStatus.SC_ACCEPTED) {
			// Run request accepted for execution by the target service.
			// Extract the runId from the response so we can wait or cancel the run in progress.
			JsonObject jsonResponse = restClient.getResponseAsJsonObject();
			if (!jsonResponse.containsKey("runId")) {
				String msg = "Missing runId property in the JSON response from POST " + url;
				logger.error(msg);
				logFailedRunnableObject();
				throw new RestBridgeRuntimeException(msg);
			}
			this.runId = jsonResponse.getString("runId");
			return statusCode;
		}

		String msg = "POST " + url + " returned unexpected HTTP status code: " + statusCode;
		logger.error(msg);
		logFailedRunnableObject();
		throw new RestBridgeRuntimeException(msg);
	}

	/**
	 * Make a "wait" call to the target REST service.
	 */
	private int restWait(RESTClient restClient) throws IOException {
		logMethodCall("restWait(RESTClient)");
		
		String url = this.classBridgeConfig.getString("url") + "/" + this.runId;

		if (logger.isDebugEnabled()) {
			logger.debug("Calling RESTClient.getResource on URL: " + url);
		}
		
		int statusCode = restClient.getResource(url);
		
		if (logger.isDebugEnabled()) {
			logger.debug("RESTClient.getResource returned HTTP statusCode: " + statusCode);
		}

		if (statusCode == HttpStatus.SC_OK || statusCode == HttpStatus.SC_ACCEPTED) {
			return statusCode;
		}
		
		String msg = "GET " + url + " returned unexpected HTTP status code: " + statusCode;
		logger.error(msg);
		logFailedRunnableObject();
		throw new RestBridgeRuntimeException(msg);
	}

	/**
	 * Make a "cancel" call to the target REST service.
	 */
	private void restCancel(RESTClient restClient) throws IOException {
		logMethodCall("restCancel(RESTClient)");
		
		String url = this.classBridgeConfig.getString("url") + "/" + this.runId;
		
		if (logger.isDebugEnabled()) {
			logger.debug("Calling RESTClient.deleteResource on URL: " + url);
		}
		
		int statusCode = restClient.deleteResource(url, MediaType.MEDIA_TYPE_WILDCARD);
		
		if (logger.isDebugEnabled()) {
			logger.debug("RESTClient.deleteResource returned HTTP statusCode: " + statusCode);
		}

		if (statusCode != HttpStatus.SC_OK) {
			logger.error("Failed to cancel runnable object: " + (this.runnableObject != null ? this.runnableObject.toString() : "null"));
			logger.error("DELETE " + url + " returned unexpected HTTP status code: " + statusCode );
		}
	}

	/**
	 * @see RestBridgeAsyncService#workingImpl(MessageContext, AsyncContext)
	 */
	@Override
	public void workingImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("workingImpl(MessageContext, AsyncContext)");
		SoapResponseHelper helper = new SoapResponseHelper(mc);
		helper.prepareRunResponse(AsynchReplyStatusEnum.working);
	}

	/**
	 * @see RestBridgeAsyncService#responseReadyImpl(MessageContext, AsyncContext)
	 */
	@Override
	public void responseReadyImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("responseReadyImpl(MessageContext, AsyncContext)");
		SoapResponseHelper helper = new SoapResponseHelper(mc);
		helper.prepareWaitResponse(AsynchReplyStatusEnum.conversationComplete);
	}

	private void logFailedRunnableObject() {
		logger.error("Failed to execute runnable object: " + (this.runnableObject != null ? this.runnableObject.toString() : "null"));
	}
}
