/*************************************************************************************************************************************************************
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * IBM Cognos Products: Cognos Analytics
 *
 * (C) Copyright IBM Corp. 2019
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *************************************************************************************************************************************************************/

package com.ibm.bi.rest.bridge.service;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

import javax.servlet.http.HttpServletRequest;

import org.apache.log.Hierarchy;
import org.apache.log.Logger;
import org.dom4j.Element;

import com.cognos.developer.schemas.bibus._3.AsynchReplyStatusEnum;
import com.cognos.pogo.async.AsyncContext;
import com.cognos.pogo.async.AsyncServiceBase;
import com.cognos.pogo.pdk.BIBusEnvelope;
import com.cognos.pogo.pdk.Fault;
import com.cognos.pogo.pdk.MessageContext;
import com.ibm.bi.json.JsonObject;
import com.ibm.bi.rest.bridge.config.ConfigKeys;
import com.ibm.bi.rest.bridge.config.RestBridgeConfiguration;
import com.ibm.bi.rest.bridge.exception.RestBridgeRuntimeException;
import com.ibm.bi.rest.bridge.methods.IServiceMethod;
import com.ibm.bi.rest.bridge.methods.MethodFactory;
import com.ibm.bi.rest.bridge.utils.SoapRequestHelper;
import com.ibm.bi.rest.bridge.utils.SoapResponseHelper;

/**
 * RestBridgeAsynService executes asynchronous requests.
 */
public final class RestBridgeAsyncService extends AsyncServiceBase {

	private static final Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor(RestBridgeAsyncService.class.getName());

	/*
		Set the default primary threshold to 1 second to quickly initiate an asynchronous conversation
		with the Monitoring Service. The MS needs a response to the primary request to know where to send
		a secondary Cancel request. If the user cancels the activity before the primary threshold expires,
		the Cancel request won't reach this service.
	*/
	private static final int DEFAULT_PRIMARY_THRESHOLD = 1; // seconds
	private static final int DEFAULT_SECONDARY_THRESHOLD = 30; // seconds

	private RestBridgeConfiguration bridgeConfig;
	private final HttpServletRequest httpServletRequest;

	private IServiceMethod methodHandler;

	public RestBridgeAsyncService(MessageContext messageContext) {
		logMethodCall("RestBridgeAsyncService() constructor");

		// invoke() method runs in a separate thread. "http_servlet_request" property is not available
		// in the MessageContext object passed to that thread. Have to grab the value here.
		this.httpServletRequest = (HttpServletRequest) messageContext.getProperty("http_servlet_request");

		// Reload restBridgeConfig.json on every incoming request to allow configuration changes without restarting CA.
		// Configuration has to be available before we call setPrimaryAsyncThreshold().
		try {
			this.bridgeConfig = new RestBridgeConfiguration();
		} catch (RestBridgeRuntimeException ex) {
			// Log the exception but do not throw. We will fault the incoming
			// request in the invoke() method. This results in better error reporting
			// in the scheduling UI.
			logger.error("Failed to create RestBridgeConfiguration", ex);
			this.bridgeConfig = null;
			return;
		}

		// Have to set the primary threshold in the main thread before
		// async toolkit calls invoke() in a separate thread.
		setPrimaryAsyncThreshold(messageContext);
	}

	/**
	 * Execute an asynchronous request to completion.
	 *
	 * Dispatcher calls this method in a separate "async" thread.
	 *
	 * @param asyncContext
	 *            The context of an ongoing asynchronous conversation.
	 */
	@Override
	public void invoke(AsyncContext asyncContext) {
		logMethodCall("invoke(AsyncContext)");

		MessageContext mc = asyncContext.getMessageContext();
		if (logger.isDebugEnabled()) {
			BIBusEnvelope requestEnvelope = (BIBusEnvelope) mc.getProperty("request.envelope");
			logBIBusEnvelope("restBridgeService request: ", requestEnvelope);
		}

		if (this.bridgeConfig == null) {
			Fault fault = new Fault("");
			fault.addDetail(" Failed to load restBridgeService configuration.");
			mc.setFault(fault);
			return;
		}

		try {
			this.methodHandler = MethodFactory.getMethodHandler(this.httpServletRequest, this.bridgeConfig, asyncContext);
			this.methodHandler.execute();
		} catch (Exception ex) {
			StringWriter strWriter = new StringWriter();
			strWriter.append(" restBridgeService request has failed. Caused by: ");
			PrintWriter printWriter = new PrintWriter(strWriter);
			ex.printStackTrace(printWriter);
			printWriter.close();

			Fault fault = new Fault("");
			fault.addDetail(strWriter.toString());
			mc.setFault(fault);
			return;
		}

		if (!mc.isFaulted() && logger.isDebugEnabled()) {
			BIBusEnvelope responseEnvelope = (BIBusEnvelope) mc.getProperty("response.envelope");
			logBIBusEnvelope("restBridgeService response: ", responseEnvelope);
		}
	}

	/**
	 * Set primary wait threshold in the pogo.async toolkit. The value is the maximum amount of time, in seconds,
	 * the service can use to process the primary request (e.g. "run") before sending a response to the client. 
	 */
	private void setPrimaryAsyncThreshold(MessageContext messageContext) {
		logMethodCall("setPrimaryAsyncThreshold(Element)");
		this.setAsyncThreshold(messageContext, ConfigKeys.PRIMARY_WAIT_THRESHOLD, DEFAULT_PRIMARY_THRESHOLD);
	}

	/**
	 * Set secondary wait threshold in the pogo.async toolkit. The value is the maximum amount of time, in seconds,
	 * the service can use to process a secondary "wait" request before sending a response to the client. 
	 */
	private void setSecondaryAsyncThreshold(MessageContext messageContext) {
		logMethodCall("setSecondaryAsyncThreshold(Element)");
		this.setAsyncThreshold(messageContext, ConfigKeys.SECONDARY_WAIT_THRESHOLD, DEFAULT_SECONDARY_THRESHOLD);
	}

	private void setAsyncThreshold(MessageContext messageContext, String thresholdOptionName, int defaultValue) {
		logMethodCall("setAsyncThreshold(MessageContext, String, int)");

		BIBusEnvelope requestEnvelope = (BIBusEnvelope) messageContext.getProperty("request.envelope");
		Element methodElement = SoapRequestHelper.getMethod(requestEnvelope);

		int threshold = SoapRequestHelper.getAsyncOptionInt(methodElement, thresholdOptionName);
		if (threshold == -1) {
			JsonObject asyncToolkitOptions = this.bridgeConfig.getObject(ConfigKeys.ASYNC_TOOLKIT);
			Long value = asyncToolkitOptions.getLong(thresholdOptionName);
			threshold = (value != null) ? value.intValue() : defaultValue;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("Setting async threshold to: " + threshold);
		}

		this.setAsyncThreshold(threshold);
	}

	private void logBIBusEnvelope(String message, BIBusEnvelope envelope) {
		try {
			logger.debug(message);
			if (envelope != null) {
				StringWriter strWriter = new StringWriter();
				envelope.writeAsXML(strWriter);
				logger.debug(strWriter.toString());
			} else {
				logger.debug("BIBusEnvelope is null");
			}
		} catch (IOException ex) {
			logger.debug("Failed to log BIBusEnvelope", ex);
		}
	}

	private void logMethodCall(String methodName) {
		if (logger.isDebugEnabled()) {
			logger.debug("Calling " + methodName);
		}
	}

	/*=========================================================================================

		The methods below override the methods in AsyncServiceBase class.
		pogo.async toolkit calls these methods while servicing the asynchronous
		conversation with the client.

		These methods run in the "Default Executor Thread" (unlike invoke method above)

	=========================================================================================*/

	/**
	 * pogo.async toolkit calls this method to let us know that primaryWaitThreshold has expired.
	 * Prepare a "working" response to the primary request.
	 *
	 * @see AsyncServiceBase#workingImpl(MessageContext, AsyncContext)
	 */
	@Override
	public void workingImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("workingImpl(MessageContext, AsyncContext)");
		if (this.methodHandler != null) {
			this.methodHandler.workingImpl(mc, ac);
		} else {
			// primaryWaitThreshold has expired but this.methodHandler is not initialized yet.
			// This can happen if RunMethod's query to CM is slow to respond.

			// Endor R2 quick fix for RTC defect 272487:
			// Prepare "runResponse" without checking the name of the primary method.
			// This is okay because "run" is the only supported primary method in restBridgeService.
			// This code should be adjusted if restBridgeService starts supporting other primary methods,
			// for example "runSpecification".
			SoapResponseHelper helper = new SoapResponseHelper(mc);
			helper.prepareRunResponse(AsynchReplyStatusEnum.working);
		}
	}

	/**
	 * pogo.async toolkit calls this method to let us know that a "wait" request has arrived.
	 * The only known purpose for this is to let us revise the async threshold.
	 *
	 * @see AsyncServiceBase#waitReceived(MessageContext, AsyncContext)
	 */
	@Override
	public void waitReceived(MessageContext mc, AsyncContext ac) {
		logMethodCall("waitReceived(MessageContext, AsyncContext)");

		BIBusEnvelope requestEnvelope = (BIBusEnvelope) mc.getProperty("request.envelope");
		if (logger.isDebugEnabled()) {
			logBIBusEnvelope("wait request: ", requestEnvelope);
		}

		this.setSecondaryAsyncThreshold(mc);
	}

	/**
	 * pogo.async toolkit calls this method to let us know that secondaryWaitThreshold has expired
	 * Prepare a "stillWorking" response to the current "wait" request.
	 *
	 * @see AsyncServiceBase#workingImpl(MessageContext, AsyncContext)
	 */
	@Override
	public void stillWorkingImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("stillWorkingImpl(MessageContext, AsyncContext)");
		SoapResponseHelper helper = new SoapResponseHelper(mc);
		helper.prepareWaitResponse(AsynchReplyStatusEnum.stillWorking);
	}

	/**
	 * pogo.async toolkit calls this method to let us know that the client has cancelled
	 * the async conversation.
	 */
	@Override
	protected void cancelImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("cancelImpl(MessageContext, AsyncContext)");
		if (this.methodHandler != null) {
			this.methodHandler.cancel();
		}
		SoapResponseHelper helper = new SoapResponseHelper(mc);
		helper.prepareCancelResponse(AsynchReplyStatusEnum.complete);
	}

	/**
	 * pogo.async toolkit calls this method to let us know that the client has abandoned
	 * the async conversation. No response is required.
	 */
	@Override
	public void abandonImpl(AsyncContext ac) {
		logMethodCall("abandonImpl(AsyncContext)");
		if (this.methodHandler != null) {
			this.methodHandler.cancel();
		}
	}

	/**
	 * pogo.async toolkit calls this method during a secondary "wait" request,
	 * when invoke() method runs to completion in a separate "async" thread.
	 * Prepare a reply to the current "wait", to complete the conversation.
	 *
	 * @see AsyncServiceBase#responseReadyImpl(MessageContext, AsyncContext)
	 */
	@Override
	public void responseReadyImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("responseReadyImpl(MessageContext, AsyncContext)");
		if (this.methodHandler != null) {
			this.methodHandler.responseReadyImpl(mc, ac);
		} else {
			// Defensive code. We should never get here.
			SoapResponseHelper helper = new SoapResponseHelper(mc);
			helper.prepareWaitResponse(AsynchReplyStatusEnum.conversationComplete);
		}
	}

	@Override
	public void getOutputImpl(MessageContext mc, AsyncContext ac) {
		logMethodCall("getOutputImpl(MessageContext, AsyncContext)");
		// do nothing; should never be called in restBridgeService.
	}

	@Override
	public void releaseImpl(MessageContext mc) {
		logMethodCall("releaseImpl(MessageContext)");
		// do nothing; should never be called in restBridgeService.
	}
}
