/*************************************************************************************************************************************************************
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * IBM Cognos Products: Cognos Analytics
 *
 * (C) Copyright IBM Corp. 2019
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *************************************************************************************************************************************************************/

package com.ibm.bi.rest.bridge.utils;

import org.apache.axis.description.TypeDesc;
import org.apache.log.Hierarchy;
import org.apache.log.Logger;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.QName;

import com.cognos.developer.schemas.bibus._3.AsynchDetail;
import com.cognos.developer.schemas.bibus._3.AsynchReply;
import com.cognos.developer.schemas.bibus._3.AsynchReplyStatusEnum;

import com.cognos.pogo.async.AsyncServiceBase;
import com.cognos.pogo.pdk.BIBusEnvelope;
import com.cognos.pogo.pdk.MessageContext;
import com.cognos.pogo.pdk.SOAPEnvelope;

/**
 * This class is responsible for building restBridgeService responses.
 */
public class SoapResponseHelper {
	
	private static final Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor(SoapResponseHelper.class.getName());

	private final static Namespace NS_RBS = Namespace.get("rbs1", "http://developer.cognos.com/schemas/restBridgeService/1");

	private final static QName runResponse = DocumentHelper.createQName("runResponse", NS_RBS);
	private final static QName waitResponse = DocumentHelper.createQName("waitResponse", NS_RBS);
	private final static QName cancelResponse = DocumentHelper.createQName("cancelResponse", NS_RBS);

	private final MessageContext messageContext;
	private final BIBusEnvelope requestEnvelope;

	public SoapResponseHelper(MessageContext mc) {
		this.messageContext = mc;
		this.requestEnvelope = (BIBusEnvelope) this.messageContext.getProperty("request.envelope");
	}

	public BIBusEnvelope prepareRunResponse(AsynchReplyStatusEnum statusEnum) {
		logMethodCall("prepareRunResponse(AsynchReplyStatusEnum)");
		return this.prepareResponse(runResponse, statusEnum);
	}

	public BIBusEnvelope prepareWaitResponse(AsynchReplyStatusEnum statusEnum) {
		logMethodCall("prepareWaitResponse(AsynchReplyStatusEnum)");
		return this.prepareResponse(waitResponse, statusEnum);
	}

	public BIBusEnvelope prepareCancelResponse(AsynchReplyStatusEnum statusEnum) {
		logMethodCall("prepareWaitResponse(AsynchReplyStatusEnum)");
		return this.prepareResponse(cancelResponse, statusEnum);
	}

	/**
	 * Builds method response.
	 */
	private BIBusEnvelope prepareResponse(QName responseQName, AsynchReplyStatusEnum statusEnum) {
		logMethodCall("prepareResponse(QName, AsynchReplyStatusEnum)");

		/*
			Response example

			<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/">
				<SOAP-ENV:Header>
					...
					...
				</SOAP-ENV:Header>
				<SOAP-ENV:Body>
					<ns2:runResponse xmlns:ns2="http://developer.cognos.com/schemas/restBridgeService/1">
						<bus:result xsi:type="bus:asynchReply">
							<bus:status xsi:type="bus:asynchReplyStatusEnum">conversationComplete</bus:status>
							<bus:details xsi:type="SOAP-ENC:Array" SOAP-ENC:arrayType="bus:asynchDetail[0]"/>
						</bus:result>
					</ns2:runResponse>
				</SOAP-ENV:Body>
			</SOAP-ENV:Envelope>
		*/

		BIBusEnvelope responseEnvelope = AsyncServiceBase.prepareSkeletalResponse(this.requestEnvelope, this.messageContext, null);
		Element responseRoot = responseEnvelope.getBody().addElement(responseQName);

		Element resultElement = this.addBIBusElement(responseRoot, "result", AsynchReply.class);

		Element statusElement = addBIBusElement(resultElement, "status", AsynchReplyStatusEnum.class);
		statusElement.setText(statusEnum.getValue());

		// Add an empty <details> array.
		// Possible future enhancement: put more details in the response.
		addBIBusArray(resultElement, "details", AsynchDetail.class, 0);

		return responseEnvelope;
	}

	/**
	 * Adds a new element to BI Bus SOAP response.
	 * 
	 * @param parentElement
	 *            The parent element to add to.
	 * @param newElementName
	 *            The name of the new element to add.
	 * @param objectModelClass
	 *            The class of the new element.
	 * 
	 * @return Newly created element.
	 */
	private Element addBIBusElement(Element parentElement, String newElementName, Class<?> objectModelClass) {
		QName newElementQName = new QName(newElementName, BIBusEnvelope.NS_BUS);
		Element newElement = parentElement.addElement(newElementQName);
		
		String type = addBIBusPrefix(getOMClassTypeName(objectModelClass));
		newElement.addAttribute(SOAPEnvelope.XSI_TYPE, type); 
		return newElement;
	}
	
	/**
	 * Adds a new "array" element to BI Bus SOAP response.
	 * 
	 * @param parentElement
	 *            The parent element to add to.
	 * @param newElementName
	 *            The name of the new element to add.
	 * @param objectModelClass
	 *            The class of array items.
	 * @param arraySize
	 *            The number of items in the array.
	 * 
	 * @return Newly created element.
	 */
	private Element addBIBusArray(Element parentElement, String newElementName, Class<?> objectModelClass, int arraySize) {
		QName newElementQName = new QName(newElementName, BIBusEnvelope.NS_BUS);
		Element newElement = parentElement.addElement(newElementQName);
		newElement.addAttribute(SOAPEnvelope.XSI_TYPE, SOAPEnvelope.SOAP_ENC_ARRAY);
		
		String type = addBIBusPrefix(getOMClassTypeName(objectModelClass) + "[" + arraySize + "]");
		newElement.addAttribute(SOAPEnvelope.SOAP_ENC_ARRAYTYPE, type);
		
		return newElement;
	}

	/**
	 * Adds BI Bus namespace prefix to a name.
	 */
	private String addBIBusPrefix(String name) {
		return BIBusEnvelope.NS_BUS.getPrefix() + ":" + name;
	}

	/**
	 * Returns the type name for CA object model class.
	 */
	private String getOMClassTypeName(Class<?> objectModelClass) {
		return TypeDesc.getTypeDescForClass(objectModelClass).getXmlType().getLocalPart();
	}

	private void logMethodCall(String methodName) {
		if (logger.isDebugEnabled()) {
			logger.debug("Calling " + methodName);
		}
	}
}
