/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.cognos.jdbc.twc;

import com.ibm.cognos.jdbc.twc.TWCConnectionString;
import com.ibm.cognos.jdbc.twc.csv.CSVSchema;
import com.ibm.cognos.jdbc.twc.messages.TWCMessageKeys;
import com.ibm.cognos.jdbc.twc.messages.TWCMessageUtil;
import com.ibm.cognos.jdbc.twc.org.apache.commons.io.FileUtils;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.SQLException;
import java.time.LocalDate;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

class CacheManager
implements Closeable,
TWCMessageKeys {
    static final String RELATIVE_DIR_ROOT = "twcjdbc";
    private static final String AVRO_EXTENSION = ".avro";
    private static final int ONE_MB = 0x100000;
    private final ConcurrentMap<String, TWCConnectionEntry> conn2DirMap = new ConcurrentHashMap<String, TWCConnectionEntry>();
    private final ConcurrentMap<String, TWCConnectionEntry> closedConn2DirMap = new ConcurrentHashMap<String, TWCConnectionEntry>();
    private ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1, r -> {
        Thread t = new Thread(r);
        t.setDaemon(true);
        t.setName("CacheManager.CleanupScheduler-" + t.getId());
        return t;
    });

    CacheManager(long fixedDelay) {
        CacheMaintenanceTask r2 = new CacheMaintenanceTask(this.conn2DirMap, this.closedConn2DirMap);
        this.scheduler.scheduleWithFixedDelay(r2, 0L, fixedDelay, TimeUnit.SECONDS);
    }

    @Override
    public void close() throws IOException {
        try {
            this.scheduler.shutdown();
            this.scheduler.awaitTermination(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException interruptedException) {
        }
        finally {
            this.scheduler.shutdownNow();
            this.cleanUpSnappyLibraries();
        }
    }

    private void cleanUpSnappyLibraries() {
        File[] matchingFiles;
        String unpackedSnappyLibLocation = System.getProperty("org.xerial.snappy.tempdir");
        if (null == unpackedSnappyLibLocation) {
            unpackedSnappyLibLocation = System.getProperty("java.io.tmpdir");
        }
        File f = new File(unpackedSnappyLibLocation);
        for (File snappyF : matchingFiles = f.listFiles((dir, name) -> name.startsWith("snappy-1.1.2-"))) {
            FileUtils.deleteQuietly(snappyF);
        }
    }

    int getNumOfRegisteredConnections() {
        return this.conn2DirMap.size();
    }

    int getNumOfDeregisteredClosedConnectionsWithPinnedAvroFiles() {
        return this.closedConn2DirMap.size();
    }

    TWCConnectionEntry getRegisteredConnectionEntry(String connUUID) {
        return (TWCConnectionEntry)this.conn2DirMap.get(connUUID);
    }

    void registerConnection(TWCConnectionString connectionStr, String connUUID) throws SQLException {
        try {
            this.conn2DirMap.compute(connUUID, (connectionId, connEntry) -> {
                if (null != connEntry) {
                    throw new IllegalStateException("dummy message");
                }
                Path connectionDir = this.constructConnectionRootDirectory(connectionStr, connUUID);
                connEntry = new TWCConnectionEntry(connectionDir, TimeUnit.SECONDS.toNanos(connectionStr.getCacheTTL()), connectionStr.getCacheSize() * 0x100000L);
                return connEntry;
            });
        }
        catch (IllegalStateException ise) {
            if ("dummy message".equals(ise.getMessage())) {
                throw new SQLException(TWCMessageUtil.getMessage("0063", connUUID));
            }
            throw new SQLException(TWCMessageUtil.getMessage("0061", ise.getMessage(), ise));
        }
    }

    void deregisterConnection(String connUUID) {
        TWCConnectionEntry connEntry = (TWCConnectionEntry)this.conn2DirMap.remove(connUUID);
        if (null != connEntry) {
            connEntry.purgeCachedEntries();
            if (!connEntry.associatedWithPinnedFiles()) {
                FileUtils.deleteQuietly(connEntry.getDirectory().toFile());
            } else {
                this.closedConn2DirMap.put(connUUID, connEntry);
            }
        }
    }

    String[] registerAvroFile(String connUUID, String resultSetId, CSVSchema csvSchema) throws SQLException {
        String avroId;
        String[] avroIdAndAvroPath = new String[]{null, null};
        avroIdAndAvroPath[0] = avroId = UUID.randomUUID().toString();
        try {
            TWCConnectionEntry connectionEntry = (TWCConnectionEntry)this.conn2DirMap.get(connUUID);
            if (null == connectionEntry) {
                throw new IllegalStateException();
            }
            Path connectionPath = connectionEntry.getDirectory();
            Path avroPath = Paths.get(connectionPath.toString(), avroId + AVRO_EXTENSION);
            avroIdAndAvroPath[1] = avroPath.toString();
            connectionEntry.registerAvroEntry(avroId, resultSetId, avroPath, csvSchema);
        }
        catch (IllegalStateException ise) {
            throw new SQLException(TWCMessageUtil.getMessage("0062", connUUID));
        }
        return avroIdAndAvroPath;
    }

    void deregisterAvroFile(String connUUID, String resultSetId, String avroId) {
        TWCConnectionEntry connectionEntry = (TWCConnectionEntry)this.conn2DirMap.get(connUUID);
        if (null != connectionEntry) {
            connectionEntry.deregisterAvroEntry(avroId, resultSetId);
        } else {
            this.closedConn2DirMap.computeIfPresent(connUUID, (uuid, entry) -> {
                entry.deregisterAvroEntry(avroId, resultSetId);
                if (!entry.associatedWithPinnedFiles()) {
                    FileUtils.deleteQuietly(entry.getDirectory().toFile());
                    return null;
                }
                return entry;
            });
        }
    }

    void cacheAvroFile(String connUUID, String avroId, CacheEntry cacheInfo, long size, long rowCount) {
        TWCConnectionEntry connectionEntry = (TWCConnectionEntry)this.conn2DirMap.get(connUUID);
        if (null != connectionEntry) {
            connectionEntry.cacheAvroEntry(avroId, cacheInfo, size, rowCount);
        }
    }

    Object[] retrieveCachedAvroFile(String connUUID, CacheEntry cacheKey, String resultSetId) {
        Object[] result;
        TWCConnectionEntry connectionEntry;
        Object[] idPathRowsCSVSchemaTuple = new Object[]{null, null, null, null};
        if (null != cacheKey && null != (connectionEntry = (TWCConnectionEntry)this.conn2DirMap.get(connUUID)) && null != (result = connectionEntry.retrieveCachedAvroFile(cacheKey, resultSetId))[0] && null != result[1] && null != result[2] && null != result[3]) {
            idPathRowsCSVSchemaTuple[0] = result[0];
            idPathRowsCSVSchemaTuple[1] = result[1];
            idPathRowsCSVSchemaTuple[2] = result[2];
            idPathRowsCSVSchemaTuple[3] = result[3];
        }
        return idPathRowsCSVSchemaTuple;
    }

    Path constructConnectionRootDirectory(TWCConnectionString connStr, String uuid) {
        Path absoluteConnPath = Paths.get(connStr.getCacheDir().toString(), RELATIVE_DIR_ROOT, uuid);
        try {
            File f = absoluteConnPath.toFile();
            if (f.exists()) {
                FileUtils.deleteDirectory(f);
            }
            if (f.mkdirs()) {
                return absoluteConnPath;
            }
            throw new IOException();
        }
        catch (Exception e) {
            throw new IllegalStateException(absoluteConnPath.toString(), e);
        }
    }

    private static final class CacheMaintenanceTask
    implements Runnable {
        private final ConcurrentMap<String, TWCConnectionEntry> connections;
        private final ConcurrentMap<String, TWCConnectionEntry> closedConnections;

        private CacheMaintenanceTask(ConcurrentMap<String, TWCConnectionEntry> conns, ConcurrentMap<String, TWCConnectionEntry> closedConns) {
            this.connections = Objects.requireNonNull(conns, TWCMessageUtil.getMessage("0007", "conns"));
            this.closedConnections = Objects.requireNonNull(closedConns, TWCMessageUtil.getMessage("0007", "closedConns"));
        }

        @Override
        public void run() {
            try {
                for (TWCConnectionEntry currentConn : this.connections.values()) {
                    long currentTTL = currentConn.getTimeToLive();
                    List<Object> purgedCandidates = new LinkedList();
                    if (currentTTL > 0L) {
                        purgedCandidates = CacheMaintenanceTask.getPurgeCadidatesByTTL(currentConn, currentTTL);
                    }
                    for (CacheEntry expiredCacheEntry : purgedCandidates) {
                        if (System.nanoTime() - expiredCacheEntry.getLastAccessed() <= currentTTL) continue;
                        currentConn.purgeCacheEntryAndTryEraseAvroFile(expiredCacheEntry);
                    }
                }
                for (TWCConnectionEntry currentConn : this.connections.values()) {
                    long currentTotalSpace = currentConn.getTotalSpace();
                    long currentAvailableSpace = currentConn.getAvailableSpace();
                    List<Object> purgedCandidates = new LinkedList();
                    if (currentTotalSpace > 0L && (double)currentAvailableSpace < 0.1 * (double)currentTotalSpace) {
                        purgedCandidates = CacheMaintenanceTask.getPurgeCadidatesBySize(currentConn, currentTotalSpace, currentAvailableSpace);
                    }
                    for (CacheEntry expiredCacheEntry : purgedCandidates) {
                        currentConn.purgeCacheEntryAndTryEraseAvroFile(expiredCacheEntry);
                    }
                }
                for (TWCConnectionEntry currentConn : this.closedConnections.values()) {
                    currentConn.purgeCachedEntries();
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }

        private static List<CacheEntry> getPurgeCadidatesByTTL(TWCConnectionEntry connectionEntry, long currentTTL) {
            LinkedList<CacheEntry> candidates = new LinkedList<CacheEntry>();
            for (CacheEntry currentEntry : connectionEntry.getCachedEntries()) {
                if (System.nanoTime() - currentEntry.getLastAccessed() <= currentTTL) continue;
                candidates.add(currentEntry);
            }
            return candidates;
        }

        private static List<CacheEntry> getPurgeCadidatesBySize(TWCConnectionEntry connectionEntry, long currentTotalSpace, long currentAvailableSpace) {
            CacheEntry currentEntry;
            LinkedList<CacheEntry> candidates = new LinkedList<CacheEntry>();
            Iterator<CacheEntry> cacheIt = connectionEntry.getCachedEntries().iterator();
            for (long sizeThreshold = currentTotalSpace / 2L - currentAvailableSpace; sizeThreshold > 0L && cacheIt.hasNext(); sizeThreshold -= currentEntry.getSize()) {
                currentEntry = cacheIt.next();
                candidates.add(currentEntry);
            }
            return candidates;
        }
    }

    static final class CacheEntry
    implements Comparable<CacheEntry> {
        private volatile String avroId;
        private final List<String> tables;
        private final List<String> postalCodes;
        private final List<LocalDate> dates;
        private volatile long lastAccessed = 0L;
        private volatile long size = 0L;
        private final AtomicBoolean invalidated = new AtomicBoolean();

        CacheEntry(List<String> tableNames, List<String> postalKeys, List<LocalDate> dateRanges) {
            Objects.requireNonNull(tableNames, TWCMessageUtil.getMessage("0007", "tableNames"));
            Objects.requireNonNull(postalKeys, TWCMessageUtil.getMessage("0007", "postalKeys"));
            Objects.requireNonNull(dateRanges, TWCMessageUtil.getMessage("0007", "dateRanges"));
            if (tableNames.isEmpty()) {
                throw new IllegalArgumentException(TWCMessageUtil.getMessage("0008", "tableNames"));
            }
            if (postalKeys.isEmpty()) {
                throw new IllegalArgumentException(TWCMessageUtil.getMessage("0008", "postalKeys"));
            }
            if (dateRanges.isEmpty()) {
                throw new IllegalArgumentException(TWCMessageUtil.getMessage("0008", "dateRanges"));
            }
            this.tables = Collections.unmodifiableList(tableNames);
            this.postalCodes = Collections.unmodifiableList(postalKeys);
            this.dates = Collections.unmodifiableList(dateRanges);
        }

        void setAvroId(String avroFileId) {
            if (this.isValid()) {
                this.avroId = avroFileId;
            }
        }

        String getAvroId() {
            return this.avroId;
        }

        void updateLastAccessed() {
            if (this.isValid()) {
                this.lastAccessed = System.nanoTime();
            }
        }

        long getLastAccessed() {
            return this.lastAccessed;
        }

        void updateSize(long theSize) {
            if (this.isValid()) {
                this.size = theSize;
            }
        }

        long getSize() {
            return this.size;
        }

        boolean markAsInvalid() {
            return this.invalidated.compareAndSet(false, true);
        }

        boolean isCompatible(CacheEntry providedKey) {
            if (!this.isValid()) {
                return false;
            }
            if (this == providedKey) {
                return true;
            }
            boolean result = false;
            if (this.isValid()) {
                boolean bl = result = this.tables.containsAll(providedKey.tables) && this.postalCodes.containsAll(providedKey.postalCodes) && this.isValid();
                if (result) {
                    Iterator<LocalDate> providedDatesIt = providedKey.dates.iterator();
                    while (result && providedDatesIt.hasNext()) {
                        LocalDate providedDateStart = providedDatesIt.next();
                        LocalDate providedDateEnd = providedDatesIt.next();
                        Iterator<LocalDate> thisDatesIt = this.dates.iterator();
                        while (result && thisDatesIt.hasNext()) {
                            LocalDate thisDateStart = thisDatesIt.next();
                            LocalDate thisDateEnd = thisDatesIt.next();
                            if (thisDateEnd.isBefore(providedDateStart) && !thisDatesIt.hasNext()) {
                                result = false;
                            } else if (!thisDateEnd.isBefore(providedDateStart)) {
                                boolean bl2 = result = !(!thisDateStart.isBefore(providedDateStart) && !thisDateStart.isEqual(providedDateStart) || !thisDateEnd.isAfter(providedDateEnd) && !thisDateEnd.isEqual(providedDateEnd));
                                if (result) break;
                            }
                            if (providedDateEnd.isBefore(thisDateStart) || providedDateEnd.isEqual(thisDateStart)) {
                                result = false;
                            }
                            result = result && this.isValid();
                        }
                        result = result && this.isValid();
                    }
                }
            }
            return result;
        }

        boolean isValid() {
            return !this.invalidated.get();
        }

        public int hashCode() {
            return (int)(this.lastAccessed & Integer.MAX_VALUE);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof CacheEntry) {
                return this.compareTo((CacheEntry)obj) == 0;
            }
            return false;
        }

        @Override
        public int compareTo(CacheEntry o) {
            if (null == o) {
                return 1;
            }
            long diff = o.getLastAccessed() - this.lastAccessed;
            if (diff < 0L) {
                return -1;
            }
            if (diff > 0L) {
                return 1;
            }
            return 0;
        }
    }

    static final class TWCAvroEntry {
        private final Path avroPath;
        private final Set<String> resultSets = ConcurrentHashMap.newKeySet();
        private final AtomicReference<CacheEntry> cacheEntry;
        private final CSVSchema csvSchema;
        private volatile long size;
        private volatile long rowCount;

        private TWCAvroEntry(Path path, CSVSchema schemaFields) {
            this.avroPath = path;
            this.csvSchema = schemaFields;
            this.cacheEntry = new AtomicReference();
            this.size = 0L;
            this.rowCount = 0L;
        }

        Path getAvroPath() {
            return this.avroPath;
        }

        CSVSchema getCSVSchema() {
            return this.csvSchema;
        }

        void registerResultSet(String resultSetId) {
            this.resultSets.add(resultSetId);
        }

        boolean deregisterResultSet(String resultSetId) {
            return this.resultSets.remove(resultSetId);
        }

        boolean isPinned() {
            return !this.resultSets.isEmpty();
        }

        int getActiveResultSets() {
            return this.resultSets.size();
        }

        boolean isCacheable() {
            return null != this.cacheEntry.get();
        }

        long getSize() {
            return this.size;
        }

        long getRowCount() {
            return this.rowCount;
        }

        void setCacheEntry(CacheEntry cacheInfo, long diskSize, long numOfRows) {
            if (this.cacheEntry.compareAndSet(null, cacheInfo)) {
                this.rowCount = numOfRows;
                this.size = diskSize;
            }
        }

        long purgeCacheEntry(CacheEntry cacheInfo) {
            if (this.cacheEntry.compareAndSet(cacheInfo, null)) {
                long diskSize = this.size;
                this.rowCount = 0L;
                this.size = 0L;
                return diskSize;
            }
            return 0L;
        }
    }

    static final class TWCConnectionEntry {
        private final Path dir;
        private final long ttl;
        private final long totalSpace;
        private final ConcurrentMap<String, TWCAvroEntry> avroFiles = new ConcurrentHashMap<String, TWCAvroEntry>();
        private final ConcurrentSkipListSet<CacheEntry> cacheEntries = new ConcurrentSkipListSet();
        private volatile long spaceUsed = 0L;

        private TWCConnectionEntry(Path directory, long timeToLive, long maxStorageSpace) {
            this.dir = directory;
            this.ttl = timeToLive;
            this.totalSpace = maxStorageSpace;
        }

        Path getDirectory() {
            return this.dir;
        }

        long getTimeToLive() {
            return this.ttl;
        }

        long getTotalSpace() {
            return this.totalSpace;
        }

        long getAvailableSpace() {
            return this.totalSpace - this.spaceUsed;
        }

        ConcurrentSkipListSet<CacheEntry> getCachedEntries() {
            return this.cacheEntries;
        }

        ConcurrentMap<String, TWCAvroEntry> getAvroFiles() {
            return this.avroFiles;
        }

        void registerAvroEntry(String avroFileId, String resultSetId, Path avroPath, CSVSchema csvSchema) {
            TWCAvroEntry avroEntry = new TWCAvroEntry(avroPath, csvSchema);
            avroEntry.registerResultSet(resultSetId);
            this.avroFiles.put(avroFileId, avroEntry);
        }

        void deregisterAvroEntry(String avroFileId, String resultSetId) {
            this.avroFiles.computeIfPresent(avroFileId, (key, entry) -> {
                entry.deregisterResultSet(resultSetId);
                if (entry.isPinned() || entry.isCacheable()) {
                    return entry;
                }
                FileUtils.deleteQuietly(entry.getAvroPath().toFile());
                this.spaceUsed -= entry.getSize();
                return null;
            });
        }

        void cacheAvroEntry(String avroFileId, CacheEntry cacheInfo, long size, long numOfRows) {
            this.avroFiles.computeIfPresent(avroFileId, (key, entry) -> {
                if (this.ttl != 0L || this.getAvailableSpace() - size > 0L) {
                    entry.setCacheEntry(cacheInfo, size, numOfRows);
                    this.spaceUsed += size;
                    cacheInfo.setAvroId(avroFileId);
                    cacheInfo.updateLastAccessed();
                    cacheInfo.updateSize(size);
                    this.cacheEntries.add(cacheInfo);
                }
                return entry;
            });
        }

        Object[] retrieveCachedAvroFile(CacheEntry cacheKey, String resultSetId) {
            Object[] idPathRowsCSVSchemaTuple = new Object[]{null, null, null, null};
            Iterator<CacheEntry> it = this.cacheEntries.iterator();
            CacheEntry matchingEntry = null;
            while (null == matchingEntry && it.hasNext()) {
                CacheEntry current = it.next();
                if (!current.isCompatible(cacheKey)) continue;
                matchingEntry = current;
            }
            CacheEntry finalMatchingEntry = matchingEntry;
            if (null != finalMatchingEntry) {
                this.avroFiles.computeIfPresent(matchingEntry.getAvroId(), (key, entry) -> {
                    if (entry.isCacheable() && finalMatchingEntry.isValid()) {
                        idPathRowsCSVSchemaTuple[0] = key;
                        idPathRowsCSVSchemaTuple[1] = entry.getAvroPath().toString();
                        idPathRowsCSVSchemaTuple[2] = entry.getRowCount();
                        idPathRowsCSVSchemaTuple[3] = entry.getCSVSchema();
                        entry.registerResultSet(resultSetId);
                        this.cacheEntries.remove(finalMatchingEntry);
                        finalMatchingEntry.updateLastAccessed();
                        this.cacheEntries.add(finalMatchingEntry);
                    }
                    return entry;
                });
            }
            return idPathRowsCSVSchemaTuple;
        }

        boolean associatedWithPinnedFiles() {
            boolean result = false;
            Iterator it = this.avroFiles.values().iterator();
            while (!result && it.hasNext()) {
                TWCAvroEntry entry = (TWCAvroEntry)it.next();
                result = entry.isPinned();
            }
            return result;
        }

        void purgeCachedEntries() {
            CacheEntry polledEntry = this.cacheEntries.pollFirst();
            while (null != polledEntry) {
                polledEntry.markAsInvalid();
                CacheEntry finalPolledEntry = polledEntry;
                this.avroFiles.computeIfPresent(polledEntry.getAvroId(), (key, entry) -> {
                    this.spaceUsed -= entry.purgeCacheEntry(finalPolledEntry);
                    if (entry.isPinned()) {
                        return entry;
                    }
                    FileUtils.deleteQuietly(entry.getAvroPath().toFile());
                    return null;
                });
                polledEntry = this.cacheEntries.pollFirst();
            }
        }

        void purgeCacheEntryAndTryEraseAvroFile(CacheEntry cacheEntry) {
            cacheEntry.markAsInvalid();
            this.avroFiles.computeIfPresent(cacheEntry.getAvroId(), (key, entry) -> {
                this.spaceUsed -= entry.purgeCacheEntry(cacheEntry);
                if (entry.isPinned()) {
                    return entry;
                }
                FileUtils.deleteQuietly(entry.getAvroPath().toFile());
                return null;
            });
            this.cacheEntries.remove(cacheEntry);
        }
    }
}

