/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.bi.platform.datasetutils.parquet;

import com.ibm.bi.platform.datasetutils.metadata.ColumnMetadata;
import com.ibm.bi.platform.datasetutils.metadata.RowSchema;
import com.ibm.bi.platform.datasetutils.metadata.types.DecimalType;
import com.ibm.bi.platform.datasetutils.metadata.types.TimeType;
import com.ibm.bi.platform.datasetutils.metadata.types.TimestampType;
import com.ibm.bi.platform.datasetutils.metadata.types.Type;
import com.ibm.bi.platform.datasetutils.parquet.ParquetMetadataGenerator;
import com.ibm.bi.platform.datasetutils.parquet.ParquetOptions;
import com.ibm.bi.platform.datasetutils.parquet.ParquetVersionEnum;
import com.ibm.bi.platform.datasetutils.utils.DateTimeUtils;
import com.ibm.bi.platform.datasetutils.utils.DecimalUtils;
import com.ibm.bi.platform.datasetutils.utils.ReadableRecord;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.text.Normalizer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Map;
import org.apache.parquet.hadoop.api.WriteSupport;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.io.api.RecordConsumer;
import org.apache.parquet.schema.MessageType;
import shaded.org.apache.hadoop.conf.Configuration;

class DatasetWriteSupport
extends WriteSupport<ReadableRecord> {
    private final RowSchema recordSchema;
    private final ParquetOptions options;
    private RecordFieldValueWriter[] valueWriters;
    private RecordConsumer recordConsumer;
    private Normalizer.Form normalizerForm;
    private ZoneId timeZoneId;
    private ZoneOffset timeZoneOffset;

    public DatasetWriteSupport(RowSchema recordSchema, ParquetOptions options) {
        if (null == recordSchema) {
            throw new NullPointerException("recordSchema was null");
        }
        this.recordSchema = recordSchema;
        this.options = null == options ? new ParquetOptions() : new ParquetOptions(options);
    }

    public WriteSupport.WriteContext init(Configuration config) {
        ParquetVersionEnum parquetCAVersion = this.options.getParquetCAVersion();
        switch (parquetCAVersion) {
            case LEGACY: {
                this.options.setLegacyParquetCompatibility();
                break;
            }
        }
        ParquetMetadataGenerator metaGenerator = new ParquetMetadataGenerator(this.options);
        metaGenerator.process(this.recordSchema);
        Map<String, String> extraMetadata = metaGenerator.getExtraMetadata();
        MessageType schema = metaGenerator.getMessageType();
        extraMetadata.put("parquet.ca.version", parquetCAVersion.getValue());
        if (this.options.isUnicodeNormalizeEnabled()) {
            this.normalizerForm = this.options.getUnicodeNormalizerForm();
            extraMetadata.put("parquet.unicode.normalize", "true");
            extraMetadata.put("parquet.unicode.normalizer.form", this.normalizerForm.name());
        }
        this.timeZoneId = this.options.getTimeZoneId();
        this.timeZoneOffset = this.options.getTimeZoneOffset();
        extraMetadata.put("parquet.time.zone-id", this.timeZoneId.getId());
        extraMetadata.put("parquet.time.zone-offset-id", this.timeZoneOffset.getId());
        int numFields = this.recordSchema.getNumOfColumns();
        this.valueWriters = new RecordFieldValueWriter[numFields];
        for (int fieldIndex = 0; fieldIndex < numFields; ++fieldIndex) {
            ColumnMetadata fieldMeta = this.recordSchema.getColumnMetadata(fieldIndex);
            this.valueWriters[fieldIndex] = this.createValueWriter(fieldMeta);
        }
        return new WriteSupport.WriteContext(schema, extraMetadata);
    }

    private RecordFieldValueWriter createValueWriter(ColumnMetadata fieldMeta) {
        RecordFieldValueWriter writer;
        Type fieldType = fieldMeta.getType();
        switch (fieldType.getIntrinsicType()) {
            case BINARY: {
                writer = this::writeBinary;
                break;
            }
            case BOOLEAN: {
                writer = this::writeBoolean;
                break;
            }
            case BYTE: {
                writer = this::writeByte;
                break;
            }
            case DATE: {
                writer = this.createDateWriter();
                break;
            }
            case DECIMAL: {
                writer = this.createDecimalWriter((DecimalType)fieldType);
                break;
            }
            case DOUBLE: {
                writer = this::writeDouble;
                break;
            }
            case FLOAT: {
                writer = this::writeFloat;
                break;
            }
            case INT: {
                writer = this::writeInt;
                break;
            }
            case LONG: {
                writer = this::writeLong;
                break;
            }
            case SHORT: {
                writer = this::writeShort;
                break;
            }
            case STRING: {
                writer = this.createStringWriter();
                break;
            }
            case TIME: {
                writer = this.createTimeWriter((TimeType)fieldType);
                break;
            }
            case TIMESTAMP: {
                writer = this.createTimestampWriter((TimestampType)fieldType);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported data type: " + (Object)((Object)fieldType.getIntrinsicType()));
            }
        }
        return writer;
    }

    private RecordFieldValueWriter createDateWriter() {
        if (this.options.isDateAsEpochMillisEnabled()) {
            return this::writeDateAsEpochMillis;
        }
        return this::writeDateAsEpochDays;
    }

    private RecordFieldValueWriter createTimestampWriter(TimestampType type) {
        if (this.options.isTimestampAsGroupEnabled()) {
            return (record, fieldIndex, consumer) -> this.writeTimestampAsGroup(record, fieldIndex, consumer, type.hasTimeZone());
        }
        return (record, fieldIndex, consumer) -> this.writeTimestampAsInt96(record, fieldIndex, consumer, type.hasTimeZone());
    }

    private RecordFieldValueWriter createTimeWriter(TimeType type) {
        if (this.options.isTimestampAsGroupEnabled()) {
            return (record, fieldIndex, consumer) -> this.writeTimeAsGroup(record, fieldIndex, consumer, type.hasTimeZone());
        }
        return (record, fieldIndex, consumer) -> this.writeTimeAsInt96(record, fieldIndex, consumer, type.hasTimeZone());
    }

    private RecordFieldValueWriter createDecimalWriter(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        if (precision <= 9) {
            return (record, fieldIndex, consumer) -> {
                BigDecimal decimal = record.getDecimal(fieldIndex);
                int unscaledInt = DecimalUtils.decimalToUnscaledInt(decimal, precision, scale);
                consumer.addInteger(unscaledInt);
            };
        }
        if (precision <= 18) {
            return (record, fieldIndex, consumer) -> {
                BigDecimal decimal = record.getDecimal(fieldIndex);
                long unscaledLong = DecimalUtils.decimalToUnscaledLong(decimal, precision, scale);
                consumer.addLong(unscaledLong);
            };
        }
        if (this.options.isDecimalAsStringEnabled()) {
            return (record, fieldIndex, consumer) -> {
                BigDecimal decimal = record.getDecimal(fieldIndex);
                Binary decStr = Binary.fromString((String)decimal.toString());
                consumer.addBinary(decStr);
            };
        }
        return (record, fieldIndex, consumer) -> {
            BigDecimal decimal = record.getDecimal(fieldIndex);
            Binary unscaledBytes = DecimalUtils.decimalToUnscaledBytes(decimal, precision, scale);
            consumer.addBinary(unscaledBytes);
        };
    }

    private RecordFieldValueWriter createStringWriter() {
        if (this.options.isUnicodeNormalizeEnabled()) {
            return this::writeNormalizedString;
        }
        return this::writeString;
    }

    public void prepareForWrite(RecordConsumer consumer) {
        this.recordConsumer = consumer;
    }

    public void write(ReadableRecord record) {
        this.recordConsumer.startMessage();
        this.writeRecord(record);
        this.recordConsumer.endMessage();
    }

    private void writeRecord(ReadableRecord record) {
        int numFields = this.recordSchema.getNumOfColumns();
        if (numFields != record.size()) {
            throw new IllegalArgumentException(String.format("Expected record with %d fields, but got %d fields", numFields, record.size()));
        }
        for (int fieldIndex = 0; fieldIndex < numFields; ++fieldIndex) {
            ColumnMetadata fieldMeta = this.recordSchema.getColumnMetadata(fieldIndex);
            this.writeRecordField(record, fieldIndex, fieldMeta);
        }
    }

    private void writeRecordField(ReadableRecord record, int fieldIndex, ColumnMetadata fieldMeta) {
        String name = fieldMeta.getName();
        if (record.isNull(fieldIndex)) {
            if (!fieldMeta.isNullable()) {
                throw new IllegalArgumentException(String.format("Got null value for non-nullable field (%s) at index %d", name, fieldIndex));
            }
            return;
        }
        this.recordConsumer.startField(name, fieldIndex);
        this.valueWriters[fieldIndex].writeValue(record, fieldIndex, this.recordConsumer);
        this.recordConsumer.endField(name, fieldIndex);
    }

    private void writeBinary(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addBinary(Binary.fromReusedByteBuffer((ByteBuffer)record.getBinary(fieldIndex)));
    }

    private void writeBoolean(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addBoolean(record.getBoolean(fieldIndex));
    }

    private void writeByte(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addInteger((int)record.getByte(fieldIndex));
    }

    private void writeShort(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addInteger((int)record.getShort(fieldIndex));
    }

    private void writeInt(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addInteger(record.getInt(fieldIndex));
    }

    private void writeLong(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addLong(record.getLong(fieldIndex));
    }

    private void writeFloat(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addFloat(record.getFloat(fieldIndex));
    }

    private void writeDouble(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addDouble(record.getDouble(fieldIndex));
    }

    private void writeDateAsEpochDays(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        LocalDate localDate = record.getDate(fieldIndex);
        consumer.addInteger((int)localDate.toEpochDay());
    }

    private void writeDateAsEpochMillis(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        LocalDate localDate = record.getDate(fieldIndex);
        Instant instant = localDate.atStartOfDay(this.timeZoneId).toInstant();
        consumer.addLong(instant.toEpochMilli());
    }

    private void writeTimeAsGroup(ReadableRecord record, int fieldIndex, RecordConsumer consumer, boolean hasTimeZoneOffset) {
        OffsetTime time = hasTimeZoneOffset ? record.getOffsetTime(fieldIndex) : record.getTime(fieldIndex).atOffset(this.timeZoneOffset);
        OffsetDateTime timeOfEpochDay = time.atDate(DateTimeUtils.EPOCH_ZERO_DATE);
        consumer.startGroup();
        consumer.startField("ms", 0);
        consumer.addLong(timeOfEpochDay.toEpochSecond() * 1000L);
        consumer.endField("ms", 0);
        consumer.startField("ns", 1);
        consumer.addInteger(timeOfEpochDay.getNano());
        consumer.endField("ns", 1);
        if (hasTimeZoneOffset) {
            consumer.startField("tz", 2);
            if (this.options.isTimeZoneOffsetInMinutesBehindUTCEnabled()) {
                consumer.addInteger(-time.getOffset().getTotalSeconds() * 60);
            } else {
                consumer.addInteger(time.getOffset().getTotalSeconds() * 1000);
            }
            consumer.endField("tz", 2);
        }
        consumer.endGroup();
    }

    private void writeTimestampAsGroup(ReadableRecord record, int fieldIndex, RecordConsumer consumer, boolean hasTimeZoneOffset) {
        ZonedDateTime timestamp = hasTimeZoneOffset ? record.getOffsetDateTime(fieldIndex).toZonedDateTime() : record.getDateTime(fieldIndex).atZone(this.timeZoneId);
        consumer.startGroup();
        consumer.startField("ms", 0);
        consumer.addLong(timestamp.toEpochSecond() * 1000L);
        consumer.endField("ms", 0);
        consumer.startField("ns", 1);
        consumer.addInteger(timestamp.getNano());
        consumer.endField("ns", 1);
        if (hasTimeZoneOffset) {
            consumer.startField("tz", 2);
            if (this.options.isTimeZoneOffsetInMinutesBehindUTCEnabled()) {
                consumer.addInteger(-timestamp.getOffset().getTotalSeconds() * 60);
            } else {
                consumer.addInteger(timestamp.getOffset().getTotalSeconds() * 1000);
            }
            consumer.endField("tz", 2);
        }
        consumer.endGroup();
    }

    private void writeTimeAsInt96(ReadableRecord record, int fieldIndex, RecordConsumer consumer, boolean hasTimeZoneOffset) {
        if (hasTimeZoneOffset) {
            OffsetTime time = record.getOffsetTime(fieldIndex);
            consumer.addBinary(DateTimeUtils.toInt96(time, this.timeZoneOffset));
        } else {
            LocalTime time = record.getTime(fieldIndex);
            consumer.addBinary(DateTimeUtils.toInt96(time));
        }
    }

    private void writeTimestampAsInt96(ReadableRecord record, int fieldIndex, RecordConsumer consumer, boolean hasTimeZoneOffset) {
        if (hasTimeZoneOffset) {
            OffsetDateTime timestamp = record.getOffsetDateTime(fieldIndex);
            consumer.addBinary(DateTimeUtils.toInt96(timestamp, this.timeZoneId));
        } else {
            LocalDateTime timestamp = record.getDateTime(fieldIndex);
            consumer.addBinary(DateTimeUtils.toInt96(timestamp));
        }
    }

    private void writeString(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        consumer.addBinary(Binary.fromString((String)record.getString(fieldIndex)));
    }

    private void writeNormalizedString(ReadableRecord record, int fieldIndex, RecordConsumer consumer) {
        String normalizedStr = Normalizer.normalize(record.getString(fieldIndex), this.normalizerForm);
        consumer.addBinary(Binary.fromString((String)normalizedStr));
    }

    @FunctionalInterface
    static interface RecordFieldValueWriter {
        public void writeValue(ReadableRecord var1, int var2, RecordConsumer var3);
    }
}

