/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.ba.flint.client;

import com.ibm.ba.flint.client.FlintClient;
import com.ibm.ba.flint.client.FlintClientFactory;
import com.ibm.ba.flint.client.Services;
import java.time.Duration;
import java.time.Instant;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlintClientPool {
    private static final Logger LOGGER = LoggerFactory.getLogger(FlintClientPool.class);
    private final FlintClientFactory factory;
    private final int coreSize;
    private final int idleSeconds;
    private final AtomicBoolean maintenanceRunning = new AtomicBoolean(false);
    private final ConcurrentLinkedQueue<Entry> entries = new ConcurrentLinkedQueue();
    private volatile Instant lastMaintenanceTime = Instant.now();

    public FlintClientPool(FlintClientFactory factory, int coreSize, int idleSeconds) {
        if (null == factory) {
            throw new NullPointerException("clientFactory was null");
        }
        if (coreSize < 0) {
            throw new IllegalArgumentException("coreSize must be >= 0");
        }
        if (idleSeconds < 1) {
            throw new IllegalArgumentException("idleSeconds must be >= 1");
        }
        this.factory = factory;
        this.coreSize = coreSize;
        this.idleSeconds = idleSeconds;
    }

    public int size() {
        return this.entries.size();
    }

    public int borrowed() {
        return (int)this.entries.stream().filter(e -> ((Entry)e).locked.get()).count();
    }

    public int available() {
        return (int)this.entries.stream().filter(e -> !((Entry)e).locked.get()).count();
    }

    public FlintClient borrowClient() throws TException {
        Iterator<Entry> iter = this.entries.iterator();
        FlintClient client = null;
        while (iter.hasNext()) {
            Entry entry = iter.next();
            if (!entry.tryLock()) continue;
            if (entry.validate()) {
                LOGGER.trace("Borrowing existing client from pool");
                entry.stampLastAccessTime();
                client = entry.client;
                break;
            }
            LOGGER.warn("Removing invalid client from pool");
            iter.remove();
            entry.client.close();
        }
        if (null == client) {
            client = this.factory.create();
            Entry newEntry = new Entry(client);
            newEntry.tryLock();
            newEntry.stampLastAccessTime();
            this.entries.add(newEntry);
            LOGGER.trace("Created new client");
        }
        this.maintain();
        return client;
    }

    public void returnClient(FlintClient client) {
        this.returnClient(client, false);
    }

    public void returnClient(FlintClient client, boolean notReusable) {
        this.maintain();
        LOGGER.trace("Returning client to pool (notReusable={})", (Object)notReusable);
        Iterator<Entry> iter = this.entries.iterator();
        while (iter.hasNext()) {
            Entry entry = iter.next();
            if (entry.client != client) continue;
            if (notReusable) {
                iter.remove();
                entry.invalid = true;
                entry.client.close();
            } else {
                entry.stampLastAccessTime();
                entry.unlock();
            }
            return;
        }
        throw new IllegalArgumentException("client was not created by the pool");
    }

    public void close() {
        LOGGER.info("Closing pool");
        for (Entry entry : this.entries) {
            entry.client.close();
        }
        this.entries.clear();
    }

    public String toString() {
        return new ToStringBuilder((Object)this, ToStringStyle.SHORT_PREFIX_STYLE).append("size", this.size()).append("borrowed", this.borrowed()).toString();
    }

    private void maintain() {
        Duration elapsed = Duration.between(this.lastMaintenanceTime, Instant.now());
        if (elapsed.getSeconds() >= (long)this.idleSeconds && this.maintenanceRunning.compareAndSet(false, true)) {
            try {
                this.lastMaintenanceTime = Instant.now();
                this.removeIdleEntries();
            }
            finally {
                this.maintenanceRunning.set(false);
            }
        }
    }

    void removeIdleEntries() {
        Iterator<Entry> iter = this.entries.iterator();
        while (iter.hasNext() && this.size() > this.coreSize) {
            Entry entry = iter.next();
            if (entry.idleTime().getSeconds() < (long)this.idleSeconds || !entry.tryLock()) continue;
            LOGGER.info("Removing idle client (clientId={}) from pool", (Object)entry.client.getClientId());
            iter.remove();
            entry.client.close();
        }
    }

    private static final class Entry {
        private final FlintClient client;
        private final AtomicBoolean locked = new AtomicBoolean(true);
        private volatile Instant lastAccessTime = Instant.now();
        private volatile boolean invalid = false;

        Entry(FlintClient theClient) {
            this.client = theClient;
        }

        boolean tryLock() {
            return this.locked.compareAndSet(false, true);
        }

        void unlock() {
            if (!this.locked.get()) {
                throw new IllegalStateException("entry was not locked");
            }
            this.locked.set(false);
        }

        void stampLastAccessTime() {
            this.lastAccessTime = Instant.now();
        }

        Duration idleTime() {
            return Duration.between(this.lastAccessTime, Instant.now());
        }

        boolean validate() {
            if (this.invalid) {
                return false;
            }
            if (!this.client.isOpen()) {
                this.invalid = true;
                return false;
            }
            try {
                this.client.getService(Services.CORE_SERVICE).ping();
                return true;
            }
            catch (Exception ex) {
                this.invalid = true;
                return false;
            }
        }
    }
}

