"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("../utils/fs");
const transpile = require("./transpile");
const constants = require("../constants");
const definitions_1 = require("./definitions");
const constants_1 = require("../constants");
const events_1 = require("../utils/events");
/**
 * Prepare a bundle by generating vizdef, manifest and meta and copying static files.
 * @param _path Root path of the bundle.
 * @returns {Promise<void>} A promise that resolves when preparation has finished.
 */
async function prepareBundle(_path) {
    const outputPath = path.join(_path, constants_1.OUTPUT_DIR);
    // Clean up previous build & ensure the output path exists.
    await fs.remove(outputPath);
    await fs.ensureDir(outputPath);
    // Start generating vizdef.xml, manifest.xml en meta.json files.
    const tasks = [
        definitions_1.genVizDef(_path, outputPath),
        definitions_1.genManifest(_path, outputPath),
        definitions_1.genMetaInfo(_path, outputPath)
    ];
    // Copy all files from ./static, ./nls and file icon.svg to the output folder.
    for (let i = 0; i < constants_1.STATIC_FILES.length; ++i) {
        const staticPath = constants_1.STATIC_FILES[i]; // e.g. './static', './nls' or 'icon.svg'
        const srcPath = path.join(_path, staticPath);
        const destPath = path.join(outputPath, staticPath);
        if (await fs.exists(srcPath))
            tasks.push(fs.copy(srcPath, destPath));
    }
    // Wait until all operations have completed and return.
    await Promise.all(tasks);
}
/**
 * Creates a function that copies a given file from a source folder to a destination
 * folder. The returned function is an async file copy function.
 * @param _sourcePath Source path to copy the file from.
 * @param _targetPath Destination path to copy the file to.
 * @returns A function that accepts a file name and returns a completion promise.
 */
function createCopyFile(_sourcePath, _targetPath) {
    return async (_fileName) => {
        const source = path.join(_sourcePath, _fileName);
        const target = path.join(_targetPath, _fileName);
        if (!(await fs.exists(source)))
            return;
        await fs.copy(source, target);
    };
}
/**
 * Creates a function that removes a given file from a folder. The returned function
 * is an async file remove function.
 * @param _targetPath Path from which the file should be removed.
 * @returns A function that accepts a file name and returns a completion promise.
 */
function createRemoveFile(_targetPath) {
    return async (_filename) => {
        await fs.remove(path.join(_targetPath, _filename));
    };
}
async function withError(_promise, _ev) {
    try {
        return await _promise;
    }
    catch (_err) {
        return events_1.emitError(_ev, _err.message || _err);
    }
}
/**
 * Creates a bundle from the specified path.
 * @param _path Path in which the bundle resides.
 * @param _transpilerOverrides Additional options that can be used to override the
 * default rollup config.
 * @returns A promise that resolves when the bundle is created.
 */
async function createBundle(_path, _transpilerOverrides) {
    events_1.emitEvent(constants.BUILDER_EVENTS.START_BUILDING);
    // Prepare by copying vizdef, manifest, meta and static files.
    await prepareBundle(_path);
    // Transpile typescript / javascript code.
    await transpile.transpile(_path, _transpilerOverrides);
    events_1.emitEvent(constants.BUILDER_EVENTS.DONE_BUILDING);
}
exports.createBundle = createBundle;
/**
 * Prepare and start watching a bundle, its configs and static files
 * @param _path Root path of the bundle.
 * @param _transpilerOverrides Additional options that can be used to override the
 * default rollup config.
 * @returns A promise that resolves with a function that can be called to stop the
 * watching process.
 */
async function watchBundle(_path, _transpilerOverrides) {
    events_1.emitEvent(constants.BUILDER_EVENTS.START_WATCHING);
    // Prepare by copying vizdef, manifest, meta and static files.
    await prepareBundle(_path);
    // Setup file watch options.
    const watchOptions = {
        ignoreInitial: true,
        interval: 300,
        cwd: _path,
    };
    const outputPath = path.join(_path, constants_1.OUTPUT_DIR);
    // Create functions to copy, remove or generate output files.
    const copyFn = createCopyFile(_path, outputPath);
    const removeFn = createRemoveFile(outputPath);
    const genVizdefFn = () => withError(definitions_1.genVizDef(_path, outputPath), 3 /* FAILED_GEN_VIZDEF */);
    const genManifestFn = () => withError(definitions_1.genManifest(_path, outputPath), 1 /* FAILED_GEN_MANIFEST */);
    const genMetaFn = () => withError(definitions_1.genMetaInfo(_path, outputPath), 2 /* FAILED_GEN_META */);
    const watchers = [
        // Watch changes to vizdef.xml.
        fs.watchFiles("vizdef.xml", watchOptions)
            .on("change", genVizdefFn),
        // Watch changes to manifest.xml.
        fs.watchFiles("manifest.xml", watchOptions)
            .on("add", genManifestFn)
            .on("change", genManifestFn)
            .on("unlink", genManifestFn),
        // Watch changes to package.json to generate new meta info.
        fs.watchFiles("package.json", watchOptions)
            .on("change", genMetaFn),
        // Watch static files being added or removed and sync them.
        fs.watchFiles(constants_1.STATIC_FILES, watchOptions)
            .on("add", copyFn)
            .on("change", copyFn)
            .on("unlink", removeFn),
    ];
    // Transpiler.
    const transpileWatcher = await transpile.watch(_path, _transpilerOverrides);
    watchers.push(transpileWatcher);
    return () => {
        events_1.emitEvent(constants.BUILDER_EVENTS.STOPPED_WATCHING);
        watchers.forEach(_w => _w.close());
    };
}
exports.watchBundle = watchBundle;
