"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs_1 = require("../utils/fs");
const transpile_1 = require("./transpile");
const common_1 = require("../common");
const bundle_1 = require("./bundle");
const errors_1 = require("../errors");
const constants_1 = require("../constants");
/**
 * Setup the dev environment for a bundle. This means copying the rollup config,
 * adding dependencies to package.json and calling npm install.
 * @param _path Root path of the bundle.
 * @returns A promise that resolves when the dev environment has been setup.
 */
async function setupDevEnvironment(_path) {
    if (!(await fs_1.isDevBundle(_path)))
        throw new Error(errors_1.INVALID_DEV_VIS);
    // Check if rollup.config.js exists in the bundle. If not generate it.
    if (!(await fs_1.exists(path.join(_path, constants_1.ROLLUP_CONFIG))))
        await transpile_1.generateRollupTemplate(_path);
    // Add necessary dependencies to package.json.
    const packageJsonChanged = await transpile_1.addDevDependenciesToPackage(_path);
    // Run install if necessary.
    const nodeModulesExists = await fs_1.exists(path.join(_path, "node_modules"));
    if (packageJsonChanged || !nodeModulesExists)
        await common_1.installDependencies(_path);
}
/**
 * Prepare and build a bundle at the `_path`
 * @param _path
 * @param _rollupOverrides additional rollup options which can be used to override the default config
 */
async function build(_path, _rollupOverrides = {}) {
    await setupDevEnvironment(_path);
    return bundle_1.createBundle(_path, _rollupOverrides);
}
exports.build = build;
/**
 * Prepare and start watching a bundle at the `_path`
 * @param _path
 * @param _rollupOverrides additional rollup options which can be used to override the default config
 */
async function watch(_path, _rollupOverrides = {}) {
    await setupDevEnvironment(_path);
    return bundle_1.watchBundle(_path, _rollupOverrides);
}
exports.watch = watch;
