"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const bundler = require("..");
const notify_1 = require("./notify");
const util_1 = require("util");
/**
 * Prints one or more error messages to the console.
 * @param args One or more error messages.
 */
function printError(...args) {
    notify_1.notify("[Error]", ...args);
}
exports.printError = printError;
/**
 * Prints one or more info messages to the console.
 * @param args One or more info messages.
 */
function printInfo(...args) {
    notify_1.notify("[Info]", ...args);
}
exports.printInfo = printInfo;
// Events that should be printed to CLI
const eventToMsg = {
    [bundler.BUILDER_EVENTS.GENERATING_MANIFEST]: "Generating manifest.xml",
    [bundler.BUILDER_EVENTS.GENERATING_VIZDEF]: "Generating vizdef.xml",
    [bundler.BUILDER_EVENTS.START_PACKING]: "Start packing...",
    [bundler.BUILDER_EVENTS.START_ZIPPING]: "Zipping...",
    [bundler.BUILDER_EVENTS.DONE_PACKING]: "Done packing %s",
    [bundler.BUILDER_EVENTS.START_TRANSPILING]: "Started building the sources...",
    [bundler.BUILDER_EVENTS.DONE_TRANSPILING]: "Finished building the sources in %s ms",
    [bundler.BUILDER_EVENTS.INSTALLING_DEPENDENCIES]: "------- Installing Dependencies ----------",
    [bundler.BUILDER_EVENTS.CREATING_NEW_PROJECT]: ".........Generating Project.........",
    [bundler.BUILDER_EVENTS.SERVER_STARTING]: "Starting a local server on port %s",
    [bundler.BUILDER_EVENTS.SERVER_STARTED]: "Started a local server on port %s",
    [bundler.BUILDER_EVENTS.SERVER_STOPPED]: "Stopped the local server",
    [bundler.BUILDER_EVENTS.DONE_CREATING_PROJECT]: ".........Done Generating Project.........\nTo start developing execute the following commands:\n  cd %s\n  customvis start",
};
const errToMsg = {
    [1 /* FAILED_GEN_MANIFEST */]: "%s",
    [2 /* FAILED_GEN_META */]: "%s",
    [3 /* FAILED_GEN_VIZDEF */]: "%s",
    [0 /* FAILED_TRANSPILING */]: "%s",
    [5 /* SERVER_FAILED_TO_START */]: "Failed to start the server\n%s",
    [4 /* SERVER_PORT_IN_USE */]: "Port %s is already in use. Please terminate the process which is using port %s before calling \"start\".",
};
/**
 * Register CLI logging to log bundler events
 * Note this should be done once during lifecycle of the CLI
 */
function registerEvents() {
    const eventEmitter = bundler.getEventEmitter();
    // listen for bundler events
    eventEmitter.on(bundler.BUILDER_EVENT, function (...args) {
        const event = args[0];
        if (eventToMsg[event])
            printInfo(util_1.format(eventToMsg[event], ...args.slice(1)));
    });
    // listen for bundler errors
    eventEmitter.on(bundler.BUILDER_ERROR, function (...args) {
        const event = args[0];
        if (errToMsg[event])
            printError(util_1.format(errToMsg[event], ...args.slice(1)));
    });
}
exports.registerEvents = registerEvents;
