"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("./utils/fs");
const events_1 = require("./utils/events");
const templates = require("./utils/templates");
const constants_1 = require("./constants");
const errors_1 = require("./errors");
/**
 * Updates the package.json found in the specified folder to contain the name of the
 * folder it was generated in.
 * @param _path Full path of a bundle holding the package.json to be updated.
 * @returns A promise that resolves if the package.json has been updated.
 */
async function updateConfig(_path) {
    const postfix = path.basename(_path);
    const pkgPath = path.join(_path, "package.json");
    const config = await fs.readJSON(pkgPath);
    // Update the meta contents of package.json and write back to package.json.
    config.meta.id = `com.ibm.vis.custom.${postfix.toLowerCase()}`;
    config.meta.name = postfix;
    await fs.writeJSON(pkgPath, config);
}
/**
 * Creates a new bundle from a template
 * @param _path the current path where the new bundle should be created in
 * @param _name the name of the new bundle, the name will be used for folder creation
 * @param _options additional options which can be used
 */
async function create(_path, _name, _options = {}) {
    // Make sure create does not overwrite an already existing folder.
    const destination = path.join(_path, _name);
    if (await fs.exists(destination) && !_options.force)
        throw new Error(errors_1.createInvalidFolderError(destination));
    events_1.emitEvent(constants_1.BUILDER_EVENTS.CREATING_NEW_PROJECT);
    // Determine the template to use.
    const templatePath = templates.getTemplatePath(_options.template);
    if (!(await fs.exists(templatePath)))
        throw new Error(errors_1.createInvalidTemplateError(_options.template));
    // Copy template files and update the copied package.json.
    await fs.copy(templatePath, destination);
    await updateConfig(destination);
    events_1.emitEvent(constants_1.BUILDER_EVENTS.DONE_CREATING_PROJECT, destination);
    return destination;
}
exports.create = create;
