"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("./utils/fs");
const path = require("path");
const constants_1 = require("./constants");
/**
 * Get info about the the current bundle.
 * @param _path The absolute path where the bundle is located.
 * @returns A promise that resolves with a string holding the (formatted) information.
 */
async function info(_path) {
    const cliVersion = (await fs.readJSON(path.resolve(__dirname, "../package.json"))).version;
    const visPkgPath = path.join(_path, "package.json");
    let libVersion = "";
    let rsVersion = "";
    let metaInfo = null;
    // Get metaInfo, rsVersion and libVersion from bundle and customvis-lib.
    const isDev = await fs.exists(visPkgPath);
    if (isDev) {
        const visPkg = await fs.readJSON(visPkgPath);
        metaInfo = visPkg.meta;
        // Check if lib package is installed (in node_modules).
        const libPkgPath = path.join(_path, constants_1.LIB_PACKAGE_PATH, "package.json");
        const hasLib = await fs.exists(libPkgPath);
        if (hasLib) {
            const libPkg = await fs.readJSON(libPkgPath);
            rsVersion = libPkg.apiversion;
            libVersion = libPkg.version;
        }
    }
    else if (await fs.exists(path.join(_path, "meta.json"))) {
        metaInfo = (await fs.readJSON(path.join(_path, "meta.json")));
        rsVersion = metaInfo.apiVersion;
        libVersion = metaInfo.libVersion;
    }
    // At least show the customvis CLI version.
    const info = `Customvis CLI
    version: ${cliVersion}`;
    // If we found any meta info, fill in the rest of the Visualization info.
    return metaInfo ? `${info}

    Visualization
        id: ${metaInfo.id}
        name: ${metaInfo.name},
        description: ${metaInfo.description}
        icon: ${metaInfo.icon}
        version: ${metaInfo.version}

        customvis-lib: ${libVersion || "not found"}
        rs-version: ${rsVersion || "not found"}` : info;
}
exports.info = info;
