"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("./utils/fs");
const build = require("./build");
const server = require("./server");
const path = require("path");
const errors_1 = require("./errors");
/**
 * Start transpiling, watchers and serve the files.
 * @param _path the absolute path where the bundle is located
 * @param _options additional options which can be passed
 */
async function start(_path, _options = {}) {
    // If there is no vizdef.xml, then it is probably not a customvis bundle.
    const vizdefPath = path.join(_path, "vizdef.xml");
    if (!(await fs.exists(vizdefPath)))
        throw new Error(errors_1.NO_VIZDEF_ERROR);
    // Keep track of the closeWatchers function so we can perform cleanup when the
    // server is stopped. Notice that we call `builder.watch` only if the bundle
    // is a 'dev' bundle (i.e. one with a package.json that can be built from the
    // sources).
    let closeWatchers;
    if (await fs.isDevBundle(_path))
        closeWatchers = await build.watch(_path);
    // Start the server; perform cleanup when closed.
    return server.start(_path, _options).on("close", () => {
        if (closeWatchers)
            closeWatchers();
        process.exit(0);
    });
}
exports.start = start;
