"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const constants_1 = require("./constants");
const errors_1 = require("./errors");
// packages with side effects
const templates_1 = require("./utils/templates");
const common_1 = require("./common");
const clean_1 = require("./clean");
const fs = require("./utils/fs");
const events_1 = require("./utils/events");
/**
 * Upgrade the customvis-lib of the bundle. Effectively this performs a 'clean'
 * followed by a copy of a new customvis-lib and a dependencies install.
 * @param _path The absolute path where the bundle is located.
 */
async function upgrade(_path) {
    events_1.emitEvent(constants_1.BUILDER_EVENTS.START_UPGRADING_LIB);
    // If there is no vizdef.xml or package.json, it is not a bundle.
    const vizDefPath = path.join(_path, "vizdef.xml");
    const packageJson = path.join(_path, "package.json");
    if (!(await fs.exists(vizDefPath)))
        throw new Error(errors_1.UPGRADE_NO_VIZDEF);
    if (!(await fs.exists(packageJson)))
        throw new Error(errors_1.UPGRADE_PACKED_ERROR);
    // Locate the package.json of customvis-lib (located in node_modules).
    const libPackagePath = path.join(_path, constants_1.LIB_PACKAGE_PATH, "package.json");
    // Catch since you can run upgrade without having installed dependencies.
    const oldLibInfo = (await fs.readJSON(libPackagePath).catch(() => { })) || {};
    // Clean up the bundle.
    await clean_1.clean(_path);
    // The sdk's templates each hold a copy of customvis-lib.tgz. So we can use the
    // location of the default template to determine the full path of customvis-lib.
    const libPath = path.join(templates_1.getTemplatePath(), constants_1.LIB_PACKGE_FILE_IN_VIS);
    // Copy the sdk's version of customvis-lib to the current bundle location.
    await fs.copy(libPath, path.join(_path, constants_1.LIB_PACKGE_FILE_IN_VIS));
    // Re-install all dependencies.
    await common_1.installDependencies(_path).catch((_err) => {
        throw new Error(errors_1.createUpgradeInstallDepError(_err.message));
    });
    // Get version information from the sdk's package.json.
    const newLibInfo = await fs.readJSON(libPackagePath);
    events_1.emitEvent(constants_1.BUILDER_EVENTS.DONE_UPGRADING_LIB);
    return {
        currentVersion: newLibInfo.version,
        oldVersion: oldLibInfo.version,
        path: _path
    };
}
exports.upgrade = upgrade;
