"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const parseString = require("xml2js").parseString;
/**
 * Checks if an XML string has valid XML syntax.
 * @param _xmlString A string containing XML content.
 * @returns A promise that resolves with true if the XML content is valid. If the
 * XML content is not valid, the promise is resolved with false.
 */
function isValidXML(_xmlString) {
    return new Promise(_resolve => {
        parseString(_xmlString, (_err, _result) => {
            if (_err)
                return _resolve(false);
            if (_result)
                return _resolve(true);
            return _resolve(false);
        });
    });
}
exports.isValidXML = isValidXML;
/**
 * Converts an XML string to a JSON object.
 * @param _xmlString A string containing XML content.
 * @returns A promise that resolves with an object representing the XML. If the
 * XML content is invalid, the promise is rejected.
 */
function getJSONFromXML(_xmlString) {
    return new Promise((_resolve, _reject) => {
        parseString(_xmlString, (_err, _result) => {
            if (_err)
                return _reject(_err);
            if (_result)
                return _resolve(_result);
            return _reject();
        });
    });
}
exports.getJSONFromXML = getJSONFromXML;
/**
 * Escape a given XML string so it can be used in a JavaScript return statement.
 * Used internally by the `asModule` function.
 * @param _content A string containing XML content.
 * @returns An encoded string.
 */
function escape(_content) {
    return _content.replace(/(['\\])/g, "\\$1")
        .replace(/[\f]/g, "\\f")
        .replace(/[\b]/g, "\\b")
        .replace(/[\n]/g, "\\n")
        .replace(/[\t]/g, "\\t")
        .replace(/[\r]/g, "\\r")
        .replace(/[\u2028]/g, "\\u2028")
        .replace(/[\u2029]/g, "\\u2029");
}
/**
 * Converts an XML string to a string that represents a JavaScript module that returns
 * that XML string. Typically this function is used when creating manifest.xml.js or
 * vizdef.xml.js.
 * @param _content A string containing XML content.
 * @returns A string representing a JavaScript module that returns the XML content.
 */
function asModule(_content) {
    return "define( [], function () { return '" + escape(_content) + "';});\n";
}
exports.asModule = asModule;
