import IRSTuple from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSTuple";
import IRSDataPoint from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSDataPoint";
import IRSSlot from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSSlot";
import IRSStructuredDataSet from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSStructuredDataSet";
import IRSItem from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSItem";
import IRSItemClass from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRSItemClass";
/**
 * Represents a value domain with a minimum and maximum value. Used in a continuous
 * slot to define the value range of values in that slot. See also [[Slot.domain]].
 */
export declare class Domain {
    /**
     * Represents an empty domain.
     */
    static empty: Domain;
    /**
     * The minimum value of the domain.
     */
    readonly min: number;
    /**
     * The maximum value of the domain.
     */
    readonly max: number;
    /**
     * @ignore
     */
    protected constructor(_min: number, _max: number);
    /**
     * Returns the domain as an array of two elements, first min then max.
     *
     * ```typescript
     * const domain = _info.data.cols[ 1 ].domain.toArray();
     * const scale = d3.scaleLinear().domain( domain ).range( [ 0, 100 ] );
     * ```
     * @returns The domain as an array of [min, max].
     */
    asArray(): number[];
}
interface IDecorated {
    readonly selected: boolean;
    readonly highlighted: boolean;
}
/**
 * A tuple is a categorical data element found in a (categorical) slot. You can retrieve
 * the list of tuples on a slot through `slot.tuples`. If your data has rows, then each
 * row can have a reference to one of the tuples in a slot.
 *
 * ```typescript
 * // Retrieve the list of tuples for slot 1.
 * const tuples: Tuple[] = _info.data.cols[ 1 ].tuples
 *
 * // Take the tuple in slot 1 from the first data row.
 * const tuple: Tuple = _info.data.rows[ 0 ].tuple( 1 );
 *
 * // Retrieve the captions for each tuple.
 * const captions: string[] = tuples.map( _tuple => _tuple.caption );
 * ```
 */
export declare class Tuple implements IDecorated {
    /**
     * The underlying VIPR source object.
     * @ignore
     */
    readonly source: IRSTuple;
    /**
     * The zero-based index of the tuple in the slot.
     */
    readonly index: number;
    /**
     * A list of one or more segments that form the tuple. Each segment has a caption
     * and a key. The caption of a tuple is a concatenation of the captions of all
     * segments, with a predefined separator.
     */
    readonly segments: Segment[];
    /**
     * A unique key value for this tuple. You can use the key when you perform d3 data
     * binding.
     *
     * ```typescript
     * // Create a d3 text element for each tuple in column #1
     * const tuples = _info.data.cols[ 1 ];
     * node.selectAll( "text" )
     *     .data( tuples, ( _t: Tuple ) => _t.key )
     *     .join( "text" )
     *         .text( _tuple => _tuple.caption );
     * ```
     */
    readonly key: string;
    /**
     * The caption of this tuple.
     */
    readonly caption: string;
    /**
     * @ignore
     */
    protected constructor(_source: IRSTuple, _index: number);
    /**
     * Returns true if the tuple is selected. Note that the visual selection state of
     * a tuple might depend on the selected state of other data elements.
     */
    get selected(): boolean;
    /**
     * Returns true if the tuple is highlighted.
     */
    get highlighted(): boolean;
}
/**
 * A Segment gives access to the data elements that form a tuple or a slot.
 * A data provider is able to provide 'stacked' data, where each tuple or
 * slot is mapped to multiple segments. These segments can be accessed through
 * the 'segments' property on a tuple or slot.
 * In many cases a visualization is only interested in the contents of the
 * entire tuple, but in some situations the visualizations needs to display
 * the individual tuple segment. Examples are the rendering of a hierachical
 * view of the associated data item, or providing an alternative representation
 * of the tuple caption.
 *
 * ```typescript
 * // Returns a caption for the given tuple.
 * function getCaption( _tuple )
 * {
 *     // Use ', ' as a separator, instead of the default ' | '.
 *     return _tuple.segments.map( _segment => _segment.caption ).join( ", " );
 * }
 * ```
 */
export declare class Segment implements IDecorated {
    /**
     * The underlying VIPR source object.
     * @ignore
     */
    readonly source: IRSItem | IRSItemClass;
    /**
     * A unique key value for the segment. You can use this key when you perform
     * data binding.
     */
    readonly key: string;
    /**
     * The caption of the segment.
     */
    readonly caption: string;
    /**
     * @ignore
     */
    protected constructor(_source: IRSItem | IRSItemClass);
    /**
     * Returns true if the segment is selected. Note that the visual selection state of
     * a segment might depend on the selected state of other data elements.
     */
    get selected(): boolean;
    /**
     * Returns true if the segment is highlighted.
     */
    get highlighted(): boolean;
}
/**
 * Format type, used in the [[Slot]].[[format]] function. A value can be formatted
 * for use as a 'label' or as 'data'. A host application is allowed to provide
 * different formatting rules for both types. For instance, the value in a tooltip
 * might have different formatting applied that same value on an axis.
 */
export declare enum FormatType {
    /**
     * Format the value for use as a label (e.g. on an axis).
     *
     * ```typescript
     * const slot = _info.data.slotMap.get( "values" );
     * const axis = d3.axisLeft( scale ).tickFormat( v =>
     * {
     *     // Format the axis value using the formatter of the slot.
     *     return slot.format( v.valueOf(), FormatType.label );
     * } );
     * ```
     */
    label = "label",
    /**
     * Format the value for use as a data item (e.g. in a tooltip).
     */
    data = "data"
}
/**
 * A slot is the entry point for the data that goes into a visualization. A slot can
 * hold either categorical or continuous items. If the slot has categorical items, you
 * can retrieve a distinct list using the `tuples` attributes. If the slot has continuous
 * items, you can retrieve the domain (min and max) of the item values.
 */
export declare class Slot {
    /**
     * The underlying VIPR source object.
     * @ignore
     */
    readonly source: IRSSlot;
    /**
     * Returns the list of tuples in this slot. Can be an empty list if there are no
     * tuples or if the slot represents continuous data items.
     */
    readonly tuples: Tuple[];
    /**
     * A list of one or more segments that form the data item on the slot. Each segment has
     * a caption and a key. The caption of a slot is a concatenation of the captions
     * of all segments, with a predefined separator.
     */
    readonly segments: Segment[];
    /**
     * Returns the value domain for this slot. Can be an empty domain (`Domain.empty`)
     * if the slot has no values or if the slot represents a categorical data item.
     */
    readonly domain: Domain;
    /**
     * The caption of the slot, taken from the data item that was mapped to this slot.
     * Note: this caption is different from the slot caption defined in the vizdef!
     */
    readonly caption: string;
    private readonly _labelFormatter;
    private readonly _dataFormatter;
    /**
     * @ignore
     */
    protected constructor(_source: IRSSlot, _tuples: Tuple[], _segments: Segment[], _domain: Domain, _caption: string);
    /**
     * Indicates if a slot is mapped. A slot is considered mapped if data has been
     * assigned to the slot. If a slot is not mapped, then [[tuples]] is `[]`, the
     * [[domain]] is [[Domain.empty]] and the [[caption]] is `""`.
     */
    get mapped(): boolean;
    /**
     * Returns the type of data that is mapped to this slot. Possible values are "cat",
     * "cont" or "none". A slot that is not mapped returns "none".
     */
    get dataType(): string;
    /**
     * Formats a value.
     * @param _value Value to format.
     * @param _type Optional type (label or data). Defaults to label if not specified.
     * @returns The formatted value. If the slot is categorical or the slot is
     * not mapped, then the 'toString' representation of the value is returned.
     */
    format(_value: number | null, _type?: FormatType): string;
}
/**
 * DataPoint objects appear as rows in a data set. Each DataPoint has a reference to a
 * tuple or value in each of the slots of the data set.
 */
export declare class DataPoint implements IDecorated {
    /**
     * The underlying VIPR source object.
     * @ignore
     */
    readonly source: IRSDataPoint;
    /**
     * A unique key value for the data point.
     */
    readonly key: string;
    /**
     * Reference to the data set holding this data point.
     */
    readonly dataSet: DataSet;
    /**
     * @ignore
     */
    protected constructor(_source: IRSDataPoint, _dataSet: DataSet);
    /**
     * Returns true if the data point is selected. Note that the visual selection state
     * of a data point might depend on the selected state of other data elements.
     */
    get selected(): boolean;
    /**
     * Returns true if the data point is highlighted.
     */
    get highlighted(): boolean;
    /**
     * Returns the tuple at the specified column. The column can be specified as
     * an index or a slot name.
     * @param _col Column index or (slot) name.
     * @returns The tuple on the specified column, or null if the column is not mapped.
     */
    tuple(_col: number | string): Tuple | null;
    /**
     * Returns the value at the specified column. The column can be specified as
     * an index or a slot name.
     * @param _col Column index or (slot) name.
     * @returns The value on the specified column, or null if the column is not mapped.
     */
    value(_col: number | string): number | null;
    /**
     * Returns the caption at the specified column. The column can be specified as
     * an index or a slot name.
     * @param _col Column index or (slot) name.
     * @returns The tuple caption or formatted value on the specified column, or an empty
     * string if the column is not mapped or the value is missing.
     */
    caption(_col: number | string): string;
    private _getSlot;
}
/**
 * A DataSet is the root class of data that goes into the visualization. A data set
 * consists of columns and rows. The columns correspond with the slots in the vizdef
 * and the rows correspond with the actual data points (the result rows of a query).
 */
export declare class DataSet {
    /**
     * The underlying VIPR source object.
     * @ignore
     */
    readonly source: IRSStructuredDataSet;
    /**
     * Data rows. Each row corresponds with a data point in the visualization.
     */
    readonly rows: DataPoint[];
    /**
     * Columns of the data set. Each column corresponds with a slot in the vizdef.
     */
    readonly cols: Slot[];
    /**
     * Maps from name to slot. Can be used as an alternative to the [[cols]] array
     * to retrieve a slot based on a name.
     */
    readonly slotMap: Map<string, Slot>;
    /**
     * Filters an array of rows based on a column and a key.
     * @param _rows The rows to filter
     * @param _col The column number to filter on. Must be a categorical column.
     * @param _key The filter key.
     * @returns A subset of the rows.
     */
    static filterRows(_rows: DataPoint[], _col: number, _key: string): DataPoint[];
    /**
     * @ignore
     */
    protected constructor(_source: any, _cols: Slot[], _slotmap: Map<string, Slot>);
    /**
     * If any of the data points in the data is selected, then this flag is set
     * to true. If there are no selected data points, this flag is false.
     */
    get hasSelections(): boolean;
}
/**
 * Creates data for RenderBase. Only called internally in RenderBase and therefore
 * ignored for documentation.
 * @param _ds0 An IRSStructuredDataSet instance.
 * @ignore
 */
export declare function createData(_ds0: any): DataSet;
export {};
