import IRSProperties from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSProperties";
import { CatPalette, ContPalette } from "./Palettes";
import { SlotResolver } from "./SlotResolver";
/**
 * Represents the collection of properties of a visualization.
 * A [[get]] function is available to retrieve the value of a property by name.
 * The dirty and active status of a property can be determined by the [[isDirty]]
 * and [[isActive]] functions repectively.
 * The properties collection can be accessed through [[RenderBase.properties]] or
 * through [[UpdateInfo.props]].
 */
export declare class Properties {
    /**
     * Reference to the underlying VIPR properties interface. Can be null, in which
     * case the properties instance represents an empty property collection.
     * @ignore
     */
    readonly source: IRSProperties | null;
    /**
     * Map holding wrappers for palette properties. This map is filled at construction
     * time and holds a wrapper `Palette` object for each palette property. It is used
     * in the `get` function of this class and  internally by the [[Encoding]] class, but
     * may also be used externally (in a [[RenderBase]] subclass).
     */
    readonly palettes: ReadonlyMap<string, CatPalette | ContPalette>;
    /**
     * You should never have to call this constructor directly. The `RenderBase` class
     * creates an instance of `Properties` at initialization time. This instance will
     * be passed in the `UpdateInfo` of each `RenderBase.update` call.
     * @param _source The VIPR properties collection.
     * @param _dataContext Data context of the visualization instance.
     * @param _resolver A resolver interface that is used to resolve a slot for a
     * palette in case that palette is not explicitly linked to a slot.
     * @ignore
     */
    constructor(_source: IRSProperties | null, _dataContext: any, _resolver: SlotResolver);
    /**
     * Retrieves the value of a property. For numbers, strings, boolean and enum values
     * the native value type is returned. A [[Color]], [[Length]] and [[Font]] is returned as a
     * object which contains a toString method. A palette is returned as a [[CatPalette]] or a [[ContPalette]] object.
     * @param _name Name of the property.
     * @returns The value of the property or null if the property does not exist. The
     * return value can be a string, number, boolean, [[Color]], [[Length]], [[Font]], [[CatPalette]], [[ContPalette]]
     * or null.
     */
    get(_name: string): any;
    /**
     * Retrieves the value of a property. The difference between peek and [[get]] is
     * that [[get]] retrieves the locked value of a property, while peek circumvents
     * the locking mechanism and returns the value directly from the host application.
     *
     * **In general you should use [[get]] to retrieve the value of a property since
     * locking ensures that the value will not change during asynchronous render
     * operations.**
     *
     * In some situations however you want to circumvent the locking mechanism. For
     * instance if in the [[updateProperty]] method you want to set the active state
     * of one property based on the value of another property, then you need to get
     * the 'peek' value of that other property.
     *
     * **Note** that 'peek' can not be used for palette properties.
     * @param _name Name of the property.
     * @returns The value of the property or null if the property does not exist. The
     * return value can be a string, number, boolean, [[Color]], [[Length]], [[Font]], [[CatPalette]], [[ContPalette]]
     * or null.
     */
    peek(_name: string): any;
    /**
     * Helper method that retrieves the value of a property. Used by 'get' and 'peek'.
     * @param _name Name of the property
     * @param _peek Flag that indicates if 'peekValue' or 'getValue' should be used
     * to retrieve the value. Note that 'peekValue' does not have effect on palette
     * properties.
     */
    private _getValue;
    /**
     * @param _name Name of the property.
     * @returns True if the property is active, false otherwise.
     */
    isActive(_name: string): boolean;
    /**
     * @param _name Name of the property.
     * @param _active Active state.
     * Marks a property as active or inactive.
     */
    setActive(_name: string, _active: boolean): void;
    /**
     * The dirty state of a property tells you if the value of the property has
     * changed since the last [[RenderBase.update]] call.
     * @param _name Name of the property.
     * @returns True if the property is dirty, false otherwise.
     */
    isDirty(_name: string): boolean;
}
