/// <reference types="promise-polyfill" />
import IRenderingService from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRenderingService";
import IRSInitParameters from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSInitParameters";
import IRSRenderInterrupt from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRenderInterrupt";
import IRSRenderInvalidation from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRenderInvalidation";
import IRSInitNotification from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSInitNotification";
import IRSEncoding from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSEncoding";
import IRSCapability from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSCapability";
import IRSPoint from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSPoint";
import IRSRegion from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRegion";
import IRSHittable from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSHittable";
import IRSAxisItem from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSAxisItem";
import IRenderingServiceData from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/data/IRenderingServiceData";
import IRSRequest from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRequest";
import IRSRenderParameters from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRenderParameters";
import IRSRenderNotification from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSRenderNotification";
import IRSInteractivity from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSInteractivity";
import { UpdateInfo } from "./UpdateInfo";
import { Point } from "./Point";
import { DataSet, DataPoint, Tuple, Segment } from "./Data";
import { Encoding } from "./Encodings";
import { Properties } from "./Properties";
import { MetaInfo } from "./MetaInfo";
/**
 * RenderBase serves as the base class for all custom visualizations. It provides a
 * custom visualization with meta information, properties, nls support and a number
 * of methods that can be overridden to implement rendering and hit testing.
 * A minimal custom visualization would look like this:
 *
 * ```typescript
 * import { RenderBase } from "@businessanalytics/customvis-lib"
 *
 * class MyCustomVis extends RenderBase
 * {
 *     create( _node: HTMLElement ): void
 *     {
 *         _node.textContent = "Hello Custom Visualization";
 *     }
 * }
 * ```
 */
export declare abstract class RenderBase implements IRenderingService {
    /**
     * Reference to the VIPR data that is set by the host. The data is not exposed directly
     * to RenderBase subclasses. Instead, a wrapper object is created each time 'update' is
     * called. The wrapper object holds a simplified data model.
     */
    private _data;
    /**
     * The HTML element that represents the renderer. Initially null, but after `init`
     * has been called `_elem` has a value.
     */
    private _elem;
    /**
     * NLS object that can be used by renderers to retrieve localized strings. Subclasses
     * can call the `nls( _id )` function to retrieve a localized string.
     */
    private _nls;
    /**
     * The data locale that was used to initialize the rendering service. This value is
     * passed to every `update` call so subclasses can use it if they need to implement
     * their own data formatting.
     */
    private _locale;
    /**
     * Resolver function that is passed to createEncodings and the properties collection.
     * This function is used to resolve a slot given a palette name and is bound to the
     * `getSlotForPalette` member function in the constructor of RenderBase.
     */
    private readonly _slotResolver;
    /**
     * The properties of the rendering service. Created at initialization time, this object
     * will not change during the lifetime of the instance. Since VIPR handles locking of
     * properties during rendering, we can safely pass this instance as part of [[UpdateInfo]].
     *
     * Properties are 'protected' to allow subclasses to access them in methods that don't
     * have an [[UpdateInfo]]. An example is [[getEncodings]], where the renderer may need to
     * return legend encodings based on certain property values. The general rule however
     * is to access properties through [[UpdateInfo.props]].
     *
     * ```typescript
     * // Retrieve the background color of the visualization.
     * const bgColor = _info.props.get( "color.background" );
     * ```
     */
    protected properties: Properties;
    /**
     * Object holding various meta information that controls how the visualization behaves.
     * For instance, an attribute can be set that controls the default shape that is used
     * for encoding categorical legend items. A good place to make changes to this meta
     * information is in the `create` method of the visualization.
     *
     * ```typescript
     * // Change the shape of all categorical legend items to 'rectangle'.
     * this.meta.legendShape = "rectangle";
     *
     * // Limit the 'categories' slot to 250 tuples.
     * this.meta.slotLimits.set( "categories", 250 );
     * ```
     */
    protected readonly meta: MetaInfo;
    /**
     * @ignore
     */
    constructor();
    /**
     * @ignore
     */
    init(_params: IRSInitParameters, _initNotify: IRSInitNotification): void;
    /**
     * @ignore
     */
    destroy(): void;
    /**
     * @ignore
     */
    getPropertyApi(): any;
    /**
     * @ignore
     */
    setData(_data: IRenderingServiceData | null): void;
    /**
     * @ignore
     */
    setProperty(_name: string, _value: any): void;
    /**
     * @ignore
     */
    getBlockingRequests(): IRSRequest[] | null;
    /**
     * @ignore
     */
    render(_params: IRSRenderParameters, _reason: IRSRenderInvalidation, _notify: IRSRenderNotification): IRSRenderInterrupt | null;
    /**
     * @ignore
     */
    getEncodings(): IRSEncoding[];
    /**
     * @ignore
     */
    getCapabilities(): IRSCapability[];
    /**
     * @ignore
     */
    getInteractivity(): IRSInteractivity | null;
    /**
     * @ignore
     */
    getVisCoordinate(_viewport: IRSPoint, _client: IRSPoint): Object;
    /**
     * @ignore
     */
    getRegionAtPoint(_viewPortPoint: IRSPoint, _radius: number): IRSRegion | null;
    /**
     * @ignore
     */
    getItemsAtPoint(_vizCoord: Object, _viewport: IRSPoint, _radius: number): IRSHittable[] | null;
    /**
     * @ignore
     */
    getItemsInPolygon(_vizcoords: Object[], _viewport: IRSPoint[]): IRSHittable[] | null;
    /**
     * @ignore
     */
    getAxisItemsAtPoint(_vizcoord: Object, _viewport: IRSPoint, _radius: number): IRSAxisItem[] | null;
    /**
     * @ignore
     */
    getState(): Object | null;
    /**
     * @ignore
     */
    setState(_state: Object): void;
    /*******************************************************************************************/
    /**
     * Load a stylesheet by appending a style tag to the head of the HTML document.
     * @param _url Absolute or relative url of the stylesheet. If the url is relative, then it
     * will be made absolute by calling [[toUrl]]. Relative means relative to the root of the
     * visualization.
     */
    loadCss(_url: string): void;
    /**
     * Returns an absolute Url based on a Url relative to the visualization. Can be used
     * to create absolute references to e.g. static resources.
     * @param _url Path of the content relative to the root of the visualization.
     * @returns An absolute url.
     *
     * ```typescript
     * const img = this.toUrl( "./static/image.png" );
     * ```
     */
    toUrl(_url: string): string;
    /**
     * @param _info Info which can be used to render the visualization. Contains the root
     * element in which rendering should take place, along with properties, locale info and
     * data. During (async) rendering this info will not change.
     * @returns Optionally a promise that resolves when rendering is complete. If no promise
     * is returned, rendering is considered to have completed synchronously.
     */
    protected update(_info: UpdateInfo): void | Promise<void>;
    /**
     * Called during initialization of the visualization. This is, after the constructor,
     * the very first method that is called on the visualization object.
     * @param _parent HTML element used as entry point for the visualization.
     * @returns An Element or a promise that resolves to an element. The element that is
     * returned here will be passed to the `update` function as part of `UpdateInfo`.
     * This element is considered the root element of the visualization.
     * A void return value is also allowed (or a Promise that resolves with void). In
     * that case, the root element of the visualization will be the _parent that was
     * passed to `create`.
     */
    protected create(_parent: HTMLElement): Element | void | Promise<Element | void>;
    /**
     * Called immediately if the host assigns a new value to a property. Also called at
     * initialization time for each property in the bundle. Subclasses of Renderbase can
     * override this method to process property changes. For instance, they can set the
     * active state of properties based on the value of another property.
     * @param _name Name of the property that has changed.
     * @param _value New value of the property.
     */
    protected updateProperty(_name: string, _value: any): void;
    /**
     * Called when the host application requests information to update the legend.
     * The default implementation creates the correct encodings needed for the host.
     * Override this method only when you need to make modifications to the automatically
     * created legend.
     * @param _data The active data set.
     * @returns A list of legend encodings.
     *
     * ```typescript
     * // Reverse the items in the categorical legend.
     * updateLegend( _data: DataSet ) : Encoding[]
     * {
     *     // Call base class implementation.
     *     encodings = super.updateLegend( _data );
     *
     *     // Retrieve the first, categorical, encoding.
     *     const encoding = encodings[ 0 ] as CatEncoding;
     *
     *     // Reverse the entries in the encoding.
     *     encoding.entries.reverse();
     * }
     * ```
     */
    protected updateLegend(_data: DataSet): Encoding[];
    /**
     * Called for every palette that is not linked to a slot. Subclasses can override this
     * method to dynamically link a palette to a slot.
     * @param _data Actual data.
     * @param _slot Name of a palette with no linked slot.
     * @returns The name of a slot that is supposed to be linked to the palette, or null
     * if the palette remains unlinked.
     */
    protected getSlotForPalette(_data: DataSet, _palette: string): string | null;
    /**
     * Performs a hit test on a given position. The position is specified in both client
     * coordinates and viewport coordinates. Client coordinates are relative to the
     * upper left corner of the browser. Viewport coordinates are relative to the top
     * left of the visualization.
     * Subclasses can override this method to perform specific hit testing.
     * > **Note:** the default implementation checks if `_elem.__data__` represents
     * a [[DataPoint]], [[Tuple]] or [[Segment]]. If this is the case, then `_elem.__data__`
     * is returned. This provides automatic hit testing for d3 visualizations when the
     * node is bound to a [[DataPoint]], [[Tuple]] or [[Segment]].
     * @param _elem Element that was found on on the hit test position. Can be null in
     * case no element was found on the specified hit test position.
     * @param _client Client coordinate of the hit test position.
     * @param _viewport Viewport coordinate of the hit test position.
     * @returns An DataPoint, Tuple or Segment element from the data model, or null if
     * nothing was hit.
     */
    protected hitTest(_elem: Element | null, _client: Point, _viewport: Point): DataPoint | Tuple | Segment | null;
    /**
     * Returns a translated string given a key.
     * @param _key The key to lookup in the translation table.
     * @returns The translated string, or an empty string if the key is unknown.
     */
    protected nls(_key: string): string;
}
