import IRSCatPaletteProperty from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSCatPaletteProperty";
import IRSContPaletteProperty from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSContPaletteProperty";
import IRSContStops from "@waca/vipr-rs-v1/com/ibm/vipr/rs/v1/IRSContStops";
import { Tuple, Slot, DataPoint } from "./Data";
import { SlotResolver } from "./SlotResolver";
import { Color } from "./Color";
/**
* Representation of a categorical palette. Can be used to determine a color given
* a data tuple.
*/
export declare class CatPalette {
    private readonly _dataContext;
    private readonly _slotResolver;
    /**
     * The underlying IRSCatPaletteProperty as defined in VIPR-RS.
     * @ignore
     */
    readonly source: IRSCatPaletteProperty;
    /**
     * Creates a categorical palette. This constructor is only used internally during
     * construction of the `Properties` instance of the `RenderBase`.
     * @param _source The underlying categorical palette property.
     * @param _dataContext The data context of the renderer.
     * @param _slotResolver A resolver interface that is used to resolve a slot in
     * case the palette is not explicitly linked to one.
     * @ignore
     */
    constructor(_source: IRSCatPaletteProperty, _dataContext: any, _slotResolver: SlotResolver);
    /**
     * Returns the name of the slot this palette is linked to, or null if the palette
     * is not linked to a slot.
     */
    get slot(): string | null;
    /**
     * Determine a color given a tuple.
     * @param _tuple The tuple to determine the color for or null to get a default color.
     * @returns The color of the tuple.
     */
    getColor(_tuple: Tuple | null): Color;
    /**
     * Helper function that returns a tuple given a data point or tuple. If a data point
     * is given, the tuple is based on this.slot or the slot from the slot resolver.
     */
    private _getTuple;
    /**
     * Determine the fill color for a data point. The fill follows the 'selected'
     * state of a data point. If the data set has selected data points, then the
     * ones that are selected get the normal color as fill color and the ones that
     * are not selected get a 60% transparent color.
     *
     * ```typescript
     * // Set fill color of elements in a d3 selection.
     * selection.attr( "fill", _d => palette.getFillColor( _d ) )
     * ```
     * @param _e The data point or tuple to determine the color for. If `null` is
     * passed, the method behaves the same as [[getColor]].
     * @returns The color of the data point as a string.
     */
    getFillColor(_e: DataPoint | Tuple | null): string;
    /**
     * Determine the outline color for a data point. The outline color depends on
     * both the 'highlight' state and the 'selected' state. If the point is neither
     * highlighted or selected, `null` is returned to indicate that no outline needs
     * to be rendered. If the point is highlighted or part of a selection, then
     * the returned color is a darker version of the actual data point color.
     *
     * ```typescript
     * // Set outline color of elements in a d3 selection.
     * selection.attr( "stroke", _d => palette.getOutlineColor( _d ) )
     * ```
     * @param _e The data point or tuple to determine the color for. If `null` is
     * passed, the method behaves the same as [[getColor]].
     * @returns The color of the data point as a string or null if no outline is needed.
     */
    getOutlineColor(_e: DataPoint | Tuple | null): string | null;
}
/**
 * Represents a color stop with both an absolute and relative value.
 */
export declare class ColorStop {
    /**
     * The color of this color stop.
     */
    readonly color: Color;
    /**
     * The relative offset of the stop within its domain. The domain may depend
     * on the context in which a color stop is used. The relative offset is a value in
     * the range (0-1).
     */
    readonly at: number;
    /**
     * The absolute value of the stop.
     */
    readonly value: number;
    /**
     * @ignore
     */
    constructor(_color: Color, _at: number, _value: number);
}
/**
 * Color stops as received by a continuous palette.
 */
export declare class ColorStops {
    private readonly _dataContext;
    private readonly _slot;
    /**
     * The underlying IRSContStops as defined in VIPR-RS.
     * @ignore
     */
    readonly source: IRSContStops;
    /**
     * The list of color stops in this collection.
     */
    readonly stops: ColorStop[];
    /**
     * The list of aligned color stops in this collection. The aligned stops always
     * match the data domain.
     */
    readonly aligned: ColorStop[];
    /**
     * Flag to indicate whether stops should be interpolated.
     */
    readonly interpolate: boolean;
    /**
     * @ignore
     */
    constructor(_source: IRSContStops, _slot: string | null, _dataContext: any);
    /**
     * Retrieves a color given a value.
     * @param _value The value to get the color for.
     * @returns The color of the value.
     */
    getColor(_value: number): Color;
    /**
     * Determine the fill color for a data point. The fill follows the 'selected'
     * state of a data point. If the data set has selected data points, then the
     * ones that are selected get the normal color as fill color and the ones that
     * are not selected get a 60% transparent color.
     * @param _dp The data point to determine the color for.
     * @returns The color of the data point as a string.
     */
    getFillColor(_dp: DataPoint): string;
    /**
     * Determine the outline color for a data point. The outline color depends on
     * both the 'highlight' state and the 'selected' state. If the point is neither
     * highlighted or selected, `null` is returned to indicate that no outline needs
     * to be rendered. If the point is highlighted or part of a selection, then
     * the returned color is a darker version of the actual data point color.
     * @param _dp The data point to determine the color for.
     * @returns The color of the data point as a string or null if no outline is needed.
     */
    getOutlineColor(_dp: DataPoint): string | null;
}
/**
 * Representation of a continuous palette. Can be used to determine a color given
 * a data value.
 */
export declare class ContPalette {
    private readonly _dataContext;
    private readonly _slotResolver;
    private _lastDataItem;
    private _cachedStops;
    /**
     * The underlying IRSContPaletteProperty as defined in VIPR-RS.
     * @ignore
     */
    readonly source: IRSContPaletteProperty;
    /**
     * Creates a continuous palette. This constructor is only used internally during
     * construction of the [[Properties]] instance of the [[RenderBase]].
     * @param _source The underlying continuous palette property.
     * @ignore
     */
    constructor(_source: IRSContPaletteProperty, _dataContext: any, _slotResolver: SlotResolver);
    /**
     * Returns the name of the slot this palette is linked to, or null if the palette
     * is not linked to a slot.
     */
    get slot(): string | null;
    /**
     * Given a slot, returns the color stops for that slot. If the slot is not defined,
     * then a set of default color stops are returned.
     * @param _slot The slot for which to return the color stops.
     * @returns The color stops for the specified slot.
     */
    getColorStops(_slot?: Slot): ColorStops;
    private _fetchColorStops;
    /**
     * Determine the fill color for a data point. The fill follows the 'selected'
     * state of a data point. If the data set has selected data points, then the
     * ones that are selected get the normal color as fill color and the ones that
     * are not selected get a 60% transparent color.
     * @param _dp The data point to determine the color for.
     * @returns The color of the data point as a string.
     */
    getFillColor(_dp: DataPoint): string;
    /**
     * Determine the outline color for a data point. The outline color depends on
     * both the 'highlight' state and the 'selected' state. If the point is neither
     * highlighted or selected, `null` is returned to indicate that no outline needs
     * to be rendered. If the point is highlighted or part of a selection, then
     * the returned color is a darker version of the actual data point color.
     * @param _dp The data point to determine the color for.
     * @returns The color of the data point as a string or null if no outline is needed.
     */
    getOutlineColor(_dp: DataPoint): string | null;
}
