"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("../utils/fs");
const xml = require("../utils/xml");
const config_1 = require("../utils/config");
const events_1 = require("../utils/events");
const constants_1 = require("../constants");
const errors_1 = require("../errors");
/**
 * Helper functions to generate manifest.xml, vizdef.xml and meta.json in the build
 * output of a bundle. For each xml file, a .js module file is generated to allow
 * CORS requests to the file.
 * This module is used by bundle.ts by the createBundle and watchBundle functions.
 */
// Default file write options.
const DEF_WRITE_OPTIONS = {
    encoding: "utf8"
};
/**
 * Creates default manifest.xml content and returns it as a string so it can be written
 * into a manifest.xml file.
 * @param _config Bundle configuration (based on the bundle's package.json).
 * @param _hasNls True if the bundle includes nls resources.
 */
function createManifestFromConfig(_config, _hasNls) {
    // Include <Bundle-Nls> element if the bundle has nls resources.
    const bundleNls = _hasNls ? `
    <Bundle-Nls lang="js">./nls/Resources</Bundle-Nls>` : "";
    return `<?xml version="1.0" encoding="UTF-8"?>
<manifest xmlns="http://www.ibm.com/xmlns/prod/ba/vipr/vizBundle/manifest.1">
    <Bundle-SymbolicName>${_config.symbolicName}</Bundle-SymbolicName>
    <Bundle-Name>${_config.name}</Bundle-Name>
    <Bundle-ManifestVersion>${_config.manifestVersion}</Bundle-ManifestVersion>
    <Bundle-Version>${_config.version}</Bundle-Version>
    <Bundle-Type>visualization</Bundle-Type>
    <ServiceDefinition>vizdef.xml</ServiceDefinition>${bundleNls}
    <ServiceBindings>
        <Service format="client" lang="js" class="renderer/Main" apiversion="${_config.apiVersion}" />
    </ServiceBindings>
</manifest>
`;
}
/**
 * Generates a vizdef.xml and vizdef.xml.js in a specified output location. The vizdef
 * is generated from a given input vizdef location.
 * @param _path Location where a source vizdef.xml can be found.
 * @param _outputPath Output location for vizdef.xml and vizdef.xml.js.
 * @returns A promise that resolves when the output vizdef is created.
 */
async function genVizDef(_path, _outputPath) {
    events_1.emitEvent(constants_1.BUILDER_EVENTS.GENERATING_VIZDEF);
    // Validate if source vizdef is present.
    const vizDefPath = path.join(_path, "vizdef.xml");
    if (!(await fs.exists(vizDefPath)))
        throw new Error(errors_1.NO_VIZDEF_ERROR);
    // Read vizdef.xml and validate it is correct xml.
    const vizDefContent = await fs.readFile(vizDefPath);
    if (!(await xml.isValidXML(vizDefContent)))
        throw new Error(errors_1.INVALID_VIZDEF_FORMAT);
    // Write a javascript module around the vizdef and copy to the output path.
    const jsContent = xml.asModule(vizDefContent);
    const outputPath = path.join(_path, constants_1.OUTPUT_DIR);
    const targetXml = path.join(outputPath, "vizdef.xml");
    const targetJS = path.join(outputPath, "vizdef.xml.js");
    await Promise.all([
        fs.writeFile(targetXml, vizDefContent, DEF_WRITE_OPTIONS),
        fs.writeFile(targetJS, jsContent, DEF_WRITE_OPTIONS),
    ]);
}
exports.genVizDef = genVizDef;
/**
 * Generates a manifest.xml and manifest.xml.js in a specified output location. The
 * manifest is either generated from an existing manifest in a source location, or
 * created from scratch based on a manifest template (see beginning of this file).
 * @param _path Location where a source manifest might be found.
 * @param _outputPath Output location for manifest.xml and manifest.xml.js.
 * @returns A promise that resolves when the output manifest is created.
 */
async function genManifest(_path, _outputPath) {
    events_1.emitEvent(constants_1.BUILDER_EVENTS.GENERATING_MANIFEST);
    const config = await config_1.getConfig(_path);
    let manifestContent;
    const manifestPath = path.join(_path, "manifest.xml");
    if ((await fs.exists(manifestPath))) {
        // Source manifest exists, read it and convert to JSON.
        manifestContent = await fs.readFile(manifestPath);
        const manifestJson = await xml.getJSONFromXML(manifestContent).catch((_err) => {
            throw new Error(errors_1.INVALID_MANIFEST_FORMAT);
        });
        // Validate symbolic-name and name in manifest against meta data.
        if (config.name !== manifestJson.manifest["Bundle-Name"][0])
            throw new Error(errors_1.INVALID_MANIFEST_META_NAME);
        if (config.symbolicName !== manifestJson.manifest["Bundle-SymbolicName"][0])
            throw new Error(errors_1.INVALID_MANIFEST_META_ID);
    }
    else {
        // Source manifest does not exist, determine content from template.
        const nlsExists = await fs.exists(path.join(_path, "nls", "Resources.js"));
        manifestContent = createManifestFromConfig(config, nlsExists);
    }
    // Write manifest content to manifest.xml and a module version to manifest.xml.js.
    const jsContent = xml.asModule(manifestContent);
    const outputPath = path.join(_path, constants_1.OUTPUT_DIR);
    const targetXml = path.join(outputPath, "manifest.xml");
    const targetJS = path.join(outputPath, "manifest.xml.js");
    await Promise.all([
        fs.writeFile(targetXml, manifestContent, DEF_WRITE_OPTIONS),
        fs.writeFile(targetJS, jsContent, DEF_WRITE_OPTIONS),
    ]);
}
exports.genManifest = genManifest;
/**
 * Generates a meta.json file in the specified output location. The meta.json is
 * generated from a source package.json and the information found in the package.json
 * of customvis-lib and cli.
 * @param _path Path where the source package.json (of the bundle) can be found.
 * @param _outputPath Output location for meta.json.
 * @returns A promise that resolves when the meta.json is created.
 */
async function genMetaInfo(_path, _outputPath) {
    events_1.emitEvent(constants_1.BUILDER_EVENTS.GENERATING_META);
    // Retrieve package.json from customvis-lib, and cli.
    const libInfo = await fs.readJSON(path.join(_path, constants_1.LIB_PACKAGE_PATH, "package.json"));
    const cliInfo = await fs.readJSON(path.resolve(__dirname, "../../package.json"));
    // Create meta.json based on config derived from bundle package.json.
    const config = await config_1.getConfig(_path);
    const output = Object.assign({ name: config.name, id: config.symbolicName, type: "renderer", version: config.version, description: config.description, cliVersion: cliInfo.version, apiVersion: config.apiVersion, libVersion: libInfo.version }, (config.icon ? { icon: config.icon } : {}));
    // Write as meta.json file on the output location.
    const outputPath = path.join(_path, constants_1.OUTPUT_DIR, "meta.json");
    await fs.writeJSON(outputPath, output);
}
exports.genMetaInfo = genMetaInfo;
