"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const typescript_1 = require("./typescript");
const constants_1 = require("../constants");
const constants_2 = require("../constants");
const errors_1 = require("../errors");
const fs = require("../utils/fs");
const events_1 = require("../utils/events");
const rollup_1 = require("./rollup");
/**
 * Extend the specified rollup config with mandatory defaults (like output file
 * name and module format).
 * @param _path Root path of the bundle. Used to determine the build output file.
 * @param _config Rollup config.
 * @returns An extended rollup config based on the specified one.
 */
function extendWithDefaults(_path, _config) {
    const externalDefault = ["requirejs"];
    const defaultPlugins = [];
    const outputOptions = (_config.output || {});
    return Object.assign(Object.assign({}, _config), { external: Array.isArray(_config.external) ? _config.external.concat(externalDefault) : externalDefault, plugins: Array.isArray(_config.plugins) ? _config.plugins.concat(defaultPlugins) : defaultPlugins, output: Object.assign(Object.assign({}, outputOptions), { paths: Object.assign(Object.assign({}, (outputOptions.paths || {})), { "requirejs": "require" }), file: path.join(_path, constants_1.OUTPUT_DIR, "renderer", "Main.js"), format: "amd" }) });
}
/**
 * Compares two (dev-)dependencies objects for equality.
 * @param _deps1 A dependencies object from package.json
 * @param _deps2 A dependencies object from package.json
 * @returns True if both object have the same entries, false otherwise.
 */
function compareDependencies(_deps1, _deps2) {
    // Fallback to empty dependencies list if undefined or null.
    const deps1 = _deps1 || {};
    const deps2 = _deps2 || {};
    // Number of entries should be same and each entry should have an exact match.
    const names1 = Object.getOwnPropertyNames(deps1);
    const names2 = Object.getOwnPropertyNames(deps2);
    if (names1.length !== names2.length)
        return false;
    return names1.every(_name => deps1[_name] === deps2[_name]);
}
/**
 * Retrieve rollup options from the specified bundle location. An _overrides parameter
 * allows you to specify override options.
 * @param _path Location of a bundle.
 * @param _overrides Overrides to merge with the rollup options found at _path.
 * @returns A promise that resolves with the merged rollup options.
 */
async function getRollupOptions(_path, _overrides) {
    const configPath = path.join(_path, constants_2.ROLLUP_CONFIG);
    if (!(await fs.exists(configPath)))
        throw new Error(errors_1.NO_ROLLUP_CONFIG);
    const config = fs.moduleRequire(configPath);
    return extendWithDefaults(_path, Object.assign(Object.assign(Object.assign({}, config), _overrides), { output: Object.assign(Object.assign({}, config.output), (_overrides.output || {})) }));
}
/**
 * Transpile the sources. Transpiling is the process of building the code for the bundle
 * into the build folder. Next to generating / copying vizdef and manifest this is part
 * of the build process.
 * @param _path Root path of the bundle.
 * @param _rollupOverrides Rollup config overrides.
 * @returns A promise that resolves when the sources are transpiled.
 */
async function transpile(_path, _rollupOverrides) {
    events_1.emitEvent(constants_1.BUILDER_EVENTS.START_TRANSPILING);
    const startTime = Date.now();
    const options = await getRollupOptions(_path, _rollupOverrides);
    // Use Typescript (node-ts) to check for ts errors (if applicable).
    await typescript_1.createTypescriptProgram(_path).typeCheck(options.input);
    await rollup_1.rollupTranspile(options);
    events_1.emitEvent(constants_1.BUILDER_EVENTS.DONE_TRANSPILING, (Date.now() - startTime));
}
exports.transpile = transpile;
/**
 * Start watching and transpile the sources on change.
 * @param _path Root path of the bundle.
 * @param _rollupOverrides Rollup config overrides.
 * @returns A promise that resolves with a file watcher instance. Callers can use this
 * instance to subscribe to events or to stop watching.
 */
async function watch(_path, _rollupOverrides) {
    let cache = undefined;
    let isBuilding = false;
    const options = await getRollupOptions(_path, _rollupOverrides);
    const tsProgram = typescript_1.createTypescriptProgram(_path);
    async function build() {
        // Only allow one build at a time.
        if (isBuilding)
            return;
        try {
            isBuilding = true;
            events_1.emitEvent(constants_1.BUILDER_EVENTS.START_TRANSPILING);
            const startTime = Date.now();
            // For type checking, swallow the error and print to the console.
            await tsProgram.typeCheck(options.input);
            const data = await rollup_1.rollupTranspile(Object.assign(Object.assign({}, options), { cache }));
            events_1.emitEvent(constants_1.BUILDER_EVENTS.DONE_TRANSPILING, (Date.now() - startTime));
            cache = data.cache;
        }
        catch (_err) {
            events_1.emitError(0 /* FAILED_TRANSPILING */, (_err && _err.message || "unknown error"));
        }
        finally {
            isBuilding = false;
        }
    }
    await build();
    // Create a chokidar watcher since rollup watcher is bugged and run build.
    // See https://www.npmjs.com/package/rollup-plugin-typescript2#watch-mode
    const watcher = fs.watchFiles([
        "renderer/**/*.ts",
        "renderer/**/*.js",
        `${constants_1.LIB_PACKAGE_PATH}/lib/index.js`,
        "./*.ts"
    ], {
        cwd: _path,
        ignoreInitial: true,
        interval: 500,
        ignored: [constants_1.OUTPUT_DIR],
    }).on("change", build).on("add", build).on("unlink", build);
    return watcher;
}
exports.watch = watch;
/**
 * Generate a rollup template. A rollup template is basically a rollup config file
 * that holds the default set of rollup options. The getRollupOptions function, used
 * in transpile and watch, reads this file and allows overriding specific options.
 * @param _path The location where to create the rollup template. Typically the
 * root path of a bundle.
 * @returns A promise that resolves when the template has been generated.
 */
async function generateRollupTemplate(_path) {
    await fs.copy(path.join(fs.getStaticPath(), constants_2.ROLLUP_CONFIG), path.join(_path, constants_2.ROLLUP_CONFIG));
}
exports.generateRollupTemplate = generateRollupTemplate;
/**
 * Add rollup (dev-)dependencies to a bundle by merging the existing devDependencies
 * with the set of dependencies found in sdk's static/rollup-dependencies.json.
 * @param _path The path of a bundle.
 * @returns A promise that resolves when the function is done updating package.json.
 * The promise resolves with true if the file has changed, or false if the file was
 * left unchanged.
 */
async function addDevDependenciesToPackage(_path) {
    // Modify the package.json to include the rollup plugin used in the default config
    const bundlePkgJsonPath = path.join(_path, "package.json");
    const [bundlePkg, rollupDeps] = await Promise.all([
        fs.readJSON(bundlePkgJsonPath),
        fs.readJSON(path.join(fs.getStaticPath(), constants_2.ROLLUP_DEPS))
    ]);
    // Add rollup dependencies to the existing dev dependencies.
    const jsonOutput = Object.assign(Object.assign({}, bundlePkg), { devDependencies: Object.assign(Object.assign({}, rollupDeps), bundlePkg.devDependencies) });
    // Only devDependencies can be changed.
    if (compareDependencies(bundlePkg.devDependencies, jsonOutput.devDependencies))
        return false; // no changes
    await fs.writeJSON(bundlePkgJsonPath, jsonOutput);
    return true; // package.json changed
}
exports.addDevDependenciesToPackage = addDevDependenciesToPackage;
