"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("../utils/fs");
const path = require("path");
const errors_1 = require("../errors");
/**
 * Lookup table to convert file extensions to http mime-type
 */
const EXTENSIONS = {
    "js": "application/javascript",
    "json": "application/json",
    "jpg": "image/jpeg",
    "png": "image/png",
    "gif": "image/gif",
    "svg": "image/svg+xml",
    "css": "text/css",
    "xml": "application/xml"
};
/**
 * Get a mime-type for a file
 * @param _filePath absolute path to a file
 */
function getMimeType(_filePath) {
    const ext = path.extname(_filePath).slice(1);
    return EXTENSIONS[ext] || "text/plain";
}
exports.getMimeType = getMimeType;
/**
 * Find a file on the disk for a specific `_url` path inside a `_contentBase` directory
 * @param _contentBase A list of directories to search in.
 * @param _urlPath A url path of a file.
 * @returns The full path of the found file or null if the file is not found.
 */
async function findFile(_contentBase, _urlPath) {
    for (let i = 0, len = _contentBase.length; i < len; ++i) {
        const fullPath = path.join(_contentBase[i], _urlPath);
        if (await (fs.exists(fullPath)))
            return fullPath;
    }
    return null;
}
/**
 * Response with "404: not found" for a specific `_filePath`
 * @param _res http response.
 * @param _filePath The path that was not found.
 * @param _headers The response headers.
 */
function notFound(_res, _filePath, _headers) {
    _res.writeHead(404, _headers);
    _res.end("404 Not Found\n\n" + _filePath, "utf-8");
}
/**
 * Return the necessary headers for CORS. This function assumes that every request
 * comes from a cross origin domain (customvis runs on localhost).
 * @param _request Incoming request
 * @returns An object that contains the CORS headers.
 */
function getCorsHeaders(_request) {
    const corsHeaders = {};
    corsHeaders["Access-Control-Allow-Origin"] = "*";
    if (_request.method === "OPTIONS") // Is this a pre-flight check?
     {
        corsHeaders["Access-Control-Allow-Methods"] = "GET, HEAD, OPTIONS";
        corsHeaders["Access-Control-Allow-Headers"] = "*";
    }
    return corsHeaders;
}
/**
 * Create a request listener function to serves static files.
 * @param _options
 */
function createReqListener(_options) {
    return async (_req, _res) => {
        // Merge CORS headers and headers from _options.
        const headers = Object.assign({}, getCorsHeaders(_req), _options.headers);
        // Stop if is an OPTIONS request.
        if (_req.method === "OPTIONS") {
            headers["Content-Length"] = 0;
            _res.writeHead(200, headers);
            _res.end();
            return;
        }
        // Request must have a URL. If not, we can not return anything because the request
        // URL determines the path of the file to return.
        if (!_req.url) {
            notFound(_res, errors_1.SERVER_INVALID_URL, headers);
            return;
        }
        // Determine part url of the request (without terminating '/'). This, together with
        // the content base path, will determine the resource to respond with. Typically
        // the _req.url will be something like "/manifest.xml" or "/nls/Resources.js".
        let urlPath = decodeURI(_req.url.split("?")[0]);
        if (urlPath.endsWith("/"))
            urlPath = urlPath.slice(0, -1);
        try {
            const filePath = await findFile(_options.contentBase, urlPath);
            if (filePath === null) {
                notFound(_res, urlPath, headers);
                return; // requested file does not exist
            }
            const fileStats = await fs.stat(filePath);
            if (fileStats.isDirectory()) {
                notFound(_res, errors_1.SERVER_INVALID_PATH, headers);
                return; // cannot request a folder resource
            }
            // Found the file, read it as a text file and return it in the response.
            headers["Content-Type"] = getMimeType(filePath);
            headers["Content-Length"] = fileStats.size || 0;
            _res.writeHead(200, headers);
            const data = await fs.readFile(filePath);
            _res.write(data);
            _res.end();
        }
        catch (_err) {
            // Treat all errors as '404 Not Found' and provide the error message.
            notFound(_res, _err.message, headers);
        }
    };
}
exports.createReqListener = createReqListener;
