"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const http = require("http");
const path = require("path");
const constants_1 = require("../constants");
const process_1 = require("../utils/process");
const events_1 = require("../utils/events");
const createReqListener_1 = require("./createReqListener");
/**
 * Installs a handler that closes the server on CLI termination events.
 * @param _server The NodeJS server instance that should be closed.
 * @returns The same NodeJS server instance.
 */
function closeServerOnTermination(_server) {
    const terminate = () => {
        _server.close();
        events_1.emitEvent(constants_1.BUILDER_EVENTS.SERVER_STOPPED);
    };
    // Note: don't chain the .on functions. The getProcess function might be mocked in
    // the unit tests, which means that on() might not return the same mocked process.
    const process = process_1.getProcess();
    process.on("SIGTERM", terminate);
    process.on("SIGINT", terminate);
    return _server;
}
/**
 * Start an http server.
 * @param _path Typically the path where the bundle is located and should be served from.
 * @param _options Optional server options.
 * @returns A running NodeJS Server instance.
 */
function start(_path, _options = {}) {
    const mergedOptions = Object.assign({ headers: {}, https: false, port: 8585, contentBase: [path.join(_path, constants_1.OUTPUT_DIR), "."] }, _options);
    events_1.emitEvent(constants_1.BUILDER_EVENTS.SERVER_STARTING, mergedOptions.port);
    const server = http.createServer(createReqListener_1.createReqListener(mergedOptions));
    server.listen(mergedOptions.port)
        .on("listening", () => {
        events_1.emitEvent(constants_1.BUILDER_EVENTS.SERVER_STARTED, mergedOptions.port);
    })
        .on("error", (_err) => {
        if (_err.errno === "EADDRINUSE")
            events_1.emitError(4 /* SERVER_PORT_IN_USE */, mergedOptions.port, mergedOptions.port);
        else
            events_1.emitError(5 /* SERVER_FAILED_TO_START */, _err.message);
    });
    return closeServerOnTermination(server);
}
exports.start = start;
