"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("./fs");
const path = require("path");
const constants = require("../constants");
const errors_1 = require("../errors");
/**
 * Get the config that is determined from package.json.
 * @param _path Path of the bundle.
 */
async function getConfig(_path) {
    // Read and validate package.json from the bundle.
    const configPath = path.join(_path, "package.json");
    const pkgJson = await fs.readJSON(configPath);
    const meta = pkgJson && pkgJson.meta;
    if (!meta)
        throw new Error(errors_1.PACKAGE_NO_META_INFO);
    if (!meta.name)
        throw new TypeError(errors_1.PACKAGE_NO_META_NAME);
    if (!meta.id)
        throw new TypeError(errors_1.PACKAGE_NO_META_ID);
    if (!meta.type || meta.type !== "renderer")
        throw new TypeError(errors_1.PACKAGE_NO_META_TYPE);
    // Read package.json from the sdk (to determine the apiVersion).
    const libPkg = await fs.readJSON(path.join(_path, constants.LIB_PACKAGE_PATH, "package.json"));
    return {
        name: meta.name.toString(),
        symbolicName: meta.id.toLowerCase() || "com.ibm.vis.custom.bundle",
        type: meta.type,
        manifestVersion: "1.0",
        description: meta.description || "",
        version: meta.version && meta.version.toLocaleString("en") || "1.0",
        apiVersion: libPkg.apiversion,
        icon: meta.icon
    };
}
exports.getConfig = getConfig;
