"use strict";
// Licensed Materials - Property of IBM
//
// IBM Watson Analytics
//
// (C) Copyright IBM Corp. 2019, 2020
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs-extra");
const path = require("path");
const chokidar = require("chokidar");
const archiver_1 = require("archiver");
const errors_1 = require("../errors");
/**
 * Returns the absolute path to the sdk's static folder. Typically this is the
 * folder where the sdk keeps rollup configuration files.
 * @returns The absolute path of the sdk folder where static files are located.
 */
function getStaticPath() {
    return path.join(__dirname, "../../static");
}
exports.getStaticPath = getStaticPath;
/**
 * Returns the absolute path to the sdk's template folder.
 * @returns The absolute path of the sdk folder where templates are located.
 */
function getTemplatePath() {
    return path.join(__dirname, "../../templates");
}
exports.getTemplatePath = getTemplatePath;
/**
 * Returns true if the bundle found at the specified location is a 'development bundle'.
 * A bundle is considered a 'development bundle' if it has a 'package.json' and contains
 * source code that can be built by customvis. A non-development bundle typically is one
 * that is already built by customvis and has no source code or package.json.
 * @param _path Path of a bundle.
 * @returns A promise that resolves with a boolean: true if the bundle is a development
 * bundle, false otherwise.
 */
async function isDevBundle(_path) {
    const pkgPath = path.join(_path, "package.json");
    return exists(pkgPath);
}
exports.isDevBundle = isDevBundle;
/**
 * Performs a 'require' of a module. If the module has an exported 'default', then
 * this is returned instead of the module.
 * @param _path Path of a module.
 * @returns The required module or the default export from the module.
 */
function moduleRequire(_path) {
    const mod = require(_path);
    return (mod.default || mod);
}
exports.moduleRequire = moduleRequire;
/**
 * Generate a zip file
 * @param _targetDir the directory which is being zipped
 * @param _destDir the directory which the zip file will be written to
 * @param _name the name of the final zip file
 */
function zipBundle(_targetDir, _destDir, _name) {
    return new Promise((_resolve, _reject) => {
        try {
            const outputPath = path.join(_destDir, _name + ".zip");
            const output = fs.createWriteStream(outputPath);
            const archive = archiver_1.create("zip", {
                zlib: { level: 8 } // Sets the compression level.
            });
            archive.pipe(output);
            output.on("close", () => {
                _resolve(outputPath);
            });
            output.on("warning", (_err) => {
                throw _err;
            });
            output.on("error", (_err) => {
                throw _err;
            });
            archive.directory(_targetDir, false);
            archive.finalize();
        }
        catch (_err) {
            _reject(errors_1.createFailedToZipError(_targetDir));
        }
    });
}
exports.zipBundle = zipBundle;
// ----------------------------------------------------------------------------
// fs-extra overrides
// ----------------------------------------------------------------------------
async function exists(_path) {
    return fs.pathExists(_path);
}
exports.exists = exists;
async function copy(_src, _dest) {
    await fs.copy(_src, _dest);
}
exports.copy = copy;
async function readJSON(_file) {
    return fs.readJSON(_file);
}
exports.readJSON = readJSON;
async function writeJSON(_file, _object, _options = { spaces: 4 }) {
    return fs.writeJSON(_file, _object, _options);
}
exports.writeJSON = writeJSON;
async function readdir(_path) {
    return fs.readdir(_path);
}
exports.readdir = readdir;
async function remove(_path) {
    return fs.remove(_path);
}
exports.remove = remove;
async function ensureDir(_path) {
    return fs.ensureDir(_path);
}
exports.ensureDir = ensureDir;
async function readFile(_sourcePath) {
    return fs.readFile(_sourcePath, { encoding: "utf8" });
}
exports.readFile = readFile;
async function writeFile(_targetPath, _content, _options) {
    return fs.writeFile(_targetPath, _content, _options);
}
exports.writeFile = writeFile;
async function stat(_filePath) {
    return fs.stat(_filePath);
}
exports.stat = stat;
// ----------------------------------------------------------------------------
// chokidar overrides
// ----------------------------------------------------------------------------
function watchFiles(_input, _options) {
    return chokidar.watch(_input, _options);
}
exports.watchFiles = watchFiles;
