/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * Runtime/Startup class
 * this is the central class which initializes all base mechanisms
 * used by the rest of the system such as
 * a) namespacing system
 * b) browser detection
 * c) loose configuration coupling
 * d) utils methods to fetch the implementation
 * e) ajaxed script loading
 * f) global eval (because it is used internally)
 * g) Structural base patterns as singleton, delegate and inheritance
 *
 * Note this class is self contained and must!!! be loaded
 * as absolute first class before going into anything else
 *
 *
 */
/** @namespace myfaces._impl.core._Runtime*/

(!window.myfaces) ? window.myfaces = {} : null;
(!myfaces._impl) ? myfaces._impl = {} : null;
(!myfaces._impl.core) ? myfaces._impl.core = {} : null;
//now this is the only time we have to do this cascaded and manually
//for the rest of the classes our reserveNamespace function will do the trick
//Note, this class uses the classical closure approach (to save code)
//it cannot be inherited by our inheritance mechanism, but must be delegated
//if you want to derive from it
//closures and prototype inheritance do not mix, closures and delegation however do
if (!myfaces._impl.core._Runtime) {
    myfaces._impl.core._Runtime = new function() {
        //the rest of the namespaces can be handled by our namespace feature
        //helper to avoid unneeded hitches
        var _T = this;

        //namespace idx to speed things up by hitting eval way less
        _T._reservedNMS = {};

        /**
         * replacement counter for plugin classes
         */
        _T._classReplacementCnt = 0;

        /**
         * global eval on scripts
         *
         * usage return _T.globalEval('myvar.myvar2;');
         *
         *
         * Note some libraries like jquery use html head attachments
         * to run the global eval, at least for loaded scripts
         * this methid was flaky and failed on chrome under certain conditions,
         * since our method works reliably in modern browsers currently in use
         * we do it via eval, we still can switch to the head method
         * if there are arguments why that one works better than ours
         */
        _T.globalEval = function(code) {
            //TODO add a config param which allows to evaluate global scripts even if the call
            //is embedded in an iframe
            if (window.execScript) {
                //execScript definitely only for IE otherwise we might have a custom
                //window extension with undefined behavior on our necks
                //window.execScript does not return anything
                //on htmlunit it return "null object"
                var ret = window.execScript(code);
                if ('undefined' != typeof ret && ret == "null" /*htmlunit bug*/) {
                    return null;
                }
                return ret;
            } else if (window.eval) {

                //fix which works in a cross browser way
                //we used to scope an anonymous function
                //but I think this is better
                //the reason is firefox applies a wrong scope
                //if we call eval by not scoping

                if (!_T.browser.isBlackBerry || _T.browser.isBlackBerry >= 6) {
                    var gEval = function () {

                        var ret = window.eval.call(window, code);
                        if ('undefined' == typeof ret) return null;
                        return ret;
                    };
                    var ret = gEval();
                    if ('undefined' == typeof ret) return null;
                    return ret;
                } else {
                    //blackberry 5- only understands the flaky head method
                    //which fails on literally all newer browsers one way or the other
                    return _T._globalEvalHeadAppendixMethod(code);
                }
                //we scope the call in window

            }
            //we probably have covered all browsers, but this is a safety net which might be triggered
            //by some foreign browser which is not covered by the above cases
            eval.call(window, code);
            return null;
        };

        /**
         * flakey head appendix method which does not work in the correct
         * order or at all for all modern browsers
         * but seems to be the only method which works on blackberry correctly
         * hence we are going to use it as fallback
         *
         * @param code the code part to be evaled
         */
        _T._globalEvalHeadAppendixMethod = function(code) {
            var location = document.getElementsByTagName("head")[0] || document.documentElement;
            var placeHolder = document.createElement("script");
            placeHolder.type = "text/javascript";
            placeHolder.text = code;
            location.insertBefore(placeHolder, location.firstChild);
            location.removeChild(placeHolder);
            return null;
        };

        /**
         * applies an object to a namespace
         * basically does what bla.my.name.space = obj does
         * note we cannot use var myNameSpace = fetchNamespace("my.name.space")
         * myNameSpace = obj because the result of fetch is already the object
         * which the namespace points to, hence this function
         *
         * @param nms the namespace to be assigned to
         * @param obj the  object to be assigned
         */
        _T.applyToGlobalNamespace = function(nms, obj) {
            var splitted = nms.split(/\./);
            if (splitted.length == 1) {
                window[nms] = obj;
                return;
            }
            var parent = splitted.slice(0, splitted.length - 1);
            var child = splitted[splitted.length - 1];
            var parentNamespace = _T.fetchNamespace(parent.join("."));
            parentNamespace[child] = obj;
        };

        /**
         * fetches the object the namespace points to
         * @param nms the namespace which has to be fetched
         * @return the object the namespace points to or null if nothing is found
         */
        _T.fetchNamespace = function(nms) {
            if ('undefined' == typeof nms || null == nms || !_T._reservedNMS[nms]) {
                return null;
            }

            var ret = null;
            try {
                //blackberries have problems as well in older non webkit versions
                if (!_T.browser.isIE) {
                    //in ie 6 and 7 we get an error entry despite the suppression
                    ret = _T.globalEval("window." + nms);
                }
                //namespace could point to numeric or boolean hence full
                //save check

            } catch (e) {/*wanted*/
            }
            //ie fallback for some ie versions path because it cannot eval namespaces
            //ie in any version does not like that particularily
            //we do it the hard way now
            if ('undefined' != typeof ret && null != ret) {
                return ret;
            }
            nms = nms.split(/\./);
            ret = window;
            var len = nms.length;

            for (var cnt = 0; cnt < len; cnt++) {
                ret = ret[nms[cnt]];
                if ('undefined' == typeof ret || null == ret) {
                    return null;
                }
            }
            return ret;

        };

        /**
         * Backported from dojo
         * a failsafe string determination method
         * (since in javascript String != "" typeof alone fails!)
         * @param it {|Object|} the object to be checked for being a string
         * @return true in case of being a string false otherwise
         */
        _T.isString = function(/*anything*/ it) {
            //	summary:
            //		Return true if it is a String
            return !!arguments.length && it != null && (typeof it == "string" || it instanceof String); // Boolean
        };

        /**
         * reserves a namespace in the specific scope
         *
         * usage:
         * if(_T.reserve("org.apache.myfaces.MyUtils")) {
         *      org.apache.myfaces.MyUtils = function() {
         *      }
         * }
         *
         * reserves a namespace and if the namespace is new the function itself is reserved
         *
         *
         *
         * or:
         * _T.reserve("org.apache.myfaces.MyUtils", function() { .. });
         *
         * reserves a namespace and if not already registered directly applies the function the namespace
         *
         * note for now the reserved namespaces reside as global maps justl like jsf.js but
         * we also use a speedup index which is kept internally to reduce the number of evals or loops to walk through those
         * namespaces (eval is a heavy operation and loops even only for namespace resolution introduce (O)2 runtime
         * complexity while a simple map lookup is (O)log n with additional speedup from the engine.
         *
         *
         * @param {|String|} nms
         * @returns true if it was not provided
         * false otherwise for further action
         */
        _T.reserveNamespace = function(nms, obj) {

            if (!_T.isString(nms)) {
                throw Error("Namespace must be a string with . as delimiter");
            }
            if (_T._reservedNMS[nms] || null != _T.fetchNamespace(nms)) {
                return false;
            }

            var entries = nms.split(/\./);
            var currNms = window;

            var tmpNmsName = [];

            for (var cnt = 0; cnt < entries.length; cnt++) {
                var subNamespace = entries[cnt];
                tmpNmsName.push(subNamespace);
                if ('undefined' == typeof currNms[subNamespace]) {
                    currNms[subNamespace] = {};
                }
                if (cnt == entries.length - 1 && 'undefined' != typeof obj) {
                    currNms[subNamespace] = obj;
                } else {
                    currNms = currNms[subNamespace];
                }
                _T._reservedNMS[tmpNmsName.join(".")] = true;
            }
            return true;
        };

        /**
         * check if an element exists in the root
         * also allows to check for subelements
         * usage
         * _T.exists(rootElem,"my.name.space")
         * @param {Object} root the root element
         * @param {String} subNms the namespace
         */
        _T.exists = function(root, subNms) {
            if (!root) {
                return false;
            }
            //special case locally reserved namespace
            if (root == window && _T._reservedNMS[subNms]) {
                return true;
            }

            //initial condition root set element not set or null
            //equals to element exists
            if (!subNms) {
                return true;
            }
            try {
                //special condition subnamespace exists as full blown key with . instead of function map
                if ('undefined' != typeof root[subNms]) {
                    return true;
                }

                //crossported from the dojo toolkit
                // summary: determine if an object supports a given method
                // description: useful for longer api chains where you have to test each object in the chain
                var p = subNms.split(".");
                var len = p.length;
                for (var i = 0; i < len; i++) {
                    //the original dojo code here was false because
                    //they were testing against ! which bombs out on exists
                    //which has a value set to false
                    // (TODO send in a bugreport to the Dojo people)

                    if ('undefined' == typeof root[p[i]]) {
                        return false;
                    } // Boolean
                    root = root[p[i]];
                }
                return true; // Boolean

            } catch (e) {
                //ie (again) has a special handling for some object attributes here which automatically throw an unspecified error if not existent
                return false;
            }
        };

        /**
         * A dojo like require to load scripts dynamically, note
         * to use this mechanism you have to set your global config param
         * myfacesScriptRoot to the root of your script files (aka under normal circumstances
         * resources/scripts)
         *
         * @param localOptions
         * @param configName
         * @param defaultValue
         */
        _T.require = function(nms) {
            //namespace exists
            if (_T.exists(nms)) return;
            var rootPath = _T.getGlobalConfig("myfacesScriptRoot", "");
            _T.loadScriptEval(rootPath + "/" + nms.replace(/\./g, "/") + ".js");
        },

        /**
         * fetches a global config entry
         * @param {String} configName the name of the configuration entry
         * @param {Object} defaultValue
         *
         * @return either the config entry or if none is given the default value
         */
                _T.getGlobalConfig = function(configName, defaultValue) {
                    /**
                     * note we could use exists but this is an heavy operation, since the config name usually
                     * given this function here is called very often
                     * is a single entry without . in between we can do the lighter shortcut
                     */
                    return (myfaces["config"] && 'undefined' != typeof myfaces.config[configName] ) ?
                            myfaces.config[configName]
                            :
                            defaultValue;
                };

        /**
         * gets the local or global options with local ones having higher priority
         * if no local or global one was found then the default value is given back
         *
         * @param {String} configName the name of the configuration entry
         * @param {String} localOptions the local options root for the configuration myfaces as default marker is added implicitely
         *
         * @param {Object} defaultValue
         *
         * @return either the config entry or if none is given the default value
         */
        _T.getLocalOrGlobalConfig = function(localOptions, configName, defaultValue) {
            /*use(myfaces._impl._util)*/
            var _local = !!localOptions;
            var _localResult;
            if (_local) {
                //note we also do not use exist here due to performance improvement reasons
                //not for now we loose the subnamespace capabilities but we do not use them anyway
                //this code will give us a performance improvement of 2-3%
                _localResult = (localOptions["myfaces"]) ? localOptions["myfaces"][configName] : undefined;
                _local = 'undefined' != typeof _localResult;
            }

            return (!_local) ? _T.getGlobalConfig(configName, defaultValue) : _localResult;
        };

        /**
         * determines the xhr level which either can be
         * 1 for classical level1
         * 1.5 for mozillas send as binary implementation
         * 2 for xhr level 2
         */
        _T.getXHRLvl = function() {
            if (!_T.XHR_LEVEL) {
                _T.getXHRObject();
            }
            return _T.XHR_LEVEL;
        };

        /**
         * encapsulated xhr object which tracks down various implementations
         * of the xhr object in a browser independent fashion
         * (ie pre 7 used to have non standard implementations because
         * the xhr object standard came after IE had implemented it first
         * newer ie versions adhere to the standard and all other new browsers do anyway)
         *
         * @return the xhr object according to the browser type
         */
        _T.getXHRObject = function() {
            //since this is a global object ie hates it if we do not check for undefined
            if (window.XMLHttpRequest) {
                var _ret = new XMLHttpRequest();
                //we now check the xhr level
                //sendAsBinary = 1.5 which means mozilla only
                //upload attribute present == level2
                /*
                 if (!_T.XHR_LEVEL) {
                 var _e = _T.exists;
                 _T.XHR_LEVEL = (_e(_ret, "sendAsBinary")) ? 1.5 : 1;
                 _T.XHR_LEVEL = (_e(_ret, "upload") && 'undefined' != typeof FormData) ? 2 : _T.XHR_LEVEL;
                 }*/
                return _ret;
            }
            //IE
            try {
                _T.XHR_LEVEL = 1;
                return new ActiveXObject("Msxml2.XMLHTTP");
            } catch (e) {

            }
            return new ActiveXObject('Microsoft.XMLHTTP');
        };

        /**
         * loads a script and executes it under a global scope
         * @param {String} src  the source of the script
         * @param {String} type the type of the script
         * @param {Boolean} defer  defer true or false, same as the javascript tag defer param
         * @param {String} charSet the charset under which the script has to be loaded
         */
        _T.loadScriptEval = function(src, type, defer, charSet) {
            var xhr = _T.getXHRObject();
            xhr.open("GET", src, false);

            if (charSet) {
                xhr.setRequestHeader("Content-Type", "application/x-javascript; charset:" + charSet);
            }

            xhr.send(null);

            //since we are synchronous we do it after not with onReadyStateChange
            if (xhr.readyState == 4) {
                if (xhr.status == 200) {
                    //defer also means we have to process after the ajax response
                    //has been processed
                    //we can achieve that with a small timeout, the timeout
                    //triggers after the processing is done!
                    if (!defer) {
                        _T.globalEval(xhr.responseText.replace("\n", "\r\n") + "\r\n//@ sourceURL=" + src);
                    } else {
                        //TODO not ideal we maybe ought to move to something else here
                        //but since it is not in use yet, it is ok
                        setTimeout(function() {
                            _T.globalEval(xhr.responseText + "\r\n//@ sourceURL=" + src);
                        }, 1);
                    }
                } else {
                    throw Error(xhr.responseText);
                }
            } else {
                throw Error("Loading of script " + src + " failed ");
            }
        };

        /**
         * load script functionality which utilizes the browser internal
         * script loading capabilities
         *
         * @param {String} src  the source of the script
         * @param {String} type the type of the script
         * @param {Boolean} defer  defer true or false, same as the javascript tag defer param
         * @param {String} charSet the charset under which the script has to be loaded
         */
        _T.loadScriptByBrowser = function(src, type, defer, charSet) {
            //if a head is already present then it is safer to simply
            //use the body, some browsers prevent head alterations
            //after the first initial rendering

            //ok this is nasty we have to do a head modification for ie pre 8
            //the rest can be finely served with body
            var d = _T.browser;
            var position = "head"
            //if(!d.isIE || d.isIE >= 8) {
            //    position = document.getElementsByTagName("body").length ? "body" : "head";
            //}

            try {
                var holder = document.getElementsByTagName(position)[0];
                if ('undefined' == typeof holder || null == holder) {
                    holder = document.createElement(position);
                    var html = document.getElementsByTagName("html");
                    html.appendChild(holder);
                }
                var script = document.createElement("script");
                script.type = type || "text/javascript";
                script.src = src;
                if (charSet) {
                    script.charset = charSet;
                }
                if (defer) {
                    script.defer = defer;
                }

                //fix for the white page issue
                // if(_T.browser.isIE && _T.browser.isIE < 7) {
                //   holder.insertBefore( script, holder.firstChild );
                //   holder.removeChild( script );
                // } else {
                holder.appendChild(script);
                // }

            } catch (e) {
                //in case of a loading error we retry via eval    
                return false;
            }

            return true;
        };

        _T.loadScript = function(src, type, defer, charSet) {
            //the chrome engine has a nasty javascript bug which prevents
            //a correct order of scripts being loaded
            //if you use script source on the head, we  have to revert
            //to xhr+ globalEval for those
            if (!_T.browser.isFF) {
                _T.loadScriptEval(src, type, defer, charSet);
            } else {
                //only firefox keeps the order, sorry ie...
                _T.loadScriptByBrowser(src, type, defer, charSet)
            }
        };

        //Base Patterns, Inheritance, Delegation and Singleton

        /**
         * delegation pattern
         * usage:
         * this.delegateObject("my.name.space", delegate,
         * {
         *  constructor_ :function(bla, bla1) {
         *      _T._callDelegate("constructor", bla1);
         *  },
         *  myFunc: function(yyy) {
         *      DoSomething;
         *      _T._callDelegate("someOtherFunc", yyyy);
         *  }, null
         * });
         *
         * or
         * usage var newClass = this.delegateObject(
         * function (var1, var2) {
         *  _T._callDelegate("constructor", var1,var2);
         * };
         * ,delegateObject);
         * newClass.prototype.myMethod = function(arg1) {
         *      _T._callDelegate("myMethod", arg1,"hello world");
         *
         *
         * @param newCls the new class name to be generated
         * @param delegateObj the delegation object
         * @param protoFuncs the prototype functions which should be attached
         * @param nmsFuncs the namespace functions which should be attached to the namespace
         */
        _T.delegateObj = function(newCls, delegateObj, protoFuncs, nmsFuncs) {
            if (!_T.isString(newCls)) {
                throw Error("new class namespace must be of type String");
            }

            if ('function' != typeof newCls) {
                newCls = _reserveClsNms(newCls, protoFuncs);
                if (!newCls) return null;
            }

            //central delegation mapping core
            var proto = newCls.prototype;

            //the trick here is to isolate the entries to bind the
            //keys in a private scope see
            //http://www.ruzee.com/blog/2008/12/javascript-inheritance-via-prototypes-and-closures
            for (var key in delegateObj) (function(key, delFn) {
                //The isolation is needed otherwise the last _key assigend would be picked
                //up internally
                if (key && typeof delFn == "function") {
                    proto[key] = function(/*arguments*/) {
                        return delFn.apply(delegateObj, arguments);
                    };
                }
            })(key, delegateObj[key]);

            proto._delegateObj = delegateObj;
            proto.constructor = newCls;

            proto._callDelegate = function(methodName) {
                var passThrough = (arguments.length == 1) ? [] : Array.prototype.slice.call(arguments, 1);
                var ret = this._delegateObj[methodName].apply(this._delegateObj, passThrough);
                if ('undefined' != ret) return ret;
            };

            //we now map the function map in
            _applyFuncs(newCls, protoFuncs, true);
            _applyFuncs(newCls, nmsFuncs, false);

            return newCls;
        };

        /**
         * prototype based delegation inheritance
         *
         * implements prototype delegaton inheritance dest <- a
         *
         * usage var newClass = _T.extends( function (var1, var2) {
         *                                          _T._callSuper("constructor", var1,var2);
         *                                     };
         *                                  ,origClass);
         *
         *       newClass.prototype.myMethod = function(arg1) {
         *              _T._callSuper("myMethod", arg1,"hello world");
         *       ....
         *
         * other option
         *
         * myfaces._impl._core._Runtime.extends("myNamespace.newClass", parent, {
         *                              init: function() {constructor...},
         *                              method1: function(f1, f2) {},
         *                              method2: function(f1, f2,f3) {
         *                                  _T._callSuper("method2", F1,"hello world");
         *                              }
         *              });
         *
         * @param {function|String} newCls either a unnamed function which can be assigned later or a namespace
         * @param {function} extendCls the function class to be extended
         * @param {Object} protoFuncs (Map) an optional map of prototype functions which in case of overwriting a base function get an inherited method
         *
         * To explain further
         * prototype functions:
         *  newClass.prototype.<prototypeFunction>
         * namspace function
         *  newClass.<namespaceFunction> = function() {...}
         */

        _T.extendClass = function(newCls, extendCls, protoFuncs, nmsFuncs) {

            if (!_T.isString(newCls)) {
                throw Error("new class namespace must be of type String");
            }

            if (_T._reservedNMS[newCls]) {
                return;
            }

            if ('function' != typeof newCls) {
                newCls = _reserveClsNms(newCls, protoFuncs);
                if (!newCls) return null;
            }
            //if the type information is known we use that one
            //with this info we can inherit from objects also
            //instead of only from classes
            //sort of like   this.extendClass(newCls, extendObj._mfClazz...

            if (extendCls._mfClazz) {
                extendCls = extendCls._mfClazz;
            }

            if ('undefined' != typeof extendCls && null != extendCls) {
                //first we have to get rid of the constructor calling problem
                //problem
                var tmpFunc = function() {
                };
                tmpFunc.prototype = extendCls.prototype;
                newCls.prototype = new tmpFunc();
                tmpFunc = null;
                newCls.prototype.constructor = newCls;
                newCls.prototype._parentCls = extendCls.prototype;

                newCls.prototype._callSuper = function(methodName) {
                    var passThrough = (arguments.length == 1) ? [] : Array.prototype.slice.call(arguments, 1);

                    //we store the descension level of each method under a mapped
                    //name to avoid name clashes
                    //to avoid name clashes with internal methods of array
                    //if we don't do this we trap the callSuper in an endless
                    //loop after descending one level
                    var _mappedName = ["_",methodName,"_mf_r"].join("");
                    this._mfClsDescLvl = this._mfClsDescLvl || new Array();
                    var descLevel = this._mfClsDescLvl;
                    //we have to detect the descension level
                    //we now check if we are in a super descension for the current method already
                    //if not we are on this level
                    var _oldDescLevel = this._mfClsDescLvl[_mappedName] || this;
                    //we now step one level down
                    var _parentCls = _oldDescLevel._parentCls;

                    try {
                        //we now store the level position as new descension level for callSuper
                        descLevel[_mappedName] = _parentCls;
                        //and call the code on this
                        _parentCls[methodName].apply(this, passThrough);
                    } finally {
                        descLevel[_mappedName] = _oldDescLevel;
                    }
                };
                //reference to its own type
                newCls.prototype._mfClazz = newCls;
            }

            //we now map the function map in
            _applyFuncs(newCls, protoFuncs, true);
            //we could add inherited but that would make debugging harder
            //see http://www.ruzee.com/blog/2008/12/javascript-inheritance-via-prototypes-and-closures on how to do it

            _applyFuncs(newCls, nmsFuncs, false);

            return newCls;
        };

        /**
         * convenience method which basically replaces an existing class
         * with a new one under the same namespace, note all old functionality will be
         * presereced by pushing the original class into an new nampespace
         *
         * @param classNms the namespace for the class, must already be existing
         * @param protoFuncs the new prototype functions which are plugins for the old ones
         * @param overWrite if set to true replaces the old funcs entirely otherwise just does an implicit
         * inheritance with super being remapped
         *
         * TODO do not use this function yet it needs some refinement, it will be interesting later
         * anyway
         */
        _T.pluginClass = function(classNms, protoFuncs, overWrite) {
            var oldClass = _T.fetchNamespace(classNms);
            if (!oldClass) throw new Error("The class namespace " + classNms + " is not existent");

            if (!overWrite) {
                var preserveNMS = classNms + "." + ("" + _T._classReplacementCnt++);
                _T.reserveNamespace(preserveNMS, oldClass);

                return _T.extendClass(classNms, preserveNMS, protoFuncs);
            } else {
                //TODO constructor mapping?
                if (protoFuncs.constructor_) {
                    //TODO needs testing if this works!
                    newCls.prototype.constructor = protoFuncs.constructor_;
                }
                _applyFuncs(oldClass, protoFuncs, true);
            }
        },

        /**
         * Extends a class and puts a singleton instance at the reserved namespace instead
         * of its original class
         *
         * @param {function|String} newCls either a unnamed function which can be assigned later or a namespace
         * @param {function} extendsCls the function class to be extended
         * @param {Object} protoFuncs (Map) an optional map of prototype functions which in case of overwriting a base function get an inherited method
         */
                _T.singletonExtendClass = function(newCls, extendsCls, protoFuncs, nmsFuncs) {
                    return _makeSingleton(_T.extendClass, newCls, extendsCls, protoFuncs, nmsFuncs);
                };

        /**
         * delegation pattern which attached singleton generation
         *
         * @param newCls the new namespace object to be generated as singletoin
         * @param delegateObj the object which has to be delegated
         * @param protoFuncs the prototype functions which are attached on prototype level
         * @param nmsFuncs the functions which are attached on the classes namespace level
         */
        _T.singletonDelegateObj = function(newCls, delegateObj, protoFuncs, nmsFuncs) {
            if (_T._reservedNMS[newCls]) {
                return;
            }
            return _makeSingleton(_T.delegateObj, newCls, delegateObj, protoFuncs, nmsFuncs);
        };

        //since the object is self contained and only
        //can be delegated we can work with real private
        //functions here, the other parts of the
        //system have to emulate them via _ prefixes
        var _makeSingleton = function(ooFunc, newCls, delegateObj, protoFuncs, nmsFuncs) {
            if (_T._reservedNMS[newCls]) {
                return;
            }

            var clazz = ooFunc(newCls + "._mfClazz", delegateObj, protoFuncs, nmsFuncs);
            if (clazz != null) {
                _T.applyToGlobalNamespace(newCls, new clazz());
            }
            _T.fetchNamespace(newCls)["_mfClazz"] = clazz;
        };

        //internal class namespace reservation depending on the type (string or function)
        var _reserveClsNms = function(newCls, protoFuncs) {
            var constr = null;

            if ('undefined' != typeof protoFuncs && null != protoFuncs) {
                constr = ('undefined' != typeof null != protoFuncs['constructor_'] && null != protoFuncs['constructor_']) ? protoFuncs['constructor_'] : function() {
                };
            } else {
                constr = function() {
                };
            }

            if (!_T.reserveNamespace(newCls, constr)) {
                return null;
            }
            newCls = _T.fetchNamespace(newCls);
            return newCls;
        };

        var _applyFuncs = function (newCls, funcs, proto) {
            if (funcs) {
                for (var key in funcs) {
                    //constructor already passed, callSuper already assigned
                    if ('undefined' == typeof key || null == key || key == "_callSuper") {
                        continue;
                    }
                    if (!proto)
                        newCls[key] = funcs[key];
                    else
                        newCls.prototype[key] = funcs[key];
                }
            }
        };

        /**
         * general type assertion routine
         *
         * @param probe the probe to be checked for the correct type
         * @param theType the type to be checked for
         */
        _T.assertType = function(probe, theType) {
            return _T.isString(theType) ? probe == typeof theType : probe instanceof theType;
        };

        /**
         * onload wrapper for chaining the onload cleanly
         * @param func the function which should be added to the load
         * chain (note we cannot rely on return values here, hence jsf.util.chain will fail)
         */
        _T.addOnLoad = function(target, func) {
            var oldonload = (target) ? target.onload : null;
            target.onload = (!oldonload) ? func : function() {
                try {
                    oldonload();
                } catch (e) {
                    throw e;
                } finally {
                    func();
                }
            };
        };

        /**
         * returns the internationalisation setting
         * for the given browser so that
         * we can i18n our messages
         *
         * @returns a map with following entires:
         * <ul>
         *      <li>language: the lowercase language iso code</li>
         *      <li>variant: the uppercase variant iso code</li>
         * </ul>
         * null is returned if the browser fails to determine the language settings
         */
        _T.getLanguage = function(lOverride) {
            var deflt = {language: "en", variant: "UK"}; //default language and variant
            try {
                var lang = lOverride || navigator.language || navigator.browserLanguage;
                if (!lang || lang.length < 2) return deflt;
                return {
                    language: lang.substr(0, 2),
                    variant: (lang.length >= 5) ? lang.substr(3, 5) : null
                };
            } catch(e) {
                return deflt;
            }
        };

        //initial browser detection, we encapsule it in a closure
        //to drop all temporary variables from ram as soon as possible
        (function() {
            /**
             * browser detection code
             * cross ported from dojo 1.2
             *
             * dojos browser detection code is very sophisticated
             * hence we port it over it allows a very fine grained detection of
             * browsers including the version number
             * this however only can work out if the user
             * does not alter the user agent, which they normally dont!
             *
             * the exception is the ie detection which relies on specific quirks in ie
             */
            var n = navigator;
            var dua = n.userAgent,
                    dav = n.appVersion,
                    tv = parseFloat(dav);

            _T.browser = {};
            var d = _T.browser;

            if (dua.indexOf("Opera") >= 0) {
                _T.isOpera = tv;
            }
            if (dua.indexOf("AdobeAIR") >= 0) {
                d.isAIR = 1;
            }
            if (dua.indexOf("BlackBerry") >= 0) {
                d.isBlackBerry = tv;
            }
            d.isKhtml = (dav.indexOf("Konqueror") >= 0) ? tv : 0;
            d.isWebKit = parseFloat(dua.split("WebKit/")[1]) || undefined;
            d.isChrome = parseFloat(dua.split("Chrome/")[1]) || undefined;

            // safari detection derived from:
            //		http://developer.apple.com/internet/safari/faq.html#anchor2
            //		http://developer.apple.com/internet/safari/uamatrix.html
            var index = Math.max(dav.indexOf("WebKit"), dav.indexOf("Safari"), 0);
            if (index && !d.isChrome) {
                // try to grab the explicit Safari version first. If we don't get
                // one, look for less than 419.3 as the indication that we're on something
                // "Safari 2-ish".
                d.isSafari = parseFloat(dav.split("Version/")[1]);
                if (!d.isSafari || parseFloat(dav.substr(index + 7)) <= 419.3) {
                    d.isSafari = 2;
                }
            }

            //>>excludeStart("webkitMobile", kwArgs.webkitMobile);

            if (dua.indexOf("Gecko") >= 0 && !d.isKhtml && !d.isWebKit) {
                d.isMozilla = d.isMoz = tv;
            }
            if (d.isMoz) {
                //We really need to get away from _T. Consider a sane isGecko approach for the future.
                d.isFF = parseFloat(dua.split("Firefox/")[1] || dua.split("Minefield/")[1] || dua.split("Shiretoko/")[1]) || undefined;
            }

            if (document.all && !d.isOpera && !d.isBlackBerry) {
                d.isIE = parseFloat(dav.split("MSIE ")[1]) || undefined;
                d.isIEMobile = parseFloat(dua.split("IEMobile")[1]);
                //In cases where the page has an HTTP header or META tag with
                //X-UA-Compatible, then it is in emulation mode, for a previous
                //version. Make sure isIE reflects the desired version.
                //document.documentMode of 5 means quirks mode.

                /** @namespace document.documentMode */
                if (d.isIE >= 8 && document.documentMode != 5) {
                    d.isIE = document.documentMode;
                }
            }
        })();

    };
}

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages base version
 * (note the base version is basically the en_US) version
 * of all messages
 *
 * @namespace myfaces._impl.i18n.Messages
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages", Object, {

    MSG_TEST:               "Testmessage",

    /*Messages*/
    MSG_DEV_MODE:           "Note, this message is only sent, because project stage is development and no " +
                            "other error listeners are registered.",
    MSG_AFFECTED_CLASS:     "Affected Class:",
    MSG_AFFECTED_METHOD:    "Affected Method:",

    MSG_ERROR_NAME:         "Error Name:",
    MSG_ERROR_MESSAGE:      "Error Name:",

    MSG_ERROR_DESC:         "Error Description:",
    MSG_ERROR_NO:           "Error Number:",
    MSG_ERROR_LINENO:       "Error Line Number:",

    /*Errors and messages*/
    ERR_FORM:               "Sourceform could not be determined, either because element is not attached to a form or we have multiple forms with named elements of the same identifier or name, stopping the ajax processing",
    ERR_VIEWSTATE:          "jsf.viewState: param value not of type form!",
    ERR_TRANSPORT:          "Transport type {0} does not exist",
    ERR_EVT_PASS:           "an event must be passed down (either a an event object null or undefined) ",
    ERR_CONSTRUCT:          "Parts of the response couldn't be retrieved when constructing the event data: {0} ",
    ERR_MALFORMEDXML:       "The server response could not be parsed, the server has returned with a response which is not xml !",
    ERR_SOURCE_FUNC:        "source cannot be a function (probably source and event were not defined or set to null",
    ERR_EV_OR_UNKNOWN:      "An event object or unknown must be passed as second parameter",
    ERR_SOURCE_NOSTR:       "source cannot be a string",
    ERR_SOURCE_DEF_NULL:    "source must be defined or null",

    //_Lang.js
    ERR_MUST_STRING:        "{0}: {1} namespace must be of type String",
    ERR_REF_OR_ID:          "{0}: {1} a reference node or identifier must be provided",
    ERR_PARAM_GENERIC:      "{0}: parameter {1} must be of type {2}",
    ERR_PARAM_STR:          "{0}: {1} param must be of type string",
    ERR_PARAM_STR_RE:       "{0}: {1} param must be of type string or a regular expression",
    ERR_PARAM_MIXMAPS:      "{0}: both a source as well as a destination map must be provided",
    ERR_MUST_BE_PROVIDED:   "{0}: an {1} and a {2} must be provided",
    ERR_MUST_BE_PROVIDED1:  "{0}: {1} must be set",

    ERR_REPLACE_EL:         "replaceElements called while evalNodes is not an array",
    ERR_EMPTY_RESPONSE:     "{0}: The response cannot be null or empty!",
    ERR_ITEM_ID_NOTFOUND:   "{0}: item with identifier {1} could not be found",
    ERR_PPR_IDREQ:          "{0}: Error in PPR Insert, id must be present",
    ERR_PPR_INSERTBEFID:    "{0}: Error in PPR Insert, before id or after id must be present",
    ERR_PPR_INSERTBEFID_1:  "{0}: Error in PPR Insert, before  node of id {1} does not exist in document",
    ERR_PPR_INSERTBEFID_2:  "{0}: Error in PPR Insert, after  node of id {1} does not exist in document",

    ERR_PPR_DELID:          "{0}: Error in delete, id not in xml markup",
    ERR_PPR_UNKNOWNCID:     "{0}:  Unknown Html-Component-ID: {1}",
    ERR_NO_VIEWROOTATTR:    "{0}: Changing of ViewRoot attributes is not supported",
    ERR_NO_HEADATTR:        "{0}: Changing of Head attributes is not supported",
    ERR_RED_URL:            "{0}: Redirect without url"

});

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages german version
 * (note the base version is basically the en_US) version
 * of all messages
 *
 * We use inheritance to overide the default messages with our
 * german one, variants can derive from the german one (like
 * suisse which do not have the emphasized s)
 *
 * By using inheritance we can be sure that we fall back to the default one
 * automatically and that our variants only have to override the parts
 * which have changed from the baseline
 *
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_de", myfaces._impl.i18n.Messages, {

    MSG_TEST:               "Testnachricht",

    /*Messages*/
    MSG_DEV_MODE:           "Sie sehen diese Nachricht, da sie sich gerade im Entwicklungsmodus befinden " +
                            "und sie keine Fehlerbehandlungsfunktionen registriert haben.",

    MSG_AFFECTED_CLASS:     "Klasse:",
    MSG_AFFECTED_METHOD:    "Methode:",

    MSG_ERROR_NAME:         "Fehler Name:",
    MSG_ERROR_MESSAGE:      "Nachricht:",

    MSG_ERROR_DESC:         "Fehlerbeschreibung:",
    MSG_ERROR_NO:           "Fehlernummer:",
    MSG_ERROR_LINENO:       "Zeilennummer:",

    /*Errors and messages*/
    ERR_FORM:                "Das Quellformular konnte nicht gefunden werden. " +
                             "Mögliche Gründe: Sie haben entweder kein formular definiert, oder es kommen mehrere Formulare vor, " +
                             "die alle das auslösende Element mit demselben Namen besitzen. " +
                             "Die Weitere Ajax Ausführung wird gestoppt.",

    ERR_VIEWSTATE:          "jsf.viewState: der Parameter ist not vom Typ form!",

    ERR_TRANSPORT:          "Transport typ {0} existiert nicht",
    ERR_EVT_PASS:           "Ein Event Objekt muss übergeben werden (entweder ein event Objekt oder null oder undefined)",
    ERR_CONSTRUCT:          "Teile des response konnten nicht ermittelt werden während die Event Daten bearbeitet wurden: {0} ",
    ERR_MALFORMEDXML:       "Es gab zwar eine Antwort des Servers, jedoch war diese nicht im erwarteten XML Format. Der Server hat kein valides XML gesendet! Bearbeitung abgebrochen.",
    ERR_SOURCE_FUNC:        "source darf keine Funktion sein",
    ERR_EV_OR_UNKNOWN:      "Ein Ereignis Objekt oder UNKNOWN muss als 2. Parameter übergeben werden",
    ERR_SOURCE_NOSTR:       "source darf kein String sein",
    ERR_SOURCE_DEF_NULL:    "source muss entweder definiert oder null sein",

    //_Lang.js
    ERR_MUST_STRING:        "{0}: {1} namespace muss vom Typ String sein",
    ERR_REF_OR_ID:          "{0}: {1} Ein Referenzknoten oder id muss übergeben werden",
    ERR_PARAM_GENERIC:      "{0}: Paramter {1} muss vom Typ {2} sein",
    ERR_PARAM_STR:          "{0}: Parameter {1} muss vom Typ String sein",
    ERR_PARAM_STR_RE:       "{0}: Parameter {1} muss entweder ein String oder ein Regulärer Ausdruck sein",
    ERR_PARAM_MIXMAPS:      "{0}: both a source as well as a destination map must be provided",
    ERR_MUST_BE_PROVIDED:   "{0}: ein {1} und ein {2} müssen übergeben werden",
    ERR_MUST_BE_PROVIDED1:  "{0}: {1} muss gesetzt sein",

    ERR_REPLACE_EL:         "replaceElements aufgerufen während evalNodes nicht ein Array ist",
    ERR_EMPTY_RESPONSE:     "{0}: Die Antwort darf nicht null oder leer sein!",
    ERR_ITEM_ID_NOTFOUND:   "{0}: Element mit ID {1} konnte nicht gefunden werden",
    ERR_PPR_IDREQ:          "{0}: Fehler im PPR Insert, ID muss gesetzt sein",
    ERR_PPR_INSERTBEFID:    "{0}: Fehler im PPR Insert, before ID oder after ID muss gesetzt sein",
    ERR_PPR_INSERTBEFID_1:  "{0}: Fehler im PPR Insert, before  Knoten mit ID {1} Existiert nicht",
    ERR_PPR_INSERTBEFID_2:  "{0}: Fehler im PPR Insert, after  Knoten mit ID {1} Existiert nicht",

    ERR_PPR_DELID:          "{0}: Fehler im PPR delete, id ist nicht im xml Markup vorhanden",
    ERR_PPR_UNKNOWNCID:     "{0}: Unbekannte Html-Komponenten-ID: {1}",
    ERR_NO_VIEWROOTATTR:    "{0}: Änderung von ViewRoot Attributen ist nicht erlaubt",
    ERR_NO_HEADATTR:        "{0}: Änderung von Head Attributen ist nicht erlaubt",
    ERR_RED_URL:            "{0}: Redirect ohne URL"

});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages base version
 * (note the base version is basically the en_US) version
 * of all messages
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_nl", myfaces._impl.i18n.Messages, {

    MSG_TEST:               "Testbericht",

    /*Messages*/
    MSG_DEV_MODE:           "Opmerking, dit bericht is enkel gestuurd omdat het project stadium develoment is en er geen " +
                            "andere listeners zijn geconfigureerd.",
    MSG_AFFECTED_CLASS:     "Betrokken Klasse:",
    MSG_AFFECTED_METHOD:    "Betrokken Methode:",

    MSG_ERROR_NAME:         "Naam foutbericht:",
    MSG_ERROR_MESSAGE:      "Naam foutbericht:",

    MSG_ERROR_DESC:         "Omschrijving fout:",
    MSG_ERROR_NO:           "Fout nummer:",
    MSG_ERROR_LINENO:       "Fout lijn nummer:",

    /*Errors and messages*/
    ERR_FORM:               "De doel form kon niet bepaald worden, ofwel omdat het element niet tot een form behoort, ofwel omdat er verschillende forms zijn met 'named element' met dezelfde identifier of naam, ajax verwerking is gestopt.",
	ERR_VIEWSTATE:          "jsf.viewState: param waarde is niet van het type form!",
    ERR_TRANSPORT:          "Transport type {0} bestaat niet",
    ERR_EVT_PASS:           "een event moet opgegegevn worden (ofwel een event object null of undefined) ",
    ERR_CONSTRUCT:          "Delen van het antwoord konden niet opgehaald worden bij het aanmaken van de event data: {0} ",
    ERR_MALFORMEDXML:       "Het antwoordt van de server kon niet ontleed worden, de server heeft een antwoord gegeven welke geen xml bevat!",
    ERR_SOURCE_FUNC:        "source kan geen functie zijn (waarschijnlijk zijn source en event niet gedefinieerd of kregen de waarde null)",
    ERR_EV_OR_UNKNOWN:      "Een event object of 'unknown' moet gespecifieerd worden als tweede parameter",
    ERR_SOURCE_NOSTR:       "source kan geen string zijn",
    ERR_SOURCE_DEF_NULL:    "source moet gedefinieerd zijn of null bevatten",

    //_Lang.js
    ERR_MUST_STRING:        "{0}: {1} namespace moet van het type String zijn",
    ERR_REF_OR_ID:          "{0}: {1} een referentie node of identifier moet opgegeven worden",
    ERR_PARAM_GENERIC:      "{0}: parameter {1} moet van het type {2} zijn",
    ERR_PARAM_STR:          "{0}: {1} parameter moet van het type string zijn",
    ERR_PARAM_STR_RE:       "{0}: {1} parameter moet van het type string zijn of een reguliere expressie",
    ERR_PARAM_MIXMAPS:      "{0}: zowel source als destination map moeten opgegeven zijn",
    ERR_MUST_BE_PROVIDED:   "{0}: een {1} en een {2} moeten opgegeven worden",
    ERR_MUST_BE_PROVIDED1:  "{0}: {1} moet gezet zijn",

    ERR_REPLACE_EL:         "replaceElements opgeroepen maar evalNodes is geen array",
    ERR_EMPTY_RESPONSE:     "{0}: Het antwoord kan geen null of leeg zijn!",
    ERR_ITEM_ID_NOTFOUND:   "{0}: item met identifier {1} kan niet gevonden worden",
    ERR_PPR_IDREQ:          "{0}: Fout in PPR Insert, id moet bestaan",
    ERR_PPR_INSERTBEFID:    "{0}: Fout in PPR Insert, before id of after id moet bestaan",
    ERR_PPR_INSERTBEFID_1:  "{0}: Fout in PPR Insert, before node van id {1} bestaat niet in het document",
    ERR_PPR_INSERTBEFID_2:  "{0}: Fout in PPR Insert, after node van id {1} bestaat niet in het document",

    ERR_PPR_DELID:          "{0}: Fout in delete, id is niet in de xml markup",
    ERR_PPR_UNKNOWNCID:     "{0}: Onbekende Html-Component-ID: {1}",
    ERR_NO_VIEWROOTATTR:    "{0}: Wijzigen van ViewRoot attributen is niet ondersteund",
    ERR_NO_HEADATTR:        "{0}: Wijzigen van Head attributen is niet ondersteund",
    ERR_RED_URL:            "{0}: Redirect zonder url"

});

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages spanish version version
 * (note the base version is basically the en_US) version
 * of all messages
 *
 */

myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_es", myfaces._impl.i18n.Messages, {


    MSG_TEST:               "Mensajeprueba",

   /*Messages*/
   MSG_DEV_MODE:           "Aviso. Este mensaje solo se envia porque el 'Project Stage' es 'Development' y no hay otros 'listeners' de errores registrados.",
   MSG_AFFECTED_CLASS:     "Clase Afectada:",
   MSG_AFFECTED_METHOD:    "Mtodo Afectado:",

   MSG_ERROR_NAME:         "Nombre del Error:",
   MSG_ERROR_MESSAGE:      "Nombre del Error:",

   MSG_ERROR_DESC:         "Descripcin del Error:",
   MSG_ERROR_NO:           "Nmero de Error:",
   MSG_ERROR_LINENO:       "Nmero de Lnea del Error:",

   /*Errors and messages*/
   ERR_FORM:               "El formulario de origen no ha podido ser determinado, debido a que el elemento no forma parte de un formulario o hay diversos formularios con elementos usando el mismo nombre o identificador. Parando el procesamiento de Ajax.",
   ERR_VIEWSTATE:          "jsf.viewState: el valor del parmetro no es de tipo 'form'!",
   ERR_TRANSPORT:          "El tipo de transporte {0} no existe",
   ERR_EVT_PASS:           "un evento debe ser transmitido (sea null o no definido)",
   ERR_CONSTRUCT:          "Partes de la respuesta no pudieron ser recuperadas cuando construyendo los datos del evento: {0} ",
   ERR_MALFORMEDXML:       "La respuesta del servidor no ha podido ser interpretada. El servidor ha devuelto una respuesta que no es xml !",
   ERR_SOURCE_FUNC:        "el origen no puede ser una funcin (probablemente 'source' y evento no han sido definidos o son 'null'",
   ERR_EV_OR_UNKNOWN:      "Un objeto de tipo evento o desconocido debe ser pasado como segundo parmetro",
   ERR_SOURCE_NOSTR:       "el origen no puede ser 'string'",
   ERR_SOURCE_DEF_NULL:    "el origen debe haber sido definido o ser 'null'",

   //_Lang.js
   ERR_MUST_STRING:        "{0}: {1} namespace debe ser de tipo String",
   ERR_REF_OR_ID:          "{0}: {1} una referencia a un nodo o identificador tiene que ser pasada",
   ERR_PARAM_GENERIC:      "{0}: el parmetro {1} tiene que ser de tipo {2}",
   ERR_PARAM_STR:          "{0}: el parmetro {1} tiene que ser de tipo string",
   ERR_PARAM_STR_RE:       "{0}: el parmetro {1} tiene que ser de tipo string o una expresin regular",
   ERR_PARAM_MIXMAPS:      "{0}: han de ser pasados tanto un origen como un destino",
   ERR_MUST_BE_PROVIDED:   "{0}: {1} y {2} deben ser pasados",
   ERR_MUST_BE_PROVIDED1:  "{0}: {1} debe estar definido",

   ERR_REPLACE_EL:         "replaceElements invocado mientras que evalNodes no es un an array",
   ERR_EMPTY_RESPONSE:     "{0}: La respuesta no puede ser de tipo 'null' o vaca!",
   ERR_ITEM_ID_NOTFOUND:   "{0}: el elemento con identificador {1} no ha sido encontrado",
   ERR_PPR_IDREQ:          "{0}: Error en PPR Insert, 'id' debe estar presente",
   ERR_PPR_INSERTBEFID:    "{0}: Error in PPR Insert, antes de 'id' o despus de 'id' deben estar presentes",
   ERR_PPR_INSERTBEFID_1:  "{0}: Error in PPR Insert, antes de nodo con id {1} no existe en el documento",
   ERR_PPR_INSERTBEFID_2:  "{0}: Error in PPR Insert, despus de nodo con id {1} no existe en el documento",

   ERR_PPR_DELID:          "{0}: Error durante borrado, id no presente en xml",
   ERR_PPR_UNKNOWNCID:     "{0}:  Desconocido Html-Component-ID: {1}",
   ERR_NO_VIEWROOTATTR:    "{0}: El cambio de atributos de ViewRoot attributes no es posible",
   ERR_NO_HEADATTR:        "{0}: El cambio de los atributos de Head attributes no es posible",
   ERR_RED_URL:            "{0}: Redireccin sin url"

});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * System messages french version version
 * (note the base version is basically the en) version
 * of all messages
 *
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_fr", myfaces._impl.i18n.Messages, {
    MSG_TEST:               "MessageTest FR",

       /*Messages*/
       MSG_DEV_MODE:           "Note : ce message n'est envoyé que parce que le projet est au stade de développement et " +
                               "qu'aucun autre listener d'erreurs n'est enregistré.",
       MSG_AFFECTED_CLASS:     "Classe affectée : ",
       MSG_AFFECTED_METHOD:    "Méthode affectée : ",

       MSG_ERROR_NAME:         "Nom de l'erreur : ",
       MSG_ERROR_MESSAGE:      "Nom de l'erreur : ",

       MSG_ERROR_DESC:         "Description de l'erreur : ",
       MSG_ERROR_NO:           "Numéro de l'erreur : ",
       MSG_ERROR_LINENO:       "Erreur à la ligne : ",

       /*Errors and messages*/
       ERR_FORM:               "Le formulaire source n'a pas pu être déterminé, soit parce que l'élément n'est rattaché à aucun formulaire, soit parce qu'ils y a plusieurs formulaires contenant des éléments avec le même nom ou identifiant. Arrêt du traitement AJAX",
       ERR_VIEWSTATE:          "jsf.viewState: La valeur de 'param' n'est pas de type 'form' !",
       ERR_TRANSPORT:          "Le type de tansport {0} n'existe pas",
       ERR_EVT_PASS:           "Un évènement doit être transmis (soit un objet évènement, soit null ou undefined) ",
       ERR_CONSTRUCT:          "Des éléments de la réponse n'ont pu être récupérés lors de la construction des données de l'évènement : {0} ",
       ERR_MALFORMEDXML:       "La réponse du serveur n'a pas pu être analysée : le serveur n'a pas renvoyé une réponse en xml !",
       ERR_SOURCE_FUNC:        "La source ne peut pas être une fonction (Il est probable que 'source' et 'event' n'ont pas été définis ou mis à null",
       ERR_EV_OR_UNKNOWN:      "Le second paramètre doit être un objet évènement ou 'unknown' ",
       ERR_SOURCE_NOSTR:       "La source ne peut pas être de type String",
       ERR_SOURCE_DEF_NULL:    "La source doit être définie ou égale à null",

       //_Lang.js
       ERR_MUST_STRING:        "{0}: Le namespace {1} doit être de type String",
       ERR_REF_OR_ID:          "{0}: {1} un noeud de référence ou un identifiant doit être passé",
       ERR_PARAM_GENERIC:      "{0}: Le paramètre {1} doit être de type {2}",
       ERR_PARAM_STR:          "{0}: Le paramètre {1} doit être de type String",
       ERR_PARAM_STR_RE:       "{0}: Le paramètre {1} doit être de type String ou être une expression régulière",
       ERR_PARAM_MIXMAPS:      "{0}: Un Map de source et un Map de destination doivent être passés",
       ERR_MUST_BE_PROVIDED:   "{0}: un(e) {1} et un(e) {2} doivent être passés",
       ERR_MUST_BE_PROVIDED1:  "{0}: {1} doit être défini",

       ERR_REPLACE_EL:         "replaceElements a été appelé alors que evalNodes n'est pas un tableau",
       ERR_EMPTY_RESPONSE:     "{0}: La réponse ne peut pas être nulle ou vide !",
       ERR_ITEM_ID_NOTFOUND:   "{0}: l'élément portant l'identifiant {1} n'a pas pu être trouvé",
       ERR_PPR_IDREQ:          "{0}: Erreur lors de l'insertion PPR, l'id doit être présent",
       ERR_PPR_INSERTBEFID:    "{0}: Erreur lors de l'insertion PPR, 'before id' ou 'after id' doivent être présents",
       ERR_PPR_INSERTBEFID_1:  "{0}: Erreur lors de l'insertion PPR, le noeud before de l'id {1} n'existe pas dans le document",
       ERR_PPR_INSERTBEFID_2:  "{0}: Erreur lors de l'insertion PPR, le noeud after  de l'id {1} n'existe pas dans le document",

       ERR_PPR_DELID:          "{0}: Erreur lors de la suppression, l'id n'est pas présent dans le xml",
       ERR_PPR_UNKNOWNCID:     "{0}:  Html-Component-ID inconnu : {1}",
       ERR_NO_VIEWROOTATTR:    "{0}: Le changement d'attributs dans ViewRoot n'est pas supporté",
       ERR_NO_HEADATTR:        "{0}: Le changement d'attributs dans Head n'est pas supporté",
       ERR_RED_URL:            "{0}: Redirection sans url"




});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages italian version version
 * (note the base version is basically the en_US) version
 * of all messages
 *
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_it", myfaces._impl.i18n.Messages, {
    /*Messages*/
    MSG_DEV_MODE:           "Questo messaggio ? stato inviato esclusivamente perch? il progetto ? in development stage e nessun altro listener ? stato registrato.",
    MSG_AFFECTED_CLASS:     "Classi coinvolte:",
    MSG_AFFECTED_METHOD:    "Metodi coinvolti:",

    MSG_ERROR_NAME:         "Nome dell'errore:",
    MSG_ERROR_MESSAGE:      "Nome dell'errore:",

    MSG_ERROR_DESC:         "Descrizione dell'errore:",
    MSG_ERROR_NO:           "Numero errore:",
    MSG_ERROR_LINENO:       "Numero di riga dell'errore:",

    /*Errors and messages*/
    ERR_FORM:               "Il Sourceform non puo' essere determinato a causa di una delle seguenti ragioni: l'elemento non e' agganciato ad un form oppure sono presenti pi? form con elementi con lo stesso nome, il che blocca l'elaborazione ajax",
    ERR_VIEWSTATE:          "jsf.viewState: il valore del parametro non ? di tipo form!",
    ERR_TRANSPORT:          "Il transport type {0} non esiste",
    ERR_EVT_PASS:           "? necessario passare un evento (sono accettati anche gli event object null oppure undefined) ",
    ERR_CONSTRUCT:          "Durante la costruzione dell' event data: {0} non ? stato possibile acquisire alcune parti della response ",
    ERR_MALFORMEDXML:       "Il formato della risposta del server non era xml, non ? stato quindi possibile effettuarne il parsing!",
    ERR_SOURCE_FUNC:        "source non puo' essere una funzione (probabilmente source and event non erano stati definiti o sono null",
    ERR_EV_OR_UNKNOWN:      "Come secondo parametro bisogna passare un event object oppure unknown",
    ERR_SOURCE_NOSTR:       "source non pu essere una stringa di testo",
    ERR_SOURCE_DEF_NULL:    "source deve essere definito oppure  null",

    //_Lang.js
    ERR_MUST_STRING:        "{0}: {1} namespace deve essere di tipo String",
    ERR_REF_OR_ID:          "{0}: {1} un reference node oppure un identificatore deve essere fornito",
    ERR_PARAM_GENERIC:      "{0}: il parametro {1} deve essere di tipo {2}",
    ERR_PARAM_STR:          "{0}: {1} parametro deve essere di tipo String",
    ERR_PARAM_STR_RE:       "{0}: {1} parametro deve essere di tipo String oppure una regular expression",
    ERR_PARAM_MIXMAPS:      "{0}: ? necessario specificare sia  source che destination map",
    ERR_MUST_BE_PROVIDED:   "{0}: ? necessario specificare sia {1} che {2} ",
    ERR_MUST_BE_PROVIDED1:  "{0}: {1} deve essere settato",

    ERR_REPLACE_EL:         "replaceElements chiamato metre evalNodes non ? un array",
    ERR_EMPTY_RESPONSE:     "{0}: La response non puo' essere nulla o vuota!",
    ERR_ITEM_ID_NOTFOUND:   "{0}: non ? stato trovato alcun item con identificativo {1}",
    ERR_PPR_IDREQ:          "{0}: Errore durante la PPR Insert, l' id deve essere specificato",
    ERR_PPR_INSERTBEFID:    "{0}: Errore durante la PPR Insert, before id o after id deve essere specificato",
    ERR_PPR_INSERTBEFID_1:  "{0}: Errore durante la PPR Insert, before node of id {1} non esiste nel document",
    ERR_PPR_INSERTBEFID_2:  "{0}: Errore durante la PPR Insert, after  node of id {1} non esiste nel in document",

    ERR_PPR_DELID:          "{0}: Errore durante la delete, l'id non e' nella forma di un markup xml",
    ERR_PPR_UNKNOWNCID:     "{0}:   Html-Component-ID: {1} sconosciuto",
    ERR_NO_VIEWROOTATTR:    "{0}: La modifica degli attributi del ViewRoot non ? supportata",
    ERR_NO_HEADATTR:        "{0}: La modifica degli attributi di Head non ? supportata",
    ERR_RED_URL:            "{0}: Redirect senza url"
});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * System messages kyrillic/russian version
 * (note the base version is basically the en_US) version
 * of all messages
 *
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.i18n.Messages_ru", myfaces._impl.i18n.Messages, {

    MSG_TEST:               "Те?товоеСообщение",

    /*Messages*/
    MSG_DEV_MODE:           "Это ?ообщение выдано, потому что 'project stage' было при?оено значение 'development', и никаких" +
            "других error listeners зареги?трировано не было.",
    MSG_AFFECTED_CLASS:     "Задей?твованный кла??:",
    MSG_AFFECTED_METHOD:    "Задей?твованный метод:",

    MSG_ERROR_NAME:         "Им? ошибки:",
    MSG_ERROR_MESSAGE:      "Им? ошибки:",

    MSG_ERROR_DESC:         "Опи?ание ошибки:",
    MSG_ERROR_NO:           "?омер ошибки:",
    MSG_ERROR_LINENO:       "?омер ?троки ошибки:",

    /*Errors and messages*/
    ERR_FORM:               "Sourceform не найдена, потому что ?лемент не находит?? внутри <form>, либо были найдены ?лементы <form> ? рдинаковым именем или идентификатором. Обработка ajax о?тановлена",
    ERR_VIEWSTATE:          "jsf.viewState: Параметру при?воено значение, не ?вл?ющее?? ?лементом <form>!",
    ERR_TRANSPORT:          "?е?уще?твующий тип тран?порта {0}",
    ERR_EVT_PASS:           "Параметр event необходим, и не может быть null или undefined",
    ERR_CONSTRUCT:          "Ча?ть ответа не удало?ь прочитать при ?оздании данных ?обыти?: {0} ",
    ERR_MALFORMEDXML:       "Ответ ?ервера не может быть обработан, он не в формате xml !",
    ERR_SOURCE_FUNC:        "source не может быть функцией (возможно, дл? source и event не были даны значени?",
    ERR_EV_OR_UNKNOWN:      "Объект event или unknown должен быть в?торым параметром",
    ERR_SOURCE_NOSTR:       "source не может быть типа string",
    ERR_SOURCE_DEF_NULL:    "source должно быть при?воено значение или null",

    //_Lang.js
    ERR_MUST_STRING:        "{0}: {1} namespace должно быть типа String",
    ERR_REF_OR_ID:          "{0}: {1} a С?ылочный узел (reference node) или идентификатор необходимы",
    ERR_PARAM_GENERIC:      "{0}: параметр {1} должен быть типа {2}",
    ERR_PARAM_STR:          "{0}: {1} параметр должен быть типа string",
    ERR_PARAM_STR_RE:       "{0}: {1} параметр должен быть типа string string или regular expression",
    ERR_PARAM_MIXMAPS:      "{0}: source b destination map необходимы",
    ERR_MUST_BE_PROVIDED:   "{0}: {1} и {2} необходимы",
    ERR_MUST_BE_PROVIDED1:  "{0}: {1} должно быть при?воено значение",

    ERR_REPLACE_EL:         "replaceElements вызвана, ? evalNodes, не ?вл?ющим?? ма??ивом",
    ERR_EMPTY_RESPONSE:     "{0}: Ответ не может бвть null или пу?тым!",
    ERR_ITEM_ID_NOTFOUND:   "{0}: Элемент ? идентификатором {1} не найден",
    ERR_PPR_IDREQ:          "{0}: Ошибка в PPR Insert, id необходим",
    ERR_PPR_INSERTBEFID:    "{0}: Ошибка в PPR Insert, before id или after id необходимы",
    ERR_PPR_INSERTBEFID_1:  "{0}: Ошибка в PPR Insert, before node c id {1} не найден в документе",
    ERR_PPR_INSERTBEFID_2:  "{0}: Ошибка в PPR Insert, after node ? id {1} не найден в документе",

    ERR_PPR_DELID:          "{0}: Ошибка в удалении, id не найден в xml документе",
    ERR_PPR_UNKNOWNCID:     "{0}: ?еопознанный Html-Component-ID: {1}",
    ERR_NO_VIEWROOTATTR:    "{0}: Изменение атрибутов ViewRoot не преду?мотрено",
    ERR_NO_HEADATTR:        "{0}: Изменение атрибутов Head не преду?мотрено",
    ERR_RED_URL:            "{0}: Перенаправление (Redirect) без url"

});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/*
 theoretically we could save some code
 by
 defining the parent object as
 var parent = new Object();
 parent.prototype = new myfaces._impl.core._Runtime();
 extendClass(function () {
 }, parent , {
 But for now we are not doing it the little bit of saved
 space is not worth the loss of readability
 */

//Intellij Warnings settings
/** @namespace myfaces._impl._util._Lang */
/** @namespace window.console */
myfaces._impl.core._Runtime.singletonDelegateObj("myfaces._impl._util._Lang", myfaces._impl.core._Runtime, {

    _processedExceptions: {},

    _installedLocale: null,

    /**
     * returns a given localized message upon a given key
     * basic java log like templating functionality is included
     *
     * @param {String} key the key for the message
     * @param {String} optional default message if none was found
     *
     * Additionally you can pass additional arguments, which are used
     * in the same way java log templates use the params
     *
     * @param key
     */
    getMessage: function(key, defaultMessage /*,vararg templateParams*/) {
        if(!this._installedLocale) {
            //we first try to install language and variant, if that one fails
            //we try to install the language only, and if that one fails
            //we install the base messages
            this.initLocale();
        }

        var msg = this._installedLocale[key] || defaultMessage || key + " - undefined message";
        for(var cnt = 2; cnt < arguments.length; cnt++) {
          msg = msg.replace(new RegExp(["\\{",cnt-2,"\\}"].join(""),"g"),new String(arguments[cnt]));   
        }
        return msg;
    },

    /**
     * (re)inits the currently installed
     * messages so that after loading the main scripts
     * a new locale can be installed optionally
     * to our i18n subsystem
     *
     * @param newLocale locale override 
     */
    initLocale: function(newLocale) {
        if(newLocale) {
            this._installedLocale = new newLocale();
            return;
        }
        var language_Variant = this._callDelegate("getLanguage", this._callDelegate("getGlobalConfig","locale")); 
        var langStr = language_Variant ? language_Variant.language:"";
        var variantStr = language_Variant ? [language_Variant.language,"_",language_Variant.variant||""].join(""):"";

        var i18nRoot = myfaces._impl.i18n;
        var i18nHolder = i18nRoot["Messages_"+variantStr] ||
                         i18nRoot["Messages_"+langStr]    ||
                         i18nRoot.Messages;

        this._installedLocale = new i18nHolder();
    },


    isExceptionProcessed: function(e) {
        return !! this._processedExceptions[e.toString()];
    },

    setExceptionProcessed: function(e) {
        this._processedExceptions[e.toString()] = true;
    },

    clearExceptionProcessed: function() {
        //ie again
        for (var key in this._processedExceptions) {
            this._processedExceptions[key] = null;
        }
        this._processedExceptions = {};
    },

    fetchNamespace : function(namespace) {
        if (!namespace || !this.isString(namespace)) {
            throw Error(this.getMessage("ERR_MUST_STRING",null,"_Lang.fetchNamespace","namespace"));
        }
        return this._callDelegate("fetchNamespace", namespace);
    },

    reserveNamespace : function(namespace) {
        if (!this.isString(namespace)) {
            throw Error(this.getMessage("ERR_MUST_STRING",null,"_Lang.reserveNamespace", "namespace"));
        }
        return this._callDelegate("reserveNamespace", namespace);
    },

    globalEval : function(code) {
        if (!this.isString(code)) {
            throw Error(this.getMessage("ERR_MUST_STRING",null,"_Lang.globalEval", "code"));
        }
        return this._callDelegate("globalEval", code);
    },


    /**
     * determines the correct event depending
     * on the browsers state
     *
     * @param evt incoming event object (note not all browsers
     * have this)
     *
     * @return an event object no matter what is incoming
     */
    getEvent: function(evt) {
        evt = (!evt) ? window.event || {} : evt;
        return evt;
    },

    /**
     * cross port from the dojo lib
     * browser save event resolution
     * @param evt the event object
     * (with a fallback for ie events if none is present)
     */
    getEventTarget: function(evt) {
        //ie6 and 7 fallback
        evt = this.getEvent(evt);
        /**
         * evt source is defined in the jsf events
         * seems like some component authors use our code
         * so we add it here see also
         * https://issues.apache.org/jira/browse/MYFACES-2458
         * not entirely a bug but makes sense to add this
         * behavior. I dont use it that way but nevertheless it
         * does not break anything so why not
         * */
        var t = evt.srcElement || evt.target || evt.source || null;
        while ((t) && (t.nodeType != 1)) {
            t = t.parentNode;
        }
        return t;
    },

    /**
     * consume event in a browser independend manner
     * @param event the event which should not be propagated anymore
     */
    consumeEvent: function(event) {
        //w3c model vs ie model again
        event = event || window.event;
        (event.stopPropagation) ? event.stopPropagation() : event.cancelBubble = true;
    },

    /**
     * equalsIgnoreCase, case insensitive comparison of two strings
     *
     * @param source
     * @param destination
     */
    equalsIgnoreCase: function(source, destination) {
        //either both are not set or null
        if (!source && !destination) {
            return true;
        }
        //source or dest is set while the other is not
        if (!source || !destination) return false;

        //in any other case we do a strong string comparison
        return source.toLowerCase() === destination.toLowerCase();
    },

    /**
     * escapes a strings special chars (crossported from dojo 1.3+)
     *
     * @param str the string
     *
     * @param except a set of exceptions
     */
    escapeString: function(/*String*/str, /*String?*/except) {
        //	summary:
        //		Adds escape sequences for special characters in regular expressions
        // except:
        //		a String with special characters to be left unescaped

        return str.replace(/([\.$?*|:{}\(\)\[\]\\\/\+^])/g, function(ch) {
            if (except && except.indexOf(ch) != -1) {
                return ch;
            }
            return "\\" + ch;
        }); // String
    },

    /**
     @see this._RT.extendClass
     */
    /*extendClass : function(newClass, extendsClass, functionMap, inherited) {
     return this._RT.extendClass(newClass, extendsClass, functionMap, inherited);
     },*/

    //core namespacing and inheritance done, now to the language extensions

    /**
     * Save document.getElementById (this code was ported over from dojo)
     * the idea is that either a string or domNode can be passed
     * @param {Object} reference the reference which has to be byIded
     */
    byId : function(/*object*/ reference) {
        if (!reference) {
            throw Error(this.getMessage("ERR_REF_OR_ID",null,"_Lang.byId","reference"));
        }
        return (this.isString(reference)) ? document.getElementById(reference) : reference;
    },

    /**
     * backported from dojo
     * Converts an array-like object (i.e. arguments, DOMCollection) to an
     array. Returns a new Array with the elements of obj.
     * @param {Object} obj the object to "arrayify". We expect the object to have, at a
     minimum, a length property which corresponds to integer-indexed
     properties.
     * @param {int} offset the location in obj to start iterating from. Defaults to 0.
     Optional.
     * @param {Array} packArr An array to pack with the properties of obj. If provided,
     properties in obj are appended at the end of startWith and
     startWith is the returned array.
     */
    /*_toArray : function(obj, offset, packArr) {
     //	summary:
     //		Converts an array-like object (i.e. arguments, DOMCollection) to an
     //		array. Returns a new Array with the elements of obj.
     //	obj:
     //		the object to "arrayify". We expect the object to have, at a
     //		minimum, a length property which corresponds to integer-indexed
     //		properties.
     //	offset:
     //		the location in obj to start iterating from. Defaults to 0.
     //		Optional.
     //	startWith:
     //		An array to pack with the properties of obj. If provided,
     //		properties in obj are appended at the end of startWith and
     //		startWith is the returned array.
     var arr = packArr || [];
     //TODO add splicing here

     for (var x = offset || 0; x < obj.length; x++) {
     arr.push(obj[x]);
     }
     return arr; // Array
     }, */

    /**
     * Helper function to provide a trim with a given splitter regular expression
     * @param {|String|} it the string to be trimmed
     * @param {|RegExp|} splitter the splitter regular expressiion
     *
     * FIXME is this still used?
     */
    trimStringInternal : function(it, splitter) {
        return this.strToArray(it, splitter).join(splitter);
    },

    /**
     * String to array function performs a string to array transformation
     * @param {String} it the string which has to be changed into an array
     * @param {RegExp} splitter our splitter reglar expression
     * @return an array of the splitted string
     */
    strToArray : function(/*string*/ it, /*regexp*/ splitter) {
        //	summary:
        //		Return true if it is a String

        if (!this.isString(it)) {
            throw Error(this.getMessage("ERR_PARAM_STR",null, "myfaces._impl._util._Lang.strToArray", "it"));
        }
        if (!splitter) {
            throw Error(this.getMessage("ERR_PARAM_STR_RE",null, "myfaces._impl._util._Lang.strToArray", "splitter"));
        }
        var retArr = it.split(splitter);
        var len = retArr.length;
        for (var cnt = 0; cnt < len; cnt++) {
            retArr[cnt] = this.trim(retArr[cnt]);
        }
        return retArr;
    },

    /**
     * hyperfast trim
     * http://blog.stevenlevithan.com/archives/faster-trim-javascript
     * crossported from dojo
     */
    trim : function(/*string*/ str) {
        if (!this.isString(str)) {
            throw Error(this.getMessage("ERR_PARAM_STR",null,"_Lang.trim", "str"));
        }
        str = str.replace(/^\s\s*/, '');
        var ws = /\s/;
        var i = str.length;
        while (ws.test(str.charAt(--i)));
        return str.slice(0, i + 1);
    },

    /**
     * Backported from dojo
     * a failsafe string determination method
     * (since in javascript String != "" typeof alone fails!)
     * @param it {|Object|} the object to be checked for being a string
     * @return true in case of being a string false otherwise
     */
    isString: function(/*anything*/ it) {
        //	summary:
        //		Return true if it is a String
        return !!arguments.length && it != null && (typeof it == "string" || it instanceof String); // Boolean
    },
    /**
     * hitch backported from dojo
     * hitch allows to assign a function to a dedicated scope
     * this is helpful in situations when function reassignments
     * can happen
     * (notably happens often in lazy xhr code)
     *
     * @param {Function} scope of the function to be executed in
     * @param {Function} method to be executed
     *
     * @return whatevery the executed method returns
     */
    hitch : function(/*Object*/scope, /*Function|String*/method /*,...*/) {
        //	summary:
        //		Returns a function that will only ever execute in the a given scope.
        //		This allows for easy use of object member functions
        //		in callbacks and other places in which the "this" keyword may
        //		otherwise not reference the expected scope.
        //		Any number of default positional arguments may be passed as parameters
        //		beyond "method".
        //		Each of these values will be used to "placehold" (similar to curry)
        //		for the hitched function.
        //	scope:
        //		The scope to use when method executes. If method is a string,
        //		scope is also the object containing method.
        //	method:
        //		A function to be hitched to scope, or the name of the method in
        //		scope to be hitched.
        //	example:
        //	|	myfaces._impl._util._Lang.hitch(foo, "bar")();
        //		runs foo.bar() in the scope of foo
        //	example:
        //	|	myfaces._impl._util._Lang.hitch(foo, myFunction);
        //		returns a function that runs myFunction in the scope of foo
        if (arguments.length > 2) {
            return this._hitchArgs._hitchArgs.apply(this._hitchArgs, arguments); // Function
        }
        if (!method) {
            method = scope;
            scope = null;
        }
        if (this.isString(method)) {
            scope = scope || window || function() {
            };
            /*since we do not have dojo global*/
            if (!scope[method]) {
                throw(['myfaces._impl._util._Lang: scope["', method, '"] is null (scope="', scope, '")'].join(''));
            }
            return function() {
                return scope[method].apply(scope, arguments || []);
            }; // Function
        }
        return !scope ? method : function() {
            return method.apply(scope, arguments || []);
        }; // Function
    }
    ,

    _hitchArgs : function(scope, method /*,...*/) {
        var pre = this.objToArray(arguments, 2);
        var named = this.isString(method);
        return function() {
            // array-fy arguments
            var args = this.objToArray(arguments);
            // locate our method
            var f = named ? (scope || this.global)[method] : method;
            // invoke with collected args
            return f && f.apply(scope || this, pre.concat(args)); // mixed
        }; // Function
    }
    ,

    /**
     * Helper function to merge two maps
     * into one
     * @param {|Object|} dest the destination map
     * @param {|Object|} src the source map
     * @param {|boolean|} overwrite if set to true the destination is overwritten if the keys exist in both maps
     **/
    mixMaps : function(dest, src, overwrite, blockFilter) {
        if (!dest || !src) {
            throw Error(this.getMessage("ERR_PARAM_MIXMAPS",null,"_Lang.mixMaps"));
        }

        /**
         * mixing code depending on the state of dest and the overwrite param
         */
        var ret = {};
        var keyIdx = {};
        var key = null;
        var _undef = "undefined";
        for (key in src) {
            if(blockFilter && blockFilter[key]) {
                continue;
            }
            /**
             *we always overwrite dest with source
             *unless overWrite is not set or source does not exist
             *but also only if dest exists otherwise source still is taken
             */
            if (!overwrite) {
                /**
                 *we use exists instead of booleans because we cannot rely
                 *on all values being non boolean, we would need an elvis
                 *operator in javascript to shorten this :-(
                 */
                ret[key] = (_undef != typeof dest[key]) ? dest[key] : src[key];
            } else {
                ret[key] = (_undef != typeof src[key]) ? src[key] : dest[key];
            }
            keyIdx[key] = true;
        }
        for (key in dest) {
            /*if result.key does not exist we push in dest.key*/
            ret[key] = (_undef != typeof ret[key]) ? ret[key] : dest[key];
        }
        return ret;
    }
    ,

    /**
     * checks if an array contains an element
     * @param {Array} arr   array
     * @param {String} str string to check for
     */
    contains : function(arr, str) {
        if (!arr || !str) {
            throw Error(this.getMessage("ERR_MUST_BE_PROVIDED",null,"_Lang.contains", "arr {array}", "str {string}"));
        }

        for (var cnt = 0; cnt < arr.length; cnt++) {
            if (arr[cnt] == str) {
                return true;
            }
        }
        return false;
    }
    ,


    arrToMap: function(arr, offset) {
        var ret = new Array(arr.length);
        var len = arr.length;
        offset = (offset) ? offset : 0;

        for (var cnt = 0; cnt < len; cnt++) {
            ret[arr[cnt]] = cnt + offset;
        }

        return ret;
    },

    /**
     * Concatenates an array to a string
     * @param {Array} arr the array to be concatenated
     * @param {String} delimiter the concatenation delimiter if none is set \n is used
     *
     * @return the concatenated array, one special behavior to enable j4fry compatibility has been added
     * if no delimiter is used the [entryNumber]+entry is generated for a single entry
     * TODO check if this is still needed it is somewhat outside of the scope of the function
     * and functionality wise dirty
     */
    arrToString : function(/*String or array*/ arr, /*string*/ delimiter) {
        if (!arr) {
            throw Error(this.getMessage("ERR_MUST_BE_PROVIDED1",null, "arr {array}"));
        }
        if (this.isString(arr)) {
            return arr;
        }

        delimiter = delimiter || "\n";
        return arr.join(delimiter);
    }
    ,


    objToArray: function(obj, offset, pack) {
        if (!obj) {
            return null;
        }
        //since offset is numeric we cannot use the shortcut due to 0 being false
        var finalOffset = ('undefined' != typeof offset || null != offset) ? offset : 0;
        var finalPack = pack || [];
        try {
            return finalPack.concat(Array.prototype.slice.call(obj, finalOffset));
        } catch (e) {
            //ie8 (again as only browser) delivers for css 3 selectors a non convertible object
            //we have to do it the hard way
            //ie8 seems generally a little bit strange in its behavior some
            //objects break the function is everything methodology of javascript
            //and do not implement apply call, or are pseudo arrays which cannot
            //be sliced
            for (var cnt = finalOffset; cnt < obj.length; cnt++) {
                finalPack.push(obj[cnt]);
            }
            return finalPack;
        }

    }
    ,

    /**
     * foreach implementation utilizing the
     * ECMAScript wherever possible
     * with added functionality
     *
     * @param arr the array to filter
     * @param func the closure to apply the function to, with the syntax defined by the ecmascript functionality
     * function (element<,key, array>)
     * @param startPos (optional) the starting position
     * @param scope (optional) the scope to apply the closure to
     */
    arrForEach: function(arr, func /*startPos, scope*/) {
        try {
            var startPos = Number(arguments[2]) || 0;
            var thisObj = arguments[3];

            //check for an existing foreach mapping on array prototypes
            if (Array.prototype.forEach) {
                (startPos) ? arr.slice(startPos).forEach(func, thisObj) : arr.forEach(func, thisObj);
            } else {
                startPos = (startPos < 0) ? Math.ceil(startPos) : Math.floor(startPos);
                if (typeof func != "function") {
                    throw new TypeError();
                }
                for (var cnt = 0; cnt < arr.length; cnt++) {
                    if (thisObj) {
                        func.call(thisObj, arr[cnt], cnt, arr);
                    } else {
                        func(arr[cnt], cnt, arr);
                    }
                }
            }
        } finally {
            func = null;
        }
    }
    ,


    /**
     * foreach implementation utilizing the
     * ECMAScript wherever possible
     * with added functionality
     *
     * @param arr the array to filter
     * @param func the closure to apply the function to, with the syntax defined by the ecmascript functionality
     * function (element<,key, array>)
     * @param startPos (optional) the starting position
     * @param scope (optional) the scope to apply the closure to
     *
     */
    arrFilter: function(arr, func /*startPos, scope*/) {
        try {
            var startPos = Number(arguments[2]) || 0;
            var thisObj = arguments[3];

            //check for an existing foreach mapping on array prototypes
            if (Array.prototype.filter) {
                return ((startPos) ? arr.slice(startPos).filter(func, thisObj) : arr.filter(func, thisObj));
            } else {
                if (typeof func != "function") {
                    throw new TypeError();
                }
                var ret = [];
                startPos = (startPos < 0) ? Math.ceil(startPos) : Math.floor(startPos);

                for (var cnt = startPos; cnt < arr.length; cnt++) {
                    if (thisObj) {
                        var elem = arr[cnt];
                        if (func.call(thisObj, elem, cnt, arr)) ret.push(elem);
                    } else {
                        var elem = arr[cnt];
                        if (func(arr[cnt], cnt, arr)) ret.push(elem);
                    }
                }
            }
        } finally {
            func = null;
        }
    }
    ,

    /**
     * adds a EcmaScript optimized indexOf to our mix,
     * checks for the presence of an indexOf functionality
     * and applies it, otherwise uses a fallback to the hold
     * loop method to determine the index
     *
     * @param arr the array
     * @param element the index to search for
     */
    arrIndexOf: function(arr, element /*fromIndex*/) {
        if (!arr) return -1;
        var pos = Number(arguments[2]) || 0;

        if (Array.prototype.indexOf) {
            return arr.indexOf(element, pos);
        }
        //var cnt = this._space;
        var len = arr.length;
        pos = (pos < 0) ? Math.ceil(pos) : Math.floor(pos);

        //if negative then it is taken from as offset from the length of the array
        if (pos < 0) {
            pos += len;
        }
        while (pos < len && arr[pos] !== element) {
            pos++;
        }
        return (pos < len) ? pos : -1;
    }
    ,


    /**
     * helper to automatically apply a delivered arguments map or array
     * to its destination which has a field "_"<key> and a full field
     *
     * @param dest the destination object
     * @param args the arguments array or map
     * @param argNames the argument names to be transferred
     */
    applyArgs: function(dest, args, argNames) {
        var _undef = 'undefined';
        if (argNames) {
            for (var cnt = 0; cnt < args.length; cnt++) {
                //dest can be null or 0 hence no shortcut
                if (_undef != typeof dest["_" + argNames[cnt]]) {
                    dest["_" + argNames[cnt]] = args[cnt];
                }
                if (_undef != typeof dest[ argNames[cnt]]) {
                    dest[argNames[cnt]] = args[cnt];
                }
            }
        } else {
            for (var key in args) {
                if (_undef != typeof dest["_" + key]) {
                    dest["_" + key] = args[key];
                }
                if (_undef != typeof dest[key]) {
                    dest[key] = args[key];
                }
            }
        }
    }
    ,
    /**
     * creates a standardized error message which can be reused by the system
     *
     * @param sourceClass the source class issuing the exception
     * @param func the function issuing the exception
     * @param error the error object itself (optional)
     */
    createErrorMsg: function(sourceClass, func, error) {
        var ret = [];

        var keyValToStr = this.keyValToStr;
        ret.push(keyValToStr(this.getMessage("MSG_AFFECTED_CLASS"), sourceClass));
        ret.push(keyValToStr(this.getMessage("MSG_AFFECTED_METHOD"), func));

        if (error) {
            var _UDEF = "undefined";

            ret.push(keyValToStr(this.getMessage("MSG_ERROR_NAME"), error.name ? error.name : _UDEF));
            ret.push(keyValToStr(this.getMessage("MSG_ERROR_MESSAGE"), error.message ? error.message : _UDEF));
            ret.push(keyValToStr(this.getMessage("MSG_ERROR_DESC"), error.description ? error.description : _UDEF));
            ret.push(keyValToStr(this.getMessage("MSG_ERROR_NO"), _UDEF != typeof error.number ? error.number : _UDEF));
            ret.push(keyValToStr(this.getMessage("MSG_ERROR_LINENO"), _UDEF != typeof error.lineNumber ? error.lineNumber : _UDEF));
        }
        return ret.join("");
    }
    ,

    /**
     * transforms a key value pair into a string
     * @param key the key
     * @param val the value
     * @param delimiter the delimiter
     */
    keyValToStr: function(key, val, delimiter) {
        var ret = [];
        ret.push(key);
        ret.push(val);
        if ('undefined' == typeof delimiter) {
            delimiter = "\n";
        }
        ret.push(delimiter);
        return ret.join("");
    }
    ,


    parseXML: function(txt) {
        try {
            var parser = null, xmlDoc = null;
            if (window.DOMParser) {
                parser = new DOMParser();
                xmlDoc = parser.parseFromString(txt, "text/xml");
            }
            else // Internet Explorer
            {
                xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
                xmlDoc.async = "false";
                xmlDoc.loadXML(txt);
            }
            return xmlDoc;
        } catch (e) {
            //undefined internal parser error
            return null;
        }
    }
    ,

    serializeXML: function(xmlNode) {
        if (xmlNode.xml) return xmlNode.xml; //IE
        //rest of the world
        return (new XMLSerializer()).serializeToString(xmlNode);
    }
    ,

    serializeChilds: function(xmlNode) {
        var buffer = [];
        if (!xmlNode.childNodes) return "";
        for (var cnt = 0; cnt < xmlNode.childNodes.length; cnt++) {
            buffer.push(this.serializeXML(xmlNode.childNodes[cnt]));
        }
        return buffer.join("");
    }
    ,
    isXMLParseError: function(xmlContent) {

        //no xml content
        if (xmlContent == null) return true;

        var findParseError = function(node) {
            if (!node || !node.childNodes) return false;
            for (var cnt = 0; cnt < node.childNodes.length; cnt++) {
                var childNode = node.childNodes[cnt];
                if (childNode.tagName && childNode.tagName == "parsererror") return true;
            }
            return false;
        };
        return !xmlContent ||
                (this.exists(xmlContent, "parseError.errorCode") && xmlContent.parseError.errorCode != 0) ||
                findParseError(xmlContent);


    }
    ,
    /**
     * creates a neutral form data wrapper over an existing form Data element
     * the wrapper delegates following methods, append
     * and adds makeFinal as finalizing method which returns the final
     * send representation of the element
     *
     * @param formData an array
     */
    createFormDataDecorator: function(formData) {
        //we simulate the dom level 2 form element here
        var _newCls = null;
        var bufInstance = null;

        if (!this.FormDataDecoratorArray) {
            this.FormDataDecoratorArray = function (theFormData) {
                this._valBuf = theFormData;
                this._idx = {};
            };
            _newCls = this.FormDataDecoratorArray;
            _newCls.prototype.append = function(key, val) {
                this._valBuf.push([encodeURIComponent(key), encodeURIComponent(val)].join("="));
                this._idx[key] = true;
            };
            _newCls.prototype.hasKey = function(key) {
                return !!this._idx[key];
            };
            _newCls.prototype.makeFinal = function() {
                return this._valBuf.join("&");
            };

        }
        if (!this.FormDataDecoratorOther) {
            this.FormDataDecoratorOther = function (theFormData) {
                this._valBuf = theFormData;
                this._idx = {};
            };
            _newCls = this.FormDataDecoratorOther;
            _newCls.prototype.append = function(key, val) {
                this._valBuf.append(key, val);
                this._idx[key] = true;
            };
            _newCls.prototype.hasKey = function(key) {
                return !!this._idx[key];
            };
            _newCls.prototype.makeFinal = function() {
                return this._valBuf;
            };
        }

        if (formData instanceof Array) {
            bufInstance = new this.FormDataDecoratorArray(formData);
        } else {
            bufInstance = new this.FormDataDecoratorOther(formData);
        }

        return bufInstance;
    }
})
        ;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * debugging replacement for lang which adds logging functionality
 * which is not yet present in the core
 * this is a full replacement class for myfaces._impl._util._Lang
 * and replaces the object entirely with
 * a delegated implementation which adds the new methods to Lang
 *
 * We use this class to move some debugging related
 * lang functions out of the core, which never will
 * be utilized directly in the core
 * but will be used externally by extension frameworks
 * and by unit tests
 */
/** @namespace myfaces._impl._util._ExtLang */
myfaces._impl.core._Runtime.singletonDelegateObj("myfaces._impl._util._ExtLang", myfaces._impl._util._Lang, {

    /**
     * we use a map switch instread of a log level
     * slightly slower but more flexible
     */
    _ERR: "error",
    _INF: "info",
    _DEB: "debug",
    _LOG: "log",
    _WRN: "warn",


    constructor_: function() {
        //we replace the original one, and since we delegated
        //we have everything in place
        myfaces._impl._util._Lang = this;
        //due to the delegation pattern we do not have access to the browser
        this._browser = myfaces._impl.core._Runtime.browser;

        this.logLevels = {};
        this.logLevels[this._ERR] = true;
        this.logLevels[this._INF] = true;
        this.logLevels[this._DEB] = true;
        this.logLevels[this._LOG] = true;
        this.logLevels[this._WRN] = true;

        //printing of a stack trace if possible
        this.stackTraceLevels = {};
        this.stackTraceLevels[this._ERR] = true;
        this.stackTraceLevels[this._INF] = false;
        this.stackTraceLevels[this._DEB] = false;
        this.stackTraceLevels[this._LOG] = false;
        this.stackTraceLevels[this._WRN] = false;
    },

    /**
     * Simple simple logging only triggering at
     * firebug compatible logging consoles
     *
     * note: ;; means the code will be stripped
     * from the production code by the build system
     */
    _log: function(logType /*+arguments*/, args) {

        var argsString = this.objToArray(arguments[1]).join(" ");
        var c = window.console;
        

        if (c && c[logType]) {
            c[logType](argsString);
            if(this.stackTraceLevels[logType] && c.trace) {
                c.trace();
            }
        }
        var logHolder = document.getElementById("myfaces.logging");

        if (logHolder) {
            var elem = document.createElement("div");
            var b = this._browser;
            if (!b.isIE || b.isIE > 7) {
                //w3 compliant class setting
                elem.setAttribute("class", "consoleLog " + logType);
            } else {
                //ie quirks compliant class setting
                elem.className ="consoleLog " + logType;
            }
            logHolder.appendChild(elem);
            elem.innerHTML = logType.toUpperCase() + ": " + argsString;
        }
    },

    logError: function(/*varargs*/) {
        if (!this.logLevels[this._ERR]) return;
        this._log(this._ERR, arguments);
    }
    ,
    logWarn: function(/*varargs*/) {
        if (!this.logLevels[this._WRN]) return;
        this._log(this._WRN, arguments);
    }
    ,
    logInfo: function(/*varargs*/) {
        if (!this.logLevels[this._INF]) return;
        this._log(this._INF, arguments);
    }
    ,
    logDebug: function(/*varargs*/) {
        //Level 1 == debug
        if (!this.logLevels[this._DEB]) return;
        this._log(this._DEB, arguments);
    }
    ,
    logTrace: function(/*varargs*/) {
        if (!this.logLevels[this._LOG]) return;
        this._log("log", arguments);
    }

});


/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** @namespace myfaces._impl._util._Queue */
myfaces._impl.core._Runtime.extendClass("myfaces._impl._util._Queue", Object, {
    //faster queue by http://safalra.com/web-design/javascript/queues/Queue.js
    //license public domain
    //The trick is to simply reduce the number of slice and slice ops to a bare minimum.

    _q : null,
    _space : 0,
    _size: -1,

    constructor_: function() {
        this._q = [];
        this._Lang = myfaces._impl._util._Lang;
    },

    length: function() {
        // return the number of elements in the queue
        return this._q.length - this._space;

    },

    isEmpty: function() {
        // return true if the queue is empty, and false otherwise
        return (this._q.length == 0);
    },

    setQueueSize: function(newSize) {
        this._size = newSize;
        this._readjust();
    },

    /**
     * adds a listener to the queue
     *
     * @param element the listener to be added
     */
    enqueue : function(/*function*/element) {
        this._q.push(element);
        //qeuesize is bigger than the limit we drop one element so that we are
        //back in line

        this._readjust();
    },

    _readjust: function() {
        var size = this._size;
        while (null != size && 'undefined' != typeof size &&
                size > -1 && this.length() > size) {
            this.dequeue();
        }
    },

    /**
     * removes a listener form the queue
     *
     * @param element the listener to be removed
     */
    remove : function(/*function*/element) {
        /*find element in queue*/
        var index = this.indexOf(element);
        /*found*/
        if (index != -1) {
            this._q.splice(index, 1);
        }
    },

    dequeue: function() {
        // initialise the element to return to be undefined
        var element = null;

        // check whether the queue is empty
        var qLen = this._q.length;
        var queue = this._q;
        
        if (qLen) {

            // fetch the oldest element in the queue
            element = queue[this._space];

            // update the amount of space and check whether a shift should occur
            //added here a max limit of 30
            //now bit shift left is a tad faster than multiplication on most vms and does the same
            //unless we run into a bit skipping which is impossible in our usecases here
            if ((++this._space) << 1 >= qLen) {

                // set the queue equal to the non-empty portion of the queue
                this._q = queue.slice(this._space);

                // reset the amount of space at the front of the queue
                this._space = 0;

            }

        }

        // return the removed element
        return element;
    },

    /**
     * simple foreach
     *
     * @param closure a closure which processes the element
     */
    each: function(closure) {
        this._Lang.arrForEach(this._q, closure, this._space);
    },

    /**
     * Simple filter
     *
     * @param closure a closure which returns true or false depending
     * whether the filter has triggered
     *
     * @return an array of filtered queue entries
     */
    arrFilter: function(closure) {
        return this._Lang.arrFilter(this._q, closure, this._space);
    },

    indexOf: function(element) {
        return this._Lang.indexOf(this._q, element);
    },

    cleanup: function() {
        this._q = [];
        this._space = 0;
    }
});


/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * a classical listener queue pattern
 */




/**
 * Simple listener queue with closures which shall be
 * called
 *
 * idea:
 * var queue = new myfaces._impl._util._ListenerQueue();
 */

/** @namespace myfaces._impl._util._ListenerQueue */
myfaces._impl.core._Runtime.extendClass("myfaces._impl._util._ListenerQueue", myfaces._impl._util._Queue, {

    constructor_: function() {
        this._callSuper("constructor");
    },

    /**
     * listener type safety assertion function
     *
     * @param listener must be of type function otherwise an error is raised
     */
    _assertListener : function(/*function*/listener) {
        if ("function" != typeof (listener)) {
            var msg = myfaces._impl._util._Lang.getMessage("ERR_PARAM_GENERIC",null,"_ListenerQueue", arguments.caller.toString(),"function" );
            throw Error(msg);
        }
    },

    /**
     * adds a listener to the queue
     *
     * @param listener the listener to be added
     */
    enqueue : function(/*function*/listener) {
        this._assertListener(listener);
        this._callSuper("enqueue", listener);
    },

    /**
     * removes a listener form the queue
     *
     * @param listener the listener to be removed
     */
    remove : function(/*function*/listener) {
        this._assertListener(listener);
        this._callSuper("remove", listener);
    },

    /**
     * generic broadcast with a number of arguments being passed down
     */
    broadcastEvent : function(/*any*/argument) {
        var _Lang = myfaces._impl._util._Lang;
        var _args = _Lang.objToArray(arguments);

        var broadCastFunc = function(element) {
            element.apply(null, _args);
        };
        try {
            this.each(broadCastFunc);
        } finally {
            broadCastFunc = null;
        }
    }
});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * A collection of dom helper routines
 * (which in later incarnations will
 * get browser specific speed optimizations)
 *
 * Since we have to be as tight as possible
 * we will focus with our dom routines to only
 * the parts which our impl uses.
 * A jquery like query API would be nice
 * but this would blow up our codebase significantly
 *
 * TODO selector shortcuts bei chrome abdrehen da knallt es
 *
 */
/** @namespace myfaces._impl._util._Dom */

/** @namespace NodeFilter */
/** @namespace NodeFilter.FILTER_ACCEPT */
/** @namespace NodeFilter.FILTER_SKIP */
/** @namespace NodeFilter.FILTER_REJECT */
/** @namespace NodeFilter.SHOW_ELEMENT */
myfaces._impl.core._Runtime.singletonExtendClass("myfaces._impl._util._Dom", Object, {
    IE_QUIRKS_EVENTS : {
        "onabort": true,
        "onload":true,
        "onunload":true,
        "onchange": true,
        "onsubmit": true,
        "onreset": true,
        "onselect": true,
        "onblur": true,
        "onfocus": true,
        "onkeydown": true,
        "onkeypress": true,
        "onkeyup": true,
        "onclick": true,
        "ondblclick": true,
        "onmousedown": true,
        "onmousemove": true,
        "onmouseout": true,
        "onmouseover": true,
        "onmouseup": true
    },

    _Lang:  myfaces._impl._util._Lang,
    _RT:    myfaces._impl.core._Runtime,
    _dummyPlaceHolder:null,

    constructor_: function() {
        //we have to trigger it upfront because mozilla runs the eval
        //after the dom updates and hence causes a race conditon if used on demand
        //under normal circumstances this works, if there are no normal ones
        //then this also will work at the second time, but the onload handler
        //should cover 99% of all use cases to avoid a loading race condition
        var b = myfaces._impl.core._Runtime.browser;

        if (b.isIE <= 6 && b.isIEMobile) {
            //winmobile hates add onLoad, and checks on the construct
            //it does not eval scripts anyway
            myfaces.config = myfaces.config || {};
            myfaces.config._autoeval = false;
            return;
        }
        this._RT.addOnLoad(window, function() {
            myfaces._impl._util._Dom.isManualScriptEval();
        });
        //safety fallback if the window onload handler is overwritten and not chained
        if (document.body) {
            this._RT.addOnLoad(document.body, function() {
                myfaces._impl._util._Dom.isManualScriptEval();
            });
        }
        //now of the onload handler also is overwritten we have a problem
    },

    /**
     * Run through the given Html item and execute the inline scripts
     * (IE doesn't do this by itself)
     * @param {|Node|} item
     */
    runScripts: function(item, xmlData) {
        var finalScripts    = [];
        var execScrpt       = this._Lang.hitch(this, function(item) {
            if (item.tagName && this._Lang.equalsIgnoreCase(item.tagName, "script")) {
                var src = item.getAttribute('src');
                if ('undefined' != typeof src
                        && null != src
                        && src.length > 0
                        ) {
                    //we have to move this into an inner if because chrome otherwise chokes
                    //due to changing the and order instead of relying on left to right
                    if ((src.indexOf("ln=scripts") == -1 && src.indexOf("ln=javax.faces") == -1) || (src.indexOf("/jsf.js") == -1
                            && src.indexOf("/jsf-uncompressed.js") == -1))
                        if (finalScripts.length) {
                            //script source means we have to eval the existing
                            //scripts before running the include
                            this._RT.globalEval(finalScripts.join("\n"));

                            finalScripts = [];
                        }
                    this._RT.loadScriptEval(src, item.getAttribute('type'), false, "UTF-8");
                } else {
                    // embedded script auto eval
                    var test = (!xmlData) ? item.text : this._Lang.serializeChilds(item);
                    var go = true;
                    while (go) {
                        go = false;
                        if (test.substring(0, 1) == " ") {
                            test = test.substring(1);
                            go = true;
                        }
                        if (test.substring(0, 4) == "<!--") {
                            test = test.substring(4);
                            go = true;
                        }
                        if (test.substring(0, 11) == "//<![CDATA[") {
                            test = test.substring(11);
                            go = true;
                        }
                    }
                    // we have to run the script under a global context
                    //we store the script for less calls to eval
                    finalScripts.push(test);

                }
            }
        });
        try {
            var scriptElements = this.findByTagName(item, "script", true);
            if (scriptElements == null) return;
            for (var cnt = 0; cnt < scriptElements.length; cnt++) {
                execScrpt(scriptElements[cnt]);
            }
            if (finalScripts.length) {
                this._RT.globalEval(finalScripts.join("\n"));
            }
        } finally {
            //the usual ie6 fix code
            //the IE6 garbage collector is broken
            //nulling closures helps somewhat to reduce
            //mem leaks, which are impossible to avoid
            //at this browser
            execScrpt = null;
        }
    },

    nodeIdOrName: function(elem) {
        if (elem) {
            elem = this.byId(elem);
            //detached element handling, we also store the element name
            //to get a fallback option in case the identifier is not determinable
            // anymore, in case of a framework induced detachment the element.name should
            // be shared if the identifier is not determinable anymore
            var elementId = elem.id || elem.name;
            if ((elementId == null || elementId == '') && elem.name) {
                elementId = elem.name;
            }
            return elementId;
        }
        return null;
    },

    /**
     * Simple delete on an existing item
     */
    deleteItem: function(itemIdToReplace) {
        var item = this.byId(itemIdToReplace);
        if (!item) {
            throw Error("_Dom.deleteItem  Unknown Html-Component-ID: " + itemIdToReplace);
        }

        this._removeNode(item, false);
    },

    /**
     * outerHTML replacement which works cross browserlike
     * but still is speed optimized
     *
     * @param item the item to be replaced
     * @param markup the markup for the replacement
     */
    outerHTML : function(item, markup) {
        if (!item) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null,"myfaces._impl._util._Dom.outerHTML", "item"));
        }
        if (!markup) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null,"myfaces._impl._util._Dom.outerHTML", "markup"));
        }

        markup = this._Lang.trim(markup);
        if (markup !== "") {
            var ret = null;

            //w3c compliant browsers with proper contextual fragments
            var parentNode;
            if (window.Range
                    && typeof Range.prototype.createContextualFragment == 'function') {
                ret = this._outerHTMLCompliant(item, markup);
            } else {

                ret = this._outerHTMLNonCompliant(item, markup);
            }

            // and remove the old item
            //first we have to save the node newly insert for easier access in our eval part
            if (this.isManualScriptEval()) {
                var isArr = ret instanceof Array;
                if (isArr && ret.length) {
                    for (var cnt = 0; cnt < ret.length; cnt++) {
                        this.runScripts(ret[cnt]);
                    }
                } else if (!isArr) {
                    this.runScripts(ret);
                }
            }
            return ret;
        }
        // and remove the old item, in case of an empty newtag and do nothing else
        this._removeNode(item, false);
        return null;
    },

    _outerHTMLCompliant: function(item, markup) {
        var evalNodes;
        //table element replacements like thead, tbody etc... have to be treated differently
        if (this._isTableElement(item)) {
            evalNodes = this._buildTableNodes(item, markup);
        } else {
            evalNodes = this._buildNodesCompliant(markup);
        }
        var evalNodeLen = evalNodes.length;

        if (evalNodeLen == 1) {
            var ret = evalNodes[0];
            item.parentNode.replaceChild(ret, item);
            return ret;
        } else {
            return this.replaceElements(item, evalNodes);
        }
    },

    /**
     * checks if the provided element is a table element
     * @param itemNodeName
     * @param innerOnly
     */
    _isTableElement: function(item) {
        var itemNodeName = (item.nodeName || item.tagName).toLowerCase();

        return this._isTableStructureElement(item) || itemNodeName == "td";
    },

    _isTableStructureElement: function(item) {
        var itemNodeName = (item.nodeName || item.tagName).toLowerCase();

        return itemNodeName == "table" || itemNodeName == "thead" ||
                itemNodeName == "tbody" || itemNodeName == "tfoot" ||
                itemNodeName == "th" || itemNodeName == "tr";
    },

    /**
     * now to the evil browsers
     * of what we are dealing with is various bugs
     * first a simple replaceElement leaks memory
     * secondly embedded scripts can be swallowed upon
     * innerHTML, we probably could also use direct outerHTML
     * but then we would run into the script swallow bug
     *
     * the entire mess is called IE6 and IE7
     *
     * @param item
     * @param markup
     */
    _outerHTMLNonCompliant: function(item, markup) {
        
        var b = this._RT.browser;
        var evalNodes = null;

        try {
            if (this._isTableElement(item)) {
                evalNodes = this._buildTableNodes(item, markup);
            } else {
                evalNodes = this._buildNodesNonCompliant(markup);
            }

            if (evalNodes.length == 1) {
                var ret = evalNodes[0];
                this.replaceElement(item, evalNodes[0]);
                return ret;
            } else {
                return this.replaceElements(item, evalNodes);
            }
       
        } finally {

            var dummyPlaceHolder = this.getDummyPlaceHolder();
            //now that Microsoft has finally given
            //ie a working gc in 8 we can skip the costly operation
            var b = myfaces._impl.core._Runtime.browser;

            if (b.isIE && b.isIE < 8) {
                this._removeChildNodes(dummyPlaceHolder, false);
            }
            dummyPlaceHolder.innerHTML = "";
        }

    },

    _buildNodesCompliant:
    function(markup) {
        var dummyPlaceHolder = this.getDummyPlaceHolder(); //document.createElement("div");
        dummyPlaceHolder.innerHTML = markup;
        return this._Lang.objToArray(dummyPlaceHolder.childNodes);
    }
    ,

    /**
     * builds up a correct dom subtree
     * if the markup is part of table nodes
     * @param item
     * @param markup
     */
    _buildTableNodes: function(item, markup) {
        var evalNodes;
        var itemNodeName = (item.nodeName || item.tagName).toLowerCase();
        var probe = this.getDummyPlaceHolder(); //document.createElement("div");
        if (itemNodeName == "td") {
            probe.innerHTML = "<table><tbody><tr><td></td></tr></tbody></table>";
        } else {
            probe.innerHTML = "<table><" + itemNodeName + "></" + itemNodeName + ">" + "</table>";
        }
        var depth = this._determineDepth(probe);
        
        this._removeChildNodes(probe, false);
        probe.innerHTML = "";

        var dummyPlaceHolder = this.getDummyPlaceHolder();//document.createElement("div");
        if (itemNodeName == "td") {
            dummyPlaceHolder.innerHTML = "<table><tbody><tr>" + markup + "</tr></tbody></table>";
        } else {
            dummyPlaceHolder.innerHTML = "<table>" + markup + "</table>";
        }
        evalNodes = dummyPlaceHolder;
        for (var cnt = 0; cnt < depth; cnt++) {
            evalNodes = evalNodes.childNodes[0];
        }
        evalNodes = (evalNodes.parentNode) ? evalNodes.parentNode.childNodes : null;
        return this._Lang.objToArray(evalNodes);
    }
    ,

    /**
     * builds the ie nodes properly in a placeholder
     * and bypasses a non script insert bug that way
     * @param markup the marku code
     */
    _buildNodesNonCompliant: function(markup) {

        var evalNodes = null;

        //now to the non w3c compliant browsers
        //http://blogs.perl.org/users/clinton_gormley/2010/02/forcing-ie-to-accept-script-tags-in-innerhtml.html
        //we have to cope with deficiencies between ie and its simulations in this case
        var probe = this.getDummyPlaceHolder();//document.createElement("div");

        probe.innerHTML = "<table><tbody><tr><td><div></div></td></tr></tbody></table>";

        //we have customers using html unit, this has a bug in the table resolution
        //hence we determine the depth dynamically
        var depth = this._determineDepth(probe);
        var newProbe = probe;
        this._removeChildNodes(probe, false);
        probe.innerHTML = "";

        var dummyPlaceHolder = this.getDummyPlaceHolder();//document.createElement("div");

        //fortunately a table element also works which is less critical than form elements regarding
        //the inner content
        dummyPlaceHolder.innerHTML = "<table><tbody><tr><td>" + markup + "</td></tr></tbody></table>";
        evalNodes = dummyPlaceHolder;

        for (var cnt = 0; cnt < depth; cnt++) {
            evalNodes = evalNodes.childNodes[0];
        }
        evalNodes = (evalNodes.parentNode) ? evalNodes.parentNode.childNodes : null;

        if ('undefined' == typeof evalNodes || null == evalNodes) {
            //fallback for htmlunit which should be good enough
            //to run the tests, maybe we have to wrap it as well
            dummyPlaceHolder.innerHTML = "<div>" + markup + "</div>";
            //note this is triggered only in htmlunit no other browser
            //so we are save here
            evalNodes = dummyPlaceHolder.childNodes[0].childNodes;
        }
        return this._Lang.objToArray(evalNodes);

    }
    ,

    _determineDepth: function(probe) {
        var depth = 0;
        var newProbe = probe;
        while (newProbe) {
            newProbe = newProbe.childNodes[0];
            depth++;
        }
        depth--;
        return depth;
    }
    ,


    //now to another nasty issue:
    //for ie we have to walk recursively over all nodes:
    //http://msdn.microsoft.com/en-us/library/bb250448%28VS.85%29.aspx
    //http://weblogs.java.net/blog/driscoll/archive/2009/11/13/ie-memory-management-and-you
    //http://home.orange.nl/jsrosman/
    //http://www.quirksmode.org/blog/archives/2005/10/memory_leaks_li.html
    //http://www.josh-davis.org/node/7
    _removeNode: function(node, breakEventsOpen) {
        if (!node) return;
        var b = this._RT.browser;
        if (!b.isIE || b.isIE >= 8) {
            //recursive descension only needed for old ie versions
            //all newer browsers cleanup the garbage just fine without it
            //thank you
            if ('undefined' != typeof node.parentNode && null != node.parentNode) //if the node has a parent
                node.parentNode.removeChild(node);
            return;
        }

        //now to the browsers with non working garbage collection
        this._removeChildNodes(node, breakEventsOpen);

        try {
            //outer HTML setting is only possible in earlier IE versions all modern browsers throw an exception here
            //again to speed things up we precheck first
            if(!this._isTableStructureElement(childNode)) {
                //we do not do a table structure innnerhtml on table elements except td
                //htmlunit rightfully complains that we should not do it
                node.innerHTML = "";
            }
            if (b.isIE && 'undefined' != typeof node.outerHTML) {//ie8+ check done earlier we skip it here
                node.outerHTML = '';
            } else {
                this._removeFromParent(node);
            }
            if (!b.isIEMobile) {
                delete node;
            }
        } catch (e) {
            //on some elements we might not have covered by our table check on the outerHTML
            // can fail we skip those in favor of stability
            try {
                // both innerHTML and outerHTML fails when <tr> is the node, but in that case 
                // we need to force node removal, otherwise it will be on the tree (IE 7 IE 6)
                this._removeFromParent(node);
            } catch (e1) {
            }
        }
    }
    ,

    _removeFromParent: function(node) {
        if ('undefined' != typeof node.parentNode && null != node.parentNode) //if the node has a parent
                    node.parentNode.removeChild(node);
    },

    /**
     * recursive delete child nodes
     * node, this method only makes sense in the context of IE6 + 7 hence
     * it is not exposed to the public API, modern browsers
     * can garbage collect the nodes just fine by doing the standard removeNode method
     * from the dom API!
     *
     * @param node  the node from which the childnodes have to be deletd
     * @param breakEventsOpen if set to true a standard events breaking is performed
     */
    _removeChildNodes: function(node, breakEventsOpen) {
        if (!node) return;

        //node types which cannot be cleared up by normal means
        var disallowedNodes = {
            "thead": true,
            "tbody": true,
            "tr": true,
            "td": true
        };

        //for now we do not enable it due to speed reasons
        //normally the framework has to do some event detection
        //which we cannot do yet, I will dig for options
        //to enable it in a speedly manner
        //ie7 fixes this area anyway
        //this.breakEvents(node);

        var b = this._RT.browser;
        if (breakEventsOpen) {
            this.breakEvents(node);
        }

        for (var cnt = node.childNodes.length - 1; cnt >= 0; cnt -= 1) {
            var childNode = node.childNodes[cnt];
            //we cannot use our generic recursive tree walking due to the needed head recursion
            //to clean it up bottom up, the tail recursion we were using in the search either would use more time
            //because we had to walk and then clean bottom up, so we are going for a direct head recusion here
            if ('undefined' != typeof childNode.childNodes && node.childNodes.length)
                this._removeChildNodes(childNode);
            try {
                var nodeName = (childNode.nodeName || childNode.tagName) ? (childNode.nodeName || childNode.tagName).toLowerCase() : null;
                //ie chokes on clearing out table inner elements, this is also covered by our empty
                //catch block, but to speed things up it makes more sense to precheck that
                if (!disallowedNodes[nodeName]) {
                    //outer HTML setting is only possible in earlier IE versions all modern browsers throw an exception here
                    //again to speed things up we precheck first
                    if(!this._isTableStructureElement(childNode)) {    //table elements cannot be deleted
                        childNode.innerHTML = "";
                    }
                    if (b.isIE && b.isIE < 8 && 'undefined' != childNode.outerHTML)
                        childNode.outerHTML = '';
                    else {
                        node.removeChild(childNode);
                    }
                    if (!b.isIEMobile) {
                        delete childNode;
                    }
                }
            } catch (e) {
                //on some elements the outerHTML can fail we skip those in favor
                //of stability

            }
        }
    }
    ,

    /**
     * break the standard events from an existing dom node
     * (note this method is not yet used, but can be used
     * by framework authors to get rid of ie circular event references)
     *
     * another way probably would be to check all attributes of a node
     * for a function and if one is present break it by nulling it
     * I have to do some further investigation on this.
     *
     * The final fix is to move away from ie6 at all which is the root cause of
     * this.
     *
     * @param node the node which has to be broken off its events
     */
    breakEvents: function(node) {
        if (!node) return;
        var evtArr = this.IE_QUIRKS_EVENTS;
        for (var key in evtArr) {
            if (key != "onunload" && node[key]) {
                node[key] = null;
            }
        }
    }
    ,


    /**
     * for performance reasons we work with replaceElement and replaceElements here
     * after measuring performance it has shown that passing down an array instead
     * of a single node makes replaceElement twice as slow, however
     * a single node case is the 95% case
     *
     * @param item
     * @param evalNodes
     */
    replaceElement: function(item, evalNode) {

        var _Browser = this._RT.browser;
        if (!_Browser.isIE || _Browser.isIE >= 8) {
            //standards conform no leaking browser
            item.parentNode.replaceChild(evalNode, item);
        } else {
            //browsers with defect garbage collection
            item.parentNode.insertBefore(evalNode, item);
            this._removeNode(item, false);
        }
    }
    ,


    /**
     * replaces an element with another element or a set of elements
     *
     * @param item the item to be replaced
     *
     * @param evalNodes the elements
     */
    replaceElements: function (item, evalNodes) {
        var evalNodesDefined = evalNodes && 'undefined' != typeof evalNodes.length;
        if (!evalNodesDefined) {
            throw new Error(this._Lang.getMessage("ERR_REPLACE_EL"));
        }

        var parentNode = item.parentNode;

        var sibling = item.nextSibling;
        var resultArr = this._Lang.objToArray(evalNodes);

        for (var cnt = 0; cnt < resultArr.length; cnt++) {
            if (cnt == 0) {
                this.replaceElement(item, resultArr[cnt]);
            } else {
                if (sibling) {
                    parentNode.insertBefore(resultArr[cnt], sibling);
                } else {
                    parentNode.appendChild(resultArr[cnt]);

                }
            }
        }

        return resultArr;
    }
    ,

    /**
     * optimized search for an array of tag names
     *
     * @param fragment the fragment which should be searched for
     * @param tagNames an map indx of tag names which have to be found
     * @param deepScan if set to true a deep scan is performed otherwise a shallow scan
     */
    findByTagNames: function(fragment, tagNames, deepScan) {


        //shortcut for single components
        if (!deepScan && tagNames[fragment.tagName.toLowerCase()]) {
            return fragment;
        }

        //shortcut elementsByTagName
        if (deepScan && this._Lang.exists(fragment, "getElementsByTagName")) {
            var retArr = [];
            for (var key in tagNames) {
                var foundElems = this.findByTagName(fragment, key, deepScan);
                if (foundElems) {
                    retArr = retArr.concat(foundElems);
                }
            }
            return retArr;
        } else if (deepScan) {
            //no node type with child tags we can handle that without node type checking
            return null;
        }

        //now the filter function checks case insensitively for the tag names needed
        var filter = function(node) {
            return node.tagName && tagNames[node.tagName.toLowerCase()];
        };

        //now we run an optimized find all on it
        try {
            return this.findAll(fragment, filter, deepScan);
        } finally {
            //the usual IE6 is broken, fix code
            filter = null;
        }
    }
    ,

    /**
     * determines the number of nodes according to their tagType
     *
     * @param {Node} fragment (Node or fragment) the fragment to be investigated
     * @param {String} tagName the tag name (lowercase)
     * @param {Boolean} deepScan if set to true a found element does not prevent to scan deeper
     * (the normal usecase is false, which means if the element is found only its
     * adjacent elements will be scanned, due to the recursive descension
     * this should work out with elements with different nesting depths but not being
     * parent and child to each other
     *
     * @return the child elements as array or null if nothing is found
     *
     */
    findByTagName : function(fragment, tagName, deepScan) {

        //remapping to save a few bytes
        var _Lang = this._Lang;

        deepScan = !!deepScan;

        //elements by tagname is the fastest, ie throws an error on fragment.getElementsByTagName, the exists type
        //via namespace array checking is safe
        if (deepScan && _Lang.exists(fragment, "getElementsByTagName")) {
            var ret = _Lang.objToArray(fragment.getElementsByTagName(tagName));
            if (fragment.tagName && _Lang.equalsIgnoreCase(fragment.tagName, tagName)) ret.unshift(fragment);
            return ret;
        } else if (deepScan) {
            //no node type with child tags we can handle that without node type checking
            return null;
        }
        //since getElementsByTagName is a standardized dom node function and ie also supports
        //it since 5.5
        //we need no fallback to the query api and the recursive filter
        //also is only needed in case of no deep scan or non dom elements

        var filter = function(node) {
            return node.tagName && _Lang.equalsIgnoreCase(node.tagName, tagName);
        };
        try {
            return this.findAll(fragment, filter, deepScan);
        } finally {
            //the usual IE6 is broken, fix code
            filter = null;
            _Lang = null;
        }

    }
    ,

    findByName : function(fragment, name, deepScan) {
        var _Lang = this._Lang;
        var filter = function(node) {
            return  node.name && _Lang.equalsIgnoreCase(node.name, name);
        };
        try {
            deepScan = !!deepScan;

            //elements byName is the fastest
            if (deepScan && _Lang.exists(fragment, "getElementsByName")) {
                var ret = _Lang.objToArray(fragment.getElementsByName(name));
                if (fragment.name == name) ret.unshift(fragment);
                return ret;

            }

            if (deepScan && _Lang.exists(fragment, "querySelectorAll")) {
                try {
                    var newName = name;
                    if (_Lang.isString(newName)) {
                        newName = _Lang.escapeString(newName);
                    }
                    var result = fragment.querySelectorAll("[name=" + newName + "]");
                    if (fragment.nodeType == 1 && filter(fragment)) {
                        result = (result == null) ? [] : _Lang.objToArray(result);
                        result.push(fragment);
                    }
                    return result;
                } catch(e) {
                    //in case the selector bombs we retry manually
                }
            }

            return this.findAll(fragment, filter, deepScan);
        } finally {
            //the usual IE6 is broken, fix code
            filter = null;
            _Lang = null;
        }
    }
    ,



    /**
     * a filtered findAll for subdom treewalking
     * (which uses browser optimizations wherever possible)
     *
     * @param {|Node|} rootNode the rootNode so start the scan
     * @param filter filter closure with the syntax {boolean} filter({Node} node)
     * @param deepScan if set to true or not set at all a deep scan is performed (for form scans it does not make much sense to deeply scan)
     */
    findAll : function(rootNode, filter, deepScan) {
        this._Lang.assertType(filter, "function");
        deepScan = !!deepScan;

        if (document.createTreeWalker && NodeFilter) {
            return this._iteratorSearchAll(rootNode, filter, deepScan);
        } else {
            return this._recursionSearchAll(rootNode, filter, deepScan);
        }

    }
    ,

    /**
     * classical recursive way which definitely will work on all browsers
     * including the IE6
     *
     * @param rootNode the root node
     * @param filter the filter to be applied to
     * @param deepScan if set to true a deep scan is performed
     */
    _recursionSearchAll: function(rootNode, filter, deepScan) {
        var ret = [];
        //fix the value to prevent undefined errors

        if (filter(rootNode)) {
            ret.push(rootNode);
            if (!deepScan) return ret;
        }

        //
        if (!rootNode.childNodes) {
            return ret;
        }

        //subfragment usecases

        var retLen = ret.length;
        var childLen = rootNode.childNodes.length;
        for (var cnt = 0; (deepScan || retLen == 0) && cnt < childLen; cnt++) {
            ret = ret.concat(this._recursionSearchAll(rootNode.childNodes[cnt], filter, deepScan));
        }
        return ret;
    }
    ,

    /**
     * the faster dom iterator based search, works on all newer browsers
     * except ie8 which already have implemented the dom iterator functions
     * of html 5 (which is pretty all standard compliant browsers)
     *
     * The advantage of this method is a faster tree iteration compared
     * to the normal recursive tree walking.
     *
     * @param rootNode the root node to be iterated over
     * @param filter the iteration filter
     * @param deepScan if set to true a deep scan is performed
     */
    _iteratorSearchAll: function(rootNode, filter, deepScan) {
        var retVal = [];
        //Works on firefox and webkit, opera and ie have to use the slower fallback mechanis
        //we have a tree walker in place this allows for an optimized deep scan
        if (filter(rootNode)) {

            retVal.push(rootNode);
            if (!deepScan) {
                return retVal;
            }
        }
        //we use the reject mechanism to prevent a deep scan reject means any
        //child elements will be omitted from the scan
        var walkerFilter = function (node) {
            var retCode = (filter(node)) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_SKIP;
            retCode = (!deepScan && retCode == NodeFilter.FILTER_ACCEPT) ? NodeFilter.FILTER_REJECT : retCode;
            if (retCode == NodeFilter.FILTER_ACCEPT || retCode == NodeFilter.FILTER_REJECT) {
                retVal.push(node);
            }
            return retCode;
        };
        var treeWalker = document.createTreeWalker(rootNode, NodeFilter.SHOW_ELEMENT, walkerFilter, false);
        //noinspection StatementWithEmptyBodyJS
        while (treeWalker.nextNode());
        return retVal;
    }
    ,

    /**
     * bugfixing for ie6 which does not cope properly with setAttribute
     */
    setAttribute : function(node, attr, val) {

        if (!node) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.setAttribute", "node {DomNode}"));
        }
        if (!attr) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.setAttribute", "attr {String}"));
        }

        //quirks mode and ie7 mode has the attributes problems ie8 standards mode behaves like
        //a good citizen
        var _Browser = this._RT.browser;
        if (!_Browser.isIE || _Browser.isIE > 7) {
            if (!node.setAttribute) {
                return;
            }
            node.setAttribute(attr, val);
            return;
        }

        /*
         Now to the broken browsers IE6+.... ie7 and ie8 quirks mode

         we deal mainly with three problems here
         class and for are not handled correctly
         styles are arrays and cannot be set directly
         and javascript events cannot be set via setAttribute as well!

         or in original words of quirksmode.org ... this is a mess!

         Btw. thank you Microsoft for providing all necessary tools for free
         for being able to debug this entire mess in the ie rendering engine out
         (which is the Microsoft ie vms, developers toolbar, Visual Web Developer 2008 express
         and the ie8 8 developers toolset!)

         also thank you http://www.quirksmode.org/
         dojotoolkit.org and   //http://delete.me.uk/2004/09/ieproto.html
         for additional information on this mess!

         The lowest common denominator tested within this code
         is IE6, older browsers for now are legacy!
         */
        attr = attr.toLowerCase();

        if (attr === "class") {
            //setAttribute does not work for winmobile browsers
            //firect calls work
            node.className = val;
        } else if (attr === "name") {
            //the ie debugger fails to assign the name via setAttr
            //in quirks mode
            node[attr] = val;
        } else if (attr === "for") {
            if (!_Browser.isIEMobile || _Browser.isIEMobile >= 7) {
                node.setAttribute("htmlFor", val);
            } else {
                node.htmlFor = val;
            }
        } else if (attr === "style") {
            //We have to split the styles here and assign them one by one
            var styles = val.split(";");
            var stylesLen = styles.length;
            for (var loop = 0; loop < stylesLen; loop++) {
                var keyVal = styles[loop].split(":");
                if (keyVal[0] != "" && keyVal[0] == "opacity") {
                    //special ie quirks handling for opacity

                    var opacityVal = Math.max(100, Math.round(parseFloat(keyVal[1]) * 10));
                    //probably does not work in ie mobile anyway
                    if (!_Browser.isIEMobile || _Browser.isIEMobile >= 7) {
                        node.style.setAttribute("arrFilter", "alpha(opacity=" + opacityVal + ")");
                    }
                    //if you need more hacks I would recommend
                    //to use the class attribute and conditional ie includes!
                } else if (keyVal[0] != "") {
                    if (!_Browser.isIEMobile || _Browser.isIEMobile >= 7) {
                        node.style.setAttribute(keyVal[0], keyVal[1]);
                    } else {
                        node.style[keyVal[0]] = keyVal[1];
                    }
                }
            }
        } else {
            //check if the attribute is an event, since this applies only
            //to quirks mode of ie anyway we can live with the standard html4/xhtml
            //ie supported events
            if (this.IE_QUIRKS_EVENTS[attr]) {
                if (this._Lang.isString(attr)) {
                    //event resolves to window.event in ie
                    node.setAttribute(attr, function() {
                        //event implicitly used
                        return this._Lang.globalEval(val);
                    });
                }
            } else {
                //unknown cases we try to catch them via standard setAttributes
                if (!_Browser.isIEMobile || _Browser.isIEMobile >= 7) {
                    node.setAttribute(attr, val);
                } else {
                    node[attr] = val;
                }
            }
        }
    }
    ,


    /**
     * fuzzy form detection which tries to determine the form
     * an item has been detached.
     *
     * The problem is some Javascript libraries simply try to
     * detach controls by reusing the names
     * of the detached input controls. Most of the times,
     * the name is unique in a jsf scenario, due to the inherent form mapping.
     * One way or the other, we will try to fix that by
     * identifying the proper form over the name
     *
     * We do it in several ways, in case of no form null is returned
     * in case of multiple forms we check all elements with a given name (which we determine
     * out of a name or id of the detached element) and then iterate over them
     * to find whether they are in a form or not.
     *
     * If only one element within a form and a given identifier found then we can pull out
     * and move on
     *
     * We cannot do much further because in case of two identical named elements
     * all checks must fail and the first elements form is served.
     *
     * Note, this method is only triggered in case of the issuer or an ajax request
     * is a detached element, otherwise already existing code has served the correct form.
     *
     * This method was added because of
     * https://issues.apache.org/jira/browse/MYFACES-2599
     * to support the integration of existing ajax libraries which do heavy dom manipulation on the
     * controls side (Dojos Dijit library for instance).
     *
     * @param {Node} elem - element as source, can be detached, undefined or null
     *
     * @return either null or a form node if it could be determined
     */
    fuzzyFormDetection : function(elem) {
        if (!document.forms || !document.forms.length) {
            return null;
        }

        // This will not work well on portlet case, because we cannot be sure
        // the returned form is right one.
        //we can cover that case by simply adding one of our config params
        //the default is the weaker, but more correct portlet code
        //you can override it with myfaces_config.no_portlet_env = true globally
        else if (1 == document.forms.length && this._RT.getGlobalConfig("no_portlet_env", false)) {
            return document.forms[0];
        }
        if (!elem) {
            return null;
        }

        //before going into the more complicated stuff we try the simple approach
        if (!this._Lang.isString(elem)) {

            //html 5 allows finally the detachement of elements
            //by introducing a form attribute

            var elemForm = this.html5FormDetection(elem);
            if (elemForm) {
                return elemForm;
            }

            //element of type form then we are already
            //at form level for the issuing element
            //https://issues.apache.org/jira/browse/MYFACES-2793

            if (this._Lang.equalsIgnoreCase(elem.tagName, "form")) {
                return elem;
            }
            var ret = this.getParent(elem, "form");
            if (ret) return ret;
        } else {
            elem = this.byId(elem);
            // element might have removed from DOM in method processUpdate 
            if (!elem){
            	return null;
            }
            var ret = this.getParent(elem, "form");
            if (ret) return ret;
        }

        var id = elem.id || null;
        var name = elem.name || null;
        //a framework in a detachment case also can replace an existing identifier element
        // with a name element
        name = name || id;
        var foundForm;

        if (id && '' != id) {
            //we have to assert that the element passed down is detached
            var domElement = this.byId(id);
            var elemForm = this.html5FormDetection(domElement);
            if (elemForm) {
                return elemForm;
            }

            if (domElement) {
                foundForm = this.getParent(domElement, "form");
                if (null != foundForm) return foundForm;
            }
        }

        /**
         * name check
         */
        var foundElements = [];

        /**
         * the lesser chance is the elements which have the same name
         * (which is the more likely case in case of a brute dom replacement)
         */
        var nameElems = document.getElementsByName(name);
        if (nameElems) {
            for (var cnt = 0; cnt < nameElems.length && foundElements.length < 2; cnt++) {
                // we already have covered the identifier case hence we only can deal with names,
                foundForm = this.getParent(nameElems[cnt], "form");
                if (null != foundForm) {
                    foundElements.push(foundForm);
                }
            }
        }

        return (1 == foundElements.length ) ? foundElements[0] : null;
    }
    ,

    html5FormDetection: function(item) {
        if (this._RT.browser.isIEMobile && this._RT.browser.isIEMobile <= 7) {
            return null;
        }
        var elemForm = this.getAttribute(item, "form");
        if (elemForm) {
            return this.byId(elemForm);
        }
        return null;
    }
    ,

    /**
     * gets a parent of an item with a given tagname
     * @param {Node} item - child element
     * @param {String} tagName - TagName of parent element
     */
    getParent : function(item, tagName) {

        if (!item) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.getParent", "item {DomNode}"));
        }

        var _Lang = this._Lang;
        var searchClosure = function(parentItem) {
            return parentItem && parentItem.tagName
                    && _Lang.equalsIgnoreCase(parentItem.tagName, tagName);
        };
        try {
            return this.getFilteredParent(item, searchClosure);
        } finally {
            searchClosure = null;
            _Lang = null;
        }
    }
    ,

    /**
     * A parent walker which uses
     * a filter closure for filtering
     *
     * @param {Node} item the root item to ascend from
     * @param {function} filter the filter closure
     */
    getFilteredParent : function(item, filter) {
        if (!item) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.getFilteredParent", "item {DomNode}"));
        }
        if (!filter) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.getFilteredParent", "filter {function}"));
        }

        //search parent tag parentName
        var parentItem = (item.parentNode) ? item.parentNode : null;

        while (parentItem && !filter(parentItem)) {
            parentItem = parentItem.parentNode;
        }
        return (parentItem) ? parentItem : null;
    }
    ,

    /**
     * a closure based child filtering routine
     * which steps one level down the tree and
     * applies the filter closure
     *
     * @param item the node which has to be investigates
     * @param filter the filter closure
     */
    getFilteredChild: function(item, filter) {
        if (!item) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.getFilteredParent", "item {DomNode}"));
        }
        if (!filter) {
            throw Error(this._Lang.getMessage("ERR_MUST_BE_PROVIDED1",null, "_Dom.getFilteredParent", "filter {function}"));
        }

        var childs = item.childNodes;
        if (!childs) {
            return null;
        }
        for (var c = 0, cLen = childs.length; c < cLen; c++) {
            if (filter(childs[c])) {
                return childs[c];
            }
        }
        return null;
    }
    ,

    /**
     * gets the child of an item with a given tag name
     * @param {Node} item - parent element
     * @param {String} childName - TagName of child element
     * @param {String} itemName - name  attribute the child can have (can be null)
     * @Deprecated
     */
    getChild: function(item, childName, itemName) {
        var _Lang = this._Lang;

        function filter(node) {
            return node.tagName
                    && _Lang.equalsIgnoreCase(node.tagName, childName)
                    && (!itemName || (itemName && itemName == node.getAttribute("name")));

        }

        return this.getFilteredChild(item, filter);
    }
    ,

    /**
     * cross ported from dojo
     * fetches an attribute from a node
     *
     * @param {String} node the node
     * @param {String} attr the attribute
     * @return the attributes value or null
     */
    getAttribute : function(/* HTMLElement */node, /* string */attr) {
        //	summary
        //	Returns the value of attribute attr from node.
        node = this.byId(node);
        // FIXME: need to add support for attr-specific accessors
        if ((!node) || (!node.getAttribute)) {
            // if(attr !== 'nwType'){
            //	alert("getAttr of '" + attr + "' with bad node");
            // }
            return null;
        }
        var ta = typeof attr == 'string' ? attr : new String(attr);

        // first try the approach most likely to succeed
        var v = node.getAttribute(ta.toUpperCase());
        if ((v) && (typeof v == 'string') && (v != "")) {
            return v;	//	string
        }

        // try returning the attributes value, if we couldn't get it as a string
        if (v && v.value) {
            return v.value;	//	string
        }

        // this should work on Opera 7, but it's a little on the crashy side
        if ((node.getAttributeNode) && (node.getAttributeNode(ta))) {
            return (node.getAttributeNode(ta)).value;	//	string
        } else if (node.getAttribute(ta)) {
            return node.getAttribute(ta);	//	string
        } else if (node.getAttribute(ta.toLowerCase())) {
            return node.getAttribute(ta.toLowerCase());	//	string
        }
        return null;	//	string
    }
    ,

    /**
     * checks whether the given node has an attribute attached
     *
     * @param {String|Object} node the node to search for
     * @param {String} attr the attribute to search for
     * @true if the attribute was found
     */
    hasAttribute : function(/* HTMLElement */node, /* string */attr) {
        //	summary
        //	Determines whether or not the specified node carries a value for the attribute in question.
        return this.getAttribute(node, attr) ? true : false;	//	boolean
    }
    ,

    /**
     * fetches the style class for the node
     * cross ported from the dojo toolkit
     * @param {String|Object} node the node to search
     * @returns the className or ""
     */
    getClass : function(node) {
        node = this.byId(node);
        if (!node) {
            return "";
        }
        var cs = "";
        if (node.className) {
            cs = node.className;
        } else {
            if (this.hasAttribute(node, "class")) {
                cs = this.getAttribute(node, "class");
            }
        }
        return cs.replace(/^\s+|\s+$/g, "");
    }
    ,
    /**
     * fetches the class for the node,
     * cross ported from the dojo toolkit
     * @param {String|Object}node the node to search
     */
    getClasses : function(node) {
        var c = this.getClass(node);
        return (c == "") ? [] : c.split(/\s+/g);
    }
    ,

    /**
     * concatenation routine which concats all childnodes of a node which
     * contains a set of CDATA blocks to one big string
     * @param {Node} node the node to concat its blocks for
     */
    concatCDATABlocks : function(/*Node*/ node) {
        var cDataBlock = [];
        // response may contain several blocks
        for (var i = 0; i < node.childNodes.length; i++) {
            cDataBlock.push(node.childNodes[i].data);
        }
        return cDataBlock.join('');
    }
    ,

    isManualScriptEval: function() {

        if (!this._Lang.exists(myfaces, "config._autoeval")) {
            var _Browser = this._RT.browser;
            //now we rely on the document being processed if called for the first time
            var evalDiv = document.createElement("div");
            this._Lang.reserveNamespace("myfaces.config._autoeval");
            //null not swallowed
            myfaces.config._autoeval = false;

            var markup = "<script type='text/javascript'> myfaces.config._autoeval = true; </script>";
            //now we rely on the same replacement mechanisms as outerhtml because
            //some browsers have different behavior of embedded scripts in the contextualfragment
            //or innerhtml case (opera for instance), this way we make sure the
            //eval detection is covered correctly
            this.setAttribute(evalDiv, "style", "display:none");

            //it is less critical in some browsers (old ie versions)
            //to append as first element than as last
            //it should not make any difference layoutwise since we are on display none anyway.
            this.insertFirst(evalDiv);

            //we remap it into a real boolean value
            if (window.Range
                    && typeof Range.prototype.createContextualFragment == 'function') {
                this._outerHTMLCompliant(evalDiv, markup);
            } else {
                this._outerHTMLNonCompliant(evalDiv, markup);
            }

        }

        return  !myfaces.config._autoeval;
        /* var d = _this.browser;


         return (_this.exists(d, "isIE") &&
         ( d.isIE > 5.5)) ||
         //firefox at version 4 beginning has dropped
         //auto eval to be compliant with the rest
         (_this.exists(d, "isFF") &&
         (d.isFF > 3.9)) ||
         (_this.exists(d, "isKhtml") &&
         (d.isKhtml > 0)) ||
         (_this.exists(d, "isWebKit") &&
         (d.isWebKit > 0)) ||
         (_this.exists(d, "isSafari") &&
         (d.isSafari > 0));
         */
        //another way to determine this without direct user agent parsing probably could
        //be to add an embedded script tag programmatically and check for the script variable
        //set by the script if existing, the add went through an eval if not then we
        //have to deal with it ourselves, this might be dangerous in case of the ie however
        //so in case of ie we have to parse for all other browsers we can make a dynamic
        //check if the browser does auto eval

    }
    ,

    isMultipartCandidate: function(executes) {
        if (this._Lang.isString(executes)) {
            executes = this._Lang.strToArray(executes, /\s+/);
        }

        for (var exec in executes) {
            var element = this.byId(executes[exec]);
            var inputs = this.findByTagName(element, "input", true);
            for (var key in inputs) {
                if (this.getAttribute(inputs[key], "type") == "file") return true;
            }
        }
        return false;
    }
    ,


    insertFirst: function(newNode) {
        var body = document.body;
        if (body.childNodes.length > 0) {
            body.insertBefore(newNode, body.firstChild);
        } else {
            body.appendChild(newNode);
        }
    }
    ,

    byId: function(id) {
        return this._Lang.byId(id);
    }
    ,

    getDummyPlaceHolder: function(markup) {
        var created = false;
        if (!this._dummyPlaceHolder) {
            this._dummyPlaceHolder = document.createElement("div");
            created = true;
        }

        //ieMobile in its 6.1-- incarnation cannot handle innerHTML detached objects so we have
        //to attach the dummy placeholder, we try to avoid it for
        //better browsers so that we do not have unecessary dom operations
        if (this._RT.browser.isIEMobile && created) {
            this.insertFirst(this._dummyPlaceHolder);

            this.setAttribute(this._dummyPlaceHolder, "style", "display: none");

        }

        return this._dummyPlaceHolder;
    }
}
        )
        ;



/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

/**
 * <p>
 *  Fallback routine if the browser embedded xml parser fails on the document
 *  This fallback is not failsafe but should give enough cover to handle all cases
 * </p>
 */

/** @namespace myfaces._impl._util._HtmlStripper */
myfaces._impl.core._Runtime.extendClass("myfaces._impl._util._HtmlStripper", Object, {
    BEGIN_TAG: "html",
    END_TAG: "lmth",

    parse : function(theString, tagNameStart) {
        this.tokens = theString.split("");
        this.tagAttributes = {};

        this._tagStart = -1;
        this._tagEnd = -1;

        this._contentStart = -1;
        this._contentEnd = -1;
        this._tokenPos = 0;

        this._tokenForward = 1;

        this.tagNameStart = (!tagNameStart) ? this.BEGIN_TAG : tagNameStart;

        //no need for ll parsing a handful of indexofs instead of slower regepx suffices

        var proposedTagStartPos = theString.indexOf("<"+tagNameStart);

        while(this._contentStart == -1 && proposedTagStartPos != -1) {
            if(this.checkBackForComment(theString, proposedTagStartPos))  {
                this._tagStart = proposedTagStartPos;
                this._contentStart = proposedTagStartPos+theString.substring(proposedTagStartPos).indexOf(">")+1;
            }
            proposedTagStartPos = theString.substring(proposedTagStartPos+tagNameStart.length+2).indexOf("<"+tagNameStart);
        }

        var proposedEndTagPos = theString.lastIndexOf("</"+tagNameStart);
        while(this._contentEnd == -1 && proposedEndTagPos > 0) {
            if(this.checkForwardForComment(theString, proposedEndTagPos))  {
                this._tagEnd = proposedEndTagPos;
                this._contentEnd = proposedEndTagPos;
            }
            proposedTagStartPos = theString.substring(proposedTagStartPos-tagNameStart.length-2).lastIndexOf("</"+tagNameStart);
        }
        if(this._contentStart != -1 && this._contentEnd != -1) {
            return theString.substring(this._contentStart, this._contentEnd);
        }
        return null;
    },
    
    checkForwardForComment: function(theStr, tagPos) {
        var toCheck = theStr.substring(tagPos);
        var firstBeginComment = toCheck.indexOf("<!--");
        var firstEndComment = toCheck.indexOf("-->");

        var firstBeginCDATA = toCheck.indexOf("<[CDATA[");
        var firstEndCDATA = toCheck.indexOf("]]>");
        
        if(this.isValidPositionCombination(firstBeginComment, firstEndComment, firstBeginCDATA, firstEndCDATA)) {
            return true;
        }

        return firstBeginComment <= firstEndComment && firstBeginCDATA <= firstEndCDATA;
    },

    checkBackForComment: function(theStr, tagPos) {
        var toCheck = theStr.substring(tagPos);
        var lastBeginComment = toCheck.lastIndexOf("<!--");
        var lastEndComment = toCheck.lastIndexOf("-->");

        var lastBeginCDATA = toCheck.lastIndexOf("<[CDATA[");
        var lastEndCDATA = toCheck.lastIndexOf("]]>");


        if(this.isValidPositionCombination(lastBeginComment, lastEndComment, lastBeginCDATA, lastEndCDATA)) {
            //TODO we have to handle the embedded cases, for now we leave them out
            return true;
        }

    },

    isValidPositionCombination: function(pos1, pos2, pos3, pos4) {
        return pos1 <= pos2 && pos3 <= pos4;
    },

    isFullyEmbedded: function(pos1, pos2, embedPos1, embedPos2) {
        return embedPos1 < pos1 < pos2 < embedPos2;
    },

    isPartiallyEmbedded: function(pos1, pos2, embedPos1, embedPos2) {
        return embedPos1 < pos1 <  embedPos2 < pos2 || pos1 < embedPos1 < pos2 <  embedPos2  ;    
    }

});




/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * legacy code to enable various aspects
 * of myfaces, used to be rendered inline
 * for jsf 2.0 we can externalized it into its own custom resource
 */

(!window.myfaces) ? window.myfaces = {} : null;
if (!myfaces.oam) {
    myfaces.oam = new function() {

        /**
         * sets a hidden input field
         * @param formname the formName
         * @param name the hidden field
         * @param value the value to be rendered
         */
        this.setHiddenInput = function(formname, name, value) {
            var form = document.forms[formname];
            if (typeof form == 'undefined') {
                form = document.getElementById(formname);
            }
    
            if (typeof form.elements[name] != 'undefined' && (form.elements[name].nodeName == 'INPUT' || form.elements[name].nodeName == 'input')) {
                form.elements[name].value = value;
            }
            else {
                var newInput = document.createElement('input');
                newInput.setAttribute('type', 'hidden');
                newInput.setAttribute('id', name);
                newInput.setAttribute('name', name);
                newInput.setAttribute('value', value);
                form.appendChild(newInput);
            }
        };
        
        /**
         * clears a hidden input field
         *
         * @param formname formName for the input
         * @param name the name of the input field
         * @param value the value to be cleared
         */
        this.clearHiddenInput = function(formname, name, value) {
            var form = document.forms[formname];
    
            if (typeof form == 'undefined') {
                form = document.getElementById(formname);
            }
    
            var hInput = form.elements[name];
            if (typeof hInput != 'undefined') {
                form.removeChild(hInput);
            }
        };
        
        /**
         * does special form submit remapping
         * remaps the issuing command link into something
         * the decode of the command link on the server can understand
         *
         * @param formName
         * @param linkId
         * @param target
         * @param params
         */
        this.submitForm = function(formName, linkId, target, params) {
    
            var clearFn = 'clearFormHiddenParams_' + formName.replace(/-/g, '\$:').replace(/:/g, '_');
            if (typeof window[clearFn] == 'function') {
                window[clearFn](formName);
            }
    
            var form = document.forms[formName];
            if (typeof form == 'undefined') {
                form = document.getElementById(formName);
            }
    
            //autoscroll code
            if (myfaces.core.config.autoScroll && typeof window.getScrolling != 'undefined') {
                myfaces.oam.setHiddenInput(formName, 'autoScroll', getScrolling());
            }
    
            if (myfaces.core.config.ieAutoSave) {
                var agentString = navigator.userAgent.toLowerCase();
                var version = navigator.appVersion;
                if (agentString.indexOf('msie') != -1) {
                    if (!(agentString.indexOf('ppc') != -1 && agentString.indexOf('windows ce') != -1 && version >= 4.0)) {
                        window.external.AutoCompleteSaveForm(form);
                    }
                }
            }
    
            var oldTarget = form.target;
            if (target != null) {
                form.target = target;
            }
            if ((typeof params != 'undefined') && params != null) {
                for (var i = 0, param; (param = params[i]); i++) {
                    myfaces.oam.setHiddenInput(formName, param[0], param[1]);
                }
    
            }
    
            myfaces.oam.setHiddenInput(formName, formName + ':' + '_idcl', linkId);
    
            if (form.onsubmit) {
                var result = form.onsubmit();
                if ((typeof result == 'undefined') || result) {
                    try {
                        form.submit();
                    }
                    catch(e) {
                    }
                }
    
            }
            else {
                try {
                    form.submit();
                }
                catch(e) {
                }
            }
    
            form.target = oldTarget;
            if ((typeof params != 'undefined') && params != null) {
    
                for (var i = 0, param; (param = params[i]); i++) {
                    myfaces.oam.clearHiddenInput(formName, param[0], param[1]);
                }
    
            }
    
            myfaces.oam.clearHiddenInput(formName, formName + ':' + '_idcl', linkId);
            return false;
        };
    }
}

//reserve a cofig namespace for impl related stuff
(!myfaces.core) ? myfaces.core = {} : null;
(!myfaces.core.config) ? myfaces.core.config = {} : null;


/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 *  A helper object which supports IE with its non working garbage collection.
 *
 * See also:
 *
 * http://msdn.microsoft.com/en-us/library/bb250448%28VS.85%29.aspx
 * http://weblogs.java.net/blog/driscoll/archive/2009/11/13/ie-memory-management-and-you
 * http://home.orange.nl/jsrosman/
 * http://www.quirksmode.org/blog/archives/2005/10/memory_leaks_li.html
 * http://www.josh-davis.org/node/7
 *
 * Author: Werner Punz (latest modification by $Author: werpu $)
 * Version: $Revision: 1.4 $ $Date: 2009/05/31 09:16:44 $
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._FinalizeableObj", Object, {

    _resettableContent: null,

    constructor_: function() {
        this._resettableContent={};
    },

    _initDefaultFinalizableFields: function() {
         for(var key in this) {
            //per default we reset everything which is not preinitalized
            if(null == this[key] && key != "_resettableContent" && key.indexOf("_mf") != 0 && key.indexOf("_") == 0) {
                this._resettableContent[key]=true;
            }
        }
    },

    /**
     * ie6 cleanup
     */
    _finalize: function() {
        if (!myfaces._impl.core._Runtime.browser.isIE || !this._resettableContent) {
            //no ie, no broken garbage collector
            return;
        }
       
        for(var key in this._resettableContent) {
            if (myfaces._impl.core._Runtime.exists(this[key],"_finalize")) {
                this[key]._finalize();
            }
            delete this[key];
        }
    }
});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** @namespace myfaces._impl.xhrCore._AjaxUtils */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._AjaxUtils", myfaces._impl.xhrCore._FinalizeableObj, {
      _processedExceptions: {},

    /**
     * Constructor
     * @param {function} onException - exception handler
     * @param {function} onWarning - warning handler
     */
    constructor_ : function(onException, onWarning) {
        this._onException = onException;
        this._onWarning = onWarning;
    },

    /**
     * determines fields to submit
     * @param {Object} request the xhr request object
     * @param {Object} context (Map)
     * @param {Node} item - item that triggered the event
     * @param {Node} parentItem - form element item is nested in
     * @param {Array} partialIds - ids fo PPS
     */
    encodeSubmittableFields : function(targetBuf, request, context, item,
                                       parentItem, partialIds) {

        try {
            if (!parentItem) {
                this._onWarning(request, context, "myfaces._impl.xhrCore._AjaxUtils", "encodeSubmittableFields " + "Html-Component is not nested in a Form-Tag");
                return null;
            }

            if (partialIds && partialIds.length > 0) {
                this.encodePartialSubmit(parentItem, item, false, partialIds, targetBuf);
            } else {
                // add all nodes
                var eLen = parentItem.elements.length;
                for (var e = 0; e < eLen; e++) {
                    this.encodeElement(parentItem.elements[e], targetBuf);
                } // end of for (formElements)
            }

            this.appendIssuingItem(item, targetBuf);
        } catch (e) {
            this._onException(request, context, "myfaces._impl.xhrCore._AjaxUtils", "encodeSubmittableFields", e);
        }
    },

    /**
     * checks recursively if contained in PPS
     * the algorithm is as follows we have an issuing item
     * the parent form of the issuing item and a set of child ids which do not
     * have to be inputs, we scan now for those ids and all inputs which are childs
     * of those ids
     *
     * Now this algorithm is up for discussion because it is relatively complex
     * but for now we will leave it as it is.
     *
     * @param {Node} node - the root node of the partial page submit
     * @param {boolean} submitAll - if set to true, all elements within this node will
     * be added to the partial page submit
     * @param {Array} partialIds - an array of partial ids which should be used for the submit
     * @param {Array} targetBuf a target string buffer which receives the encoded elements
     */
    encodePartialSubmit : function(node, issuingItem, submitAll,
                                   partialIds, targetBuf) {
        var _Lang = myfaces._impl._util._Lang;
        var _Impl = myfaces._impl.core.Impl;
        var _Dom = myfaces._impl._util._Dom;

        var partialIdsFilter = function(curNode) {
            if (curNode.nodeType != 1) return false;
            if (submitAll && node != curNode) return true;

            var id = curNode.id || curNode.name;

            return (id && _Lang.contains(partialIds, id)) || id == _Impl.P_VIEWSTATE;
        };

        //shallow scan because we have a second scanning step, to find the encodable childs of
        //the result nodes, that way we can reduce the number of nodes
        var nodes = _Dom.findAll(node, partialIdsFilter, false);

        var allowedTagNames = {"input":true, "select":true, "textarea":true};

        if (nodes && nodes.length) {
            for (var cnt = 0; cnt < nodes.length; cnt++) {
                //we can shortcut the form any other nodetype
                //must get a separate investigation
                var subNodes = (nodes[cnt].tagName.toLowerCase() == "form") ?
                        node.elements :
                        _Dom.findByTagNames(nodes[cnt], allowedTagNames, true);

                if (subNodes && subNodes.length) {
                    for (var cnt2 = 0; cnt2 < subNodes.length; cnt2++) {
                        this.encodeElement(subNodes[cnt2], targetBuf);
                    }
                } else {
                    this.encodeElement(nodes[cnt], targetBuf);
                }
            }
        }

        this.appendViewState(node, targetBuf);
    },

    /**
     * appends the viewstate element if not given already
     *
     * @param parentNode
     * @param targetBuf
     *
     * TODO dom level2 handling here, for dom level2 we can omit the check and readd the viewstate
     */
    appendViewState: function(parentNode, targetBuf) {
        var _Dom = myfaces._impl._util._Dom;
        var _Impl = myfaces._impl.core.Impl;

        //viewstate covered, do a preemptive check
        if (targetBuf.hasKey(_Impl.P_VIEWSTATE)) return;

        var viewStates = _Dom.findByName(parentNode, _Impl.P_VIEWSTATE, true);
        if (viewStates && viewStates.length) {
            for (var cnt2 = 0; cnt2 < viewStates.length; cnt2++) {
                this.encodeElement(viewStates[cnt2], targetBuf);
            }
        }
    },

    /**
     * appends the issuing item if not given already
     * @param item
     * @param targetBuf
     */
    appendIssuingItem: function (item, targetBuf) {
        // if triggered by a Button send it along
        if (item && item.type && item.type.toLowerCase() == "submit") {
            targetBuf.append(item.name, item.value);
        }
    },


    /**
     * encodes a single input element for submission
     *
     * @param {Node} element - to be encoded
     * @param {} targetBuf - a target array buffer receiving the encoded strings
     */
    encodeElement : function(element, targetBuf) {

        //browser behavior no element name no encoding (normal submit fails in that case)
        //https://issues.apache.org/jira/browse/MYFACES-2847
        if (!element.name) {
            return;
        }

        var _RT = myfaces._impl.core._Runtime;
        var name = element.name;
        var tagName = element.tagName.toLowerCase();
        var elemType = element.type;
        if (elemType != null) {
            elemType = elemType.toLowerCase();
        }

        // routine for all elements
        // rules:
        // - process only inputs, textareas and selects
        // - elements muest have attribute "name"
        // - elements must not be disabled
        if (((tagName == "input" || tagName == "textarea" || tagName == "select") &&
                (name != null && name != "")) && !element.disabled) {

            // routine for select elements
            // rules:
            // - if select-one and value-Attribute exist => "name=value"
            // (also if value empty => "name=")
            // - if select-one and value-Attribute don't exist =>
            // "name=DisplayValue"
            // - if select multi and multple selected => "name=value1&name=value2"
            // - if select and selectedIndex=-1 don't submit
            if (tagName == "select") {
                // selectedIndex must be >= 0 sein to be submittet
                if (element.selectedIndex >= 0) {
                    var uLen = element.options.length;
                    for (var u = 0; u < uLen; u++) {
                        // find all selected options
                        //var subBuf = [];
                        if (element.options[u].selected) {
                            var elementOption = element.options[u];
                            targetBuf.append(name, (elementOption.getAttribute("value") != null) ?
                                    elementOption.value : elementOption.text);
                        }
                    }
                }
            }

            // routine for remaining elements
            // rules:
            // - don't submit no selects (processed above), buttons, reset buttons, submit buttons,
            // - submit checkboxes and radio inputs only if checked
            if ((tagName != "select" && elemType != "button"
                    && elemType != "reset" && elemType != "submit" && elemType != "image")
                    && ((elemType != "checkbox" && elemType != "radio") || element.checked)) {
                if ('undefined' != typeof element.files && element.files != null && _RT.getXHRLvl() >= 2 && element.files.length) {
                    //xhr level2
                    targetBuf.append(name, element.files[0]);
                } else {
                    targetBuf.append(name, element.value);
                }
            }

        }
    },

    _finalize: function() {
        delete this._onException;
        delete this._onWarning;
    }

});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/
/** @namespace myfaces._impl.xhrCore._AjaxRequestQueue */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._AjaxRequestQueue", myfaces._impl._util._Queue, {

    /**
     * a pointer towards the currently processed
     * request in our queue
     */
    _curReq : null,

    /**
     * the standard constructur of our class
     */
    constructor_: function() {
        this._callSuper("constructor");
    },

    /**
     * delay request, then call enqueue
     * @param {Object} request (myfaces._impl.xhrCore._AjaxRequest) request to send
     */
    enqueue : function(request) {
        if (typeof request._delay == "number") {
            this.clearDelayTimeout();
            var _Lang = myfaces._impl._util._Lang;
            this._delayTimeoutId = window.setTimeout(
                    _Lang.hitch(this, function() {
                        this.clearDelayTimeout();
                        //lets clear the delay time to enqueue correctly 
                        delete request._delay;
                        this.enqueue(request);
                    }), request._delay);
        } else {
            if (this._curReq == null) {
                this._curReq = request;
                this._curReq.send();
            } else {
                this._callSuper("enqueue", request);
                if (request._queueSize != this._size) {
                    this.setQueueSize(request._queueSize);
                }
            }
        }
    },

    /**
     * timeout clearing routine
     * for timeout requests
     */
    clearDelayTimeout : function() {
        try {
            if (typeof this._delayTimeoutId == "number") {
                window.clearTimeout(this._delayTimeoutId);
                delete this._delayTimeoutId;
            }
        } catch (e) {
            // already timed out
        }
    },


    /**
     * process queue, send request, if exists
     */
    processQueue: function() {
        this._curReq = this.dequeue();
        if (this._curReq) {
            this._curReq.send();
        }
    },

    /**
     * cleanup queue
     */
    cleanup: function() {
        this._curReq = null;
        this._callSuper("cleanup");
    }
});


/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 *
 * Abstract base request encapuslating all methods and variables
 * shared over all different request objects
 *
 * Author: Werner Punz (latest modification by $Author: ganeshpuri $)
 * Version: $Revision: 1.4 $ $Date: 2009/05/31 09:16:44 $
 */
/** @namespace myfaces._impl.xhrCore._AjaxRequest */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._BaseRequest", myfaces._impl.xhrCore._FinalizeableObj, {

    _Dom: myfaces._impl._util._Dom,
    _Lang: myfaces._impl._util._Lang,
    _RT: myfaces._impl.core._Runtime,

    _contentType: "application/x-www-form-urlencoded",
    _source: null,
    _xhr: null,
    _encoding:null ,

    _context:null,
    _ajaxUtil: null,
    _sourceForm: null,
    _passThrough: null,
    _requestParameters: null,
    _exception: null,
    _timeout: null,
    _delay:null,
    _queueSize:-1,

    _partialIdsArray : null,
    //callbacks for onDone... done issues
    //onSuccess everything has passed through
    //onError server side error

    //onException exception thrown by the client
    //onWarning warning issued by the client
    _onDone : null,
    _onSuccess: null,
    _onError: null,
    _onException: null,
    _onWarning: null,
    _onTimeout:null,

    /*response object which is exposed to the queue*/
    _response: null,

    _timeoutId: null,

    /*all instance vars can be set from the outside
     * via a parameter map*/
    _ajaxType: "POST",

    /*
     * constants used internally
     */
    _CONTENT_TYPE:"Content-Type",
    _HEAD_FACES_REQ:"Faces-Request",

    _READY_STATE_DONE: 4,
    _STATUS_OK_MINOR: 200,
    _STATUS_OK_MAJOR: 300,

    _VAL_AJAX: "partial/ajax",

    


    //abstract methods which have to be implemented
    //since we do not have abstract methods we simulate them
    //by using empty ones
    constructor_: function() {
        this._callSuper("constructor");
        this._Lang = myfaces._impl._util._Lang;
        this._Dom = myfaces._impl._util._Dom;
        this._initDefaultFinalizableFields();
    },

    /**
     * send the request
     */
    send: function() {
    },

    /**
     * the callback function after the request is done
     */
    callback: function() {

    },

    //non abstract ones
    /**
     * Spec. 13.3.1
     * Collect and encode input elements.
     * Additionally the hidden element javax.faces.ViewState
     *
     *
     * @return  an element of formDataWrapper
     * which keeps the final Send Representation of the
     */
    getViewState : function() {
        var ret = this._Lang.createFormDataDecorator(new Array());

        this._ajaxUtil.encodeSubmittableFields(ret, this._xhr, this._context, this._source,
                this._sourceForm, this._partialIdsArray);
       
        return ret;
    }

    
});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * An implementation of an xhr request object
 * with partial page submit functionality, and jsf
 * ppr request and timeout handling capabilities
 *
 * Author: Ganesh Jung (latest modification by $Author: ganeshpuri $)
 * Version: $Revision: 1.4 $ $Date: 2009/05/31 09:16:44 $
 */
/** @namespace myfaces._impl.xhrCore._AjaxRequest */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._AjaxRequest", myfaces._impl.xhrCore._BaseRequest, {


    /*
     note there is a load of common properties
     inherited by the base class which define the corner
     parameters and the general internal behavior
     like _onError etc...
     */

    /**
     * Constructor
     * @arguments  an arguments map which an override any of the given protected
     * instance variables, by a simple name value pair combination
     *
     */
    constructor_: function(arguments) {


        try {
            this._callSuper("constructor", arguments);
            /*namespace remapping for readability*/
            //we fetch in the standard arguments
            //and apply them to our protected attributes
            this._Lang.applyArgs(this, arguments);

            //if our response handler is not set
            if (!this._response) {
                this._response = new myfaces._impl.xhrCore._AjaxResponse(this._onException, this._onWarning);
            }

            this._ajaxUtil = new myfaces._impl.xhrCore._AjaxUtils(this._onException, this._onWarning);
        } catch (e) {
            //_onError
            this._onException(this._context, "myfaces._impl.xhrCore._AjaxRequest", "constructor", e);
        }
    },

    /**
     * Sends an Ajax request
     */
    send : function() {
        try {
            //we have to encode at send time, otherwise
            //we pick up old viewstates
            this._initRequestParams();
            this._startXHR();
            this._startTimeout();
        } catch (e) {
            //_onError//_onError
            this._onException(this._xhr, this._context, "myfaces._impl.xhrCore._AjaxRequest", "send", e);
        }
    },


    /**
     * gets the ViewState including all now passed in sideconstraints
     */
    _initRequestParams: function() {
        this._requestParameters = this.getViewState();
        for (var key in this._passThrough) {
            this._requestParameters.append(key, this._passThrough[key]);
        }
    },

    /**
     * starts the asynchronous xhr request
     */
    _startXHR: function() {
        this._preCreateXHR();
        this._xhr = myfaces._impl.core._Runtime.getXHRObject();
        this._postCreateXHR();

        var targetURL;
        if (typeof this._sourceForm.elements["javax.faces.encodedURL"] == 'undefined') {
              targetURL = this._sourceForm.action;
        } else {
              targetURL = this._sourceForm.elements["javax.faces.encodedURL"].value;
        }
        
        this._xhr.open(this._ajaxType, targetURL +
                ((this._ajaxType == "GET")? "?"+this._requestParameters.makeFinal():"")
                , true);

        var contentType = this._contentType;
        if (this._encoding) {
            contentType = contentType + "; charset:" + this._encoding;
        }

        this._xhr.setRequestHeader(this._CONTENT_TYPE, this._contentType);
        this._xhr.setRequestHeader(this._HEAD_FACES_REQ, this._VAL_AJAX);

        this._xhr.onreadystatechange = this._Lang.hitch(this, this.callback);
        var _Impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
        _Impl.sendEvent(this._xhr, this._context, myfaces._impl.core.Impl.BEGIN);

        this._preSend();

        try {
            this._xhr.send((this._ajaxType != "GET")? this._requestParameters.makeFinal():null);
        } finally {
            this._postSend();
        }
    },



    /**
     * starts the timeout
     * which is able to terminate the xhr upfront early
     */
    _startTimeout: function() {
        if (this._timeout && this._onTimeout) {
            var _req = this._xhr;
            var _context = this._context;
            if (this._timeoutId) {
                window.clearTimeout(this._timeoutId);
                this._timeoutId = null;
            }
            this._timeoutId = window.setTimeout(
                //we unify the api, there must be always a request passed to the external function
                //and always a context, no matter what
                    this._Lang.hitch(this,
                                    function() {
                                        //the hitch has to be done due to the setTimeout refocusing the scope of this
                                        //to window
                                        try {
                                            _req.onreadystatechange = function() {
                                            };

                                            //to avoid malformed whatever, we have
                                            //the timeout covered already on the _onTimeout function
                                            _req.abort();
                                            this._onTimeout(_req, _context);
                                        } catch (e) {
                                            alert(e);
                                        } finally {
                                        }
                                    })
                    , this._timeout);
        }
    },

    /**
     * Callback method to process the Ajax response
     * triggered by RequestQueue
     */
    callback : function() {
        
        try {
            var _Impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);

            if (this._xhr.readyState == this._READY_STATE_DONE) {
                if (this._timeoutId) {
                    //normally the timeout should not cause anything anymore
                    //but just to make sure
                    window.clearTimeout(this._timeoutId);
                    this._timeoutId = null;
                }
                this._onDone(this._xhr, this._context);
                if (this._xhr.status >= this._STATUS_OK_MINOR && this._xhr.status < this._STATUS_OK_MAJOR) {
                    this._onSuccess(this._xhr, this._context);
                } else {
                    this._onError(this._xhr, this._context);
                }
            }
        } catch (e) {
            if (this._onException)
                this._onException(this._xhr, this._context, "myfaces._impl.xhrCore._AjaxRequest", "callback", e);
            else
                alert(e.toString());
        } finally {
            //final cleanup to terminate everything
            this._Lang.clearExceptionProcessed();

            //this._context.source;
            if (this._xhr.readyState == this._READY_STATE_DONE) {
                this._callSuper("_finalize");
            }


        }
    },

    /*
     * various lifecycle callbacks which can be used by differing AjaxRequest impls
     * (namely level 1.5 (Mozilla XHR) and level 2 (html5 xhr) to run special initialisation code
     **/
    _preCreateXHR : function() {
        //called after the xhr object has been created
    },

    _postCreateXHR : function() {
        //called after the xhr object has been created
    },
    _preSend : function() {
        //called before the xhr object is sent
    },
    _postSend : function() {
        //called after the xhr object is sent for cleanup purposes
    }

});


/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * iframe request for communications over iframe
 *
 * This method can be used by older browsers and if you have
 * a multipart request which includes
 * a fileupload element, fileuploads cannot be handled by
 * normal xhr requests. The standard html 4+ compliant way to do this
 * is to use an iframe as submit target for a form.
 *
 * Note on almost all browsers this method induces a real asynchronity, the only
 * exception is firefox 3.6- which blocks the ui, this is resolved in Firefox 4
 */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._IFrameRequest", myfaces._impl.xhrCore._BaseRequest, {

    _FRAME_ID: "_mf_comm_frm",
    _frame: null,
    _RT: myfaces._impl.core._Runtime,
    CLS_NAME: "myfaces._impl.xhrCore._IFrameRequest",
    JX_PART_IFRAME: "javax.faces.partial.iframe",
    MF_PART_IFRAME: "org.apache.myfaces.partial.iframe",

    /**
     * constructor which shifts the arguments
     * to the protected properties of this clas
     *
     * @param arguments
     */
    constructor_: function(arguments) {
        try {
            //we fetch in the standard arguments
            this._callSuper("constructor", arguments);
            this._Lang.applyArgs(this, arguments);

            if (!this._response) {
                this._response = new myfaces._impl.xhrCore._AjaxResponse(this._onException, this._onWarning);
            }
            this._ajaxUtil = new myfaces._impl.xhrCore._AjaxUtils(this._onException, this._onWarning);
        } catch (e) {
            //_onError
            this._onException(null, this._context, this.CLS_NAME, "constructor", e);
        }
    },

    /**
     * send method, central callback which sends the
     * request
     */
    send: function() {
        var _Impl = this._RT.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
        var _RT = myfaces._impl.core._Runtime;

        this._frame = this._createTransportFrame();

        //we append an onload handler to the frame
        //to cover the starting and loading events,
        //timeouts cannot be covered in a cross browser way

        //we point our onload handler to the frame, we do not use addOnLoad
        //because the frame should not have other onload handlers in place
        if (!_RT.browser.isIE) {
            this._frame.onload = this._Lang.hitch(this, this.callback);
        } else {
            //ie has a bug, onload is not settable outside of innerHTML on iframes
            this._frame.onload_IE = this._Lang.hitch(this, this.callback);
        }
       
        //now to the parameter passing:
        _Impl.sendEvent(this._xhr, this._context, _Impl.BEGIN);

        //viewstate should be in our parent form which we will isse we however have to add the execute and
        //render parameters as well as the usual javax.faces.request params to our target

        var oldTarget = this._sourceForm.target;
        var oldMethod = this._sourceForm.method;
        var _progress = 0;
        var _srcFrm = this._sourceForm;
        try {
            this._initAjaxParams();
            _srcFrm.target = this._frame.name;
            _srcFrm.method = this._ajaxType;
            _srcFrm.submit();
        } finally {
            this._removeAjaxParams(oldTarget);
            _srcFrm.target = oldTarget;
            _srcFrm.method = oldMethod;
        }
    },

    /**
     * the callback function after the request is done
     */
    callback: function() {
        //now we have to do the processing, for that we have to parse the result, if it is a http 404 then
        //nothing could be delivered and we bomb out with an error anything else has to be parsed
        //via our xml parser
        var request = {};
        try {
            request.responseText = this._getFrameText();
            request.responseXML = this._getFrameXml();
            request.readyState = this._READY_STATE_DONE;

            this._onDone(request, this._context);

            if (!this._Lang.isXMLParseError(request.responseXML)) {
                request.status = 201;
                this._onSuccess(request, this._context);
            } else {
                request.status = 0;
                this._onError(request, this._context);
            }
        } catch (e) {
            //_onError
            this._onException(null, this._context, this.CLS_NAME, "constructor", e);
        } finally {
            //this closes any hanging or pending comm channel caused by the iframe
            this._clearFrame();
            this._frame = null;
        }
    },

    /**
     * returns the frame text in a browser independend manner
     */
    _getFrameDocument: function() {
      //we cover various browsers here, because almost all browsers keep the document in a different
      //position
      return this._frame.contentWindow.document || this._frame.contentDocument || this._frame.document  ;
    },

    _getFrameText: function() {
        var framedoc = this._getFrameDocument();
        //also ie keeps the body in framedoc.body the rest in documentElement
        var body = framedoc.body || framedoc.documentElement ;
        return  body.innerHTML;
    },

    _clearFrame: function() {
        var framedoc = this._getFrameDocument();
        var body = framedoc.documentElement || framedoc.body;
        //ie8 in 7 mode chokes on the innerHTML method
        //direct dom removal is less flakey and works
        //over all browsers, but is slower
        this._Dom._removeChildNodes(body, false);
    },

    /**
     * returns the processed xml from the frame
     */
    _getFrameXml: function() {
        var framedoc = this._getFrameDocument();
        //same situation here, the xml is hosted either in xmlDocument or
        //is located directly under the frame document
        return  framedoc.XMLDocument ||  framedoc;
    },


    _initAjaxParams: function() {
        var _Impl = myfaces._impl.core.Impl;
        //this._appendHiddenValue(_Impl.P_AJAX, "");
        var appendHiddenValue = this._Lang.hitch(this, this._appendHiddenValue);
        for (var key in this._passThrough) {

            appendHiddenValue(key, this._passThrough[key]);
        }
        //marker that this is an ajax iframe request
        appendHiddenValue(this.JX_PART_IFRAME, "true");
        appendHiddenValue(this.MF_PART_IFRAME, "true");

    },

    _removeAjaxParams: function(oldTarget) {
        var _Impl = myfaces._impl.core.Impl;
        this._sourceForm.target = oldTarget;
        //some browsers optimize this and loose their scope that way,
        //I am still not sure why, but probably because the function itself
        //was called under finally and I ran into a bug in the fox 4
        //scripting engine
        var toDelete = [];
        var possibleKeys = {};
        for(var key in this._passThrough) {
            possibleKeys[key] = true;
        }
        possibleKeys[this.JX_PART_IFRAME] = true;
        possibleKeys[this.MF_PART_IFRAME] = true;
        (possibleKeys["javax.faces.ViewState"])? delete possibleKeys["javax.faces.ViewState"]:null;

        for(var cnt = this._sourceForm.elements.length -1; cnt >= 0; cnt--) {
            var elem = this._sourceForm.elements[cnt];
            if(possibleKeys[elem.name] && elem.type == "hidden") {
                elem.parentNode.removeChild(elem);
                delete elem;
            }
        }
    },

    _appendHiddenValue: function(key, value) {
        if ('undefined' == typeof value) {
            return;
        }
        var input = document.createElement("input");
        //the dom is a singleton nothing can happen by remapping
        this._Dom.setAttribute(input, "type", "hidden");
        this._Dom.setAttribute(input, "name", key);
        this._Dom.setAttribute(input, "style", "display:none");
        this._Dom.setAttribute(input, "value", value);
        this._sourceForm.appendChild(input);
    },

    _removeHiddenValue: function(key) {
        var elem = this._Dom.findByName(this._sourceForm, key, true);
        if (elem.length) {

            elem[0].parentNode.removeChild(elem[0]);
            delete elem[0];
        }
    },

    _createTransportFrame: function() {
        var _RT = this._RT;
        var frame = document.getElementById(this._FRAME_ID);
        //normally this code should not be called
        //but just to be sure
        if (!frame) {
            if (!_RT.browser.isIE) {
                frame = document.createElement('iframe');

                //probably the ie method would work on all browsers
                //but this code is the safe bet it works on all standards
                //compliant browsers in a clean manner

                this._Dom.setAttribute(frame, "src", "about:blank");
                this._Dom.setAttribute(frame, "id", this._FRAME_ID);
                this._Dom.setAttribute(frame, "name", this._FRAME_ID);
                this._Dom.setAttribute(frame, "type", "content");
                this._Dom.setAttribute(frame, "collapsed", "true");
                this._Dom.setAttribute(frame, "style", "display:none");

                document.body.appendChild(frame);
            } else { //Now to the non compliant browsers
                var node = document.createElement("div");
                this._Dom.setAttribute(node, "style", "display:none");
                //we are dealing with two well known iframe ie bugs here
                //first the iframe has to be set via innerHTML to be present
                //secondly the onload handler is immutable on ie, we have to
                //use a dummy onload handler in this case and call that one
                //from the onload handler
                node.innerHTML = "<iframe id='" + this._FRAME_ID + "' name='" + this._FRAME_ID + "' style='display:none;' src='about:blank' type='content' onload='this.onload_IE();'  ></iframe>";

                //avoid the ie open tag problem
                var body = document.body;
                if (body.firstChild) {
                    body.insertBefore(node, document.body.firstChild);
                } else {
                    body.appendChild(node);
                }
            }

        }
        //helps to for the onload handlers and innerhtml to be in sync again
        return document.getElementById(this._FRAME_ID);

    }

    //TODO pps, the idea behind pps is to generate another form
    // and temporarily shift the elements over which have to be
    // ppsed, but it is up for discussion if we do pps at all in case of
    // an iframe, so I wont implement anything for now
});
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** @namespace myfaces._impl.xhrCore._AjaxResponse */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._AjaxResponse", myfaces._impl.xhrCore._FinalizeableObj, {

    /*partial response types*/
    RESP_PARTIAL : "partial-response",
    RESP_TYPE_ERROR : "error",
    RESP_TYPE_REDIRECT : "redirect",
    RESP_TYPE_CHANGES : "changes",

    /*partial commands*/
    CMD_CHANGES : "changes",
    CMD_UPDATE : "update",
    CMD_DELETE : "delete",
    CMD_INSERT : "insert",
    CMD_EVAL : "eval",
    CMD_ERROR : "error",
    CMD_ATTRIBUTES : "attributes",
    CMD_EXTENSION : "extension",
    CMD_REDIRECT : "redirect",

    /*other constants*/
    P_VIEWSTATE: "javax.faces.ViewState",
    P_VIEWROOT: "javax.faces.ViewRoot",
    P_VIEWHEAD: "javax.faces.ViewHead",
    P_VIEWBODY: "javax.faces.ViewBody",


    /**
     * Constructor
     * @param {function} base request classed parent object
     * @param {function} onException
     * @param {function} onWarning
     */
    constructor_: function(onException, onWarning) {
        //List of non form elements to be updated (which can have forms embedded)
        this._updateElems = [];
        // List of forms to be updated if any inner block is updated
        this._updateForms = [];
        this._onException = onException;
        this._onWarning = onWarning;

        this.appliedViewState = null;

        this._Lang = myfaces._impl._util._Lang;
        this._Dom = myfaces._impl._util._Dom;
        this._RT = myfaces._impl.core._Runtime;
        this._Impl = this._RT.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
    },
    /**
     * uses response to start Html element replacement
     *
     * @param {Object} request (xhrRequest) - xhr request object
     * @param {Object} context (Map) - AJAX context
     *
     * A special handling has to be added to the update cycle
     * according to the JSDoc specs if the CDATA block contains html tags the outer rim must be stripped
     * if the CDATA block contains a head section the document head must be replaced
     * and if the CDATA block contains a body section the document body must be replaced!
     *
     */
    processResponse : function(request, context) {
        //mfinternal handling, note, the mfinternal is only optional
 	   //according to the spec
        context._mfInternal =  context._mfInternal || {};
        try {
            var _Impl = this._Impl;

            // TODO:
            // Solution from
            // http://www.codingforums.com/archive/index.php/t-47018.html
            // to solve IE error 1072896658 when a Java server sends iso88591
            // istead of ISO-8859-1

            if (!request) {
                throw Exception(this._Lang.getMessage("ERR_EMPTY_RESPONSE",null,"jsf.ajaxResponse"));
            }

            if (!this._Lang.exists(request, "responseXML")) {
                _Impl.sendError(request, context, myfaces._impl.core.Impl.EMPTY_RESPONSE);
                return;
            }
            //check for a parseError under certain browsers

            var xmlContent = request.responseXML;
            //ie6+ keeps the parsing response under xmlContent.parserError
            //while the rest of the world keeps it as element under the first node

            if (this._Lang.isXMLParseError(xmlContent)) {
                _Impl.sendError(request, context, myfaces._impl.core.Impl.MALFORMEDXML);
                return;
            }
            var partials = xmlContent.childNodes[0];
            if ('undefined' == typeof partials || partials == null) {
                _Impl.sendError(request, context, _Impl.MALFORMEDXML);
                return;
            } else {
                if (partials.tagName != this.RESP_PARTIAL) {
                    // IE 8 sees XML Header as first sibling ...
                    partials = partials.nextSibling;
                    if (!partials || partials.tagName != this.RESP_PARTIAL) {
                        _Impl.sendError(request, context, myfaces._impl.core.Impl.MALFORMEDXML);
                        return;
                    }
                }
            }

            var childNodesLength = partials.childNodes.length;

            for (var loop = 0; loop < childNodesLength; loop++) {
                var childNode = partials.childNodes[loop];
                var tagName = childNode.tagName;
                /**
                 * <eval>
                 *      <![CDATA[javascript]]>
                 * </eval>
                 */

                //this ought to be enough for eval
                //however the run scripts still makes sense
                //in the update and insert area for components
                //which do not use the response writer properly
                //we might add this one as custom option in update and
                //insert!
                if (tagName == this.CMD_ERROR) {
                    this.processError(request, context, childNode);
                    return;
                } else if (tagName == this.CMD_REDIRECT) {
                    if (!this.processRedirect(request, context, childNode)) return;
                } else if (tagName == this.CMD_CHANGES) {
                    if (!this.processChanges(request, context, childNode)) return;
                }
            }

            //fixup missing viewStates due to spec deficiencies
            this.fixViewStates(context);
        } catch (e) {
            this._onException(request, context, "myfaces._impl.xhrCore._AjaxResponse", "processResponse", e);
        }
    },

    fixViewStates : function(context) {

        if (null == this.appliedViewState) {
            return;
        }
        //if we set our no portlet env we safely can update all forms with
        //the new viewstate
        if (this._RT.getLocalOrGlobalConfig(context, "no_portlet_env", false)) {
            for (var cnt = document.forms.length - 1; cnt >= 0; cnt --) {
                this._setVSTForm(document.forms[cnt]);
            }
            return;
        }

        // Now update the forms that were not replaced but forced to be updated, because contains child ajax tags
        // we should only update forms with view state hidden field. If by some reason, the form was set to be
        // updated but the form was replaced, it does not have hidden view state, so later in changeTrace processing the
        // view state is updated.

        //set the viewstates of all outer forms parents of our updated elements
        this._Lang.arrForEach(this._updateForms, this._setVSTForm, 0, this);

        //set the viewstate of all forms within our updated elements
        this._Lang.arrForEach(this._updateElems, this._setVSTInnerForms, 0, this);
    },

    /**
     * sets the viewstate element in a given form
     *
     * @param theForm the form to which the element has to be set to
     * @param doNotChange if set to true no change is performed if the element is found already to be rendered
     */
    _setVSTForm: function(theForm, doNotChange) {
        var viewStateField = (theForm.elements) ? theForm.elements[this.P_VIEWSTATE] : null;//this._Dom.findFormElement(elem, this.P_VIEWSTATE);

        if (viewStateField && !doNotChange) {
            this._Dom.setAttribute(viewStateField, "value", this.appliedViewState);
        } else if (!viewStateField) {
            var element = this._Dom.getDummyPlaceHolder();
            element.innerHTML = ["<input type='hidden'", "id='", this.P_VIEWSTATE ,"' name='", this.P_VIEWSTATE ,"' value='" , this.appliedViewState , "' />"].join("");
            //now we go to proper dom handling after having to deal with another ie screwup
            try {
                theForm.appendChild(element.childNodes[0]);
            } finally {
                element.innerHTML = "";
            }
        }
    },

    _setVSTInnerForms: function(elem) {
        var replacedForms = this._Dom.findByTagName(elem, "form", false);
        var applyWithoutChange = this._Lang.hitch(this, function(elem) {
            this._setVSTForm(elem, true);
        });

        try {
            this._Lang.arrForEach(replacedForms, applyWithoutChange, 0, this);
        } finally {
            applyWithoutChange = false;
        }
    },



    processError : function(request, context, node) {
        /**
         * <error>
         *      <error-name>String</error-name>
         *      <error-message><![CDATA[message]]></error-message>
         * <error>
         */
        var errorName = node.firstChild.textContent || "";
        var errorMessage = node.childNodes[1].firstChild.data || "";

        var _Impl = this._Impl;

        _Impl.sendError(request, context, myfaces._impl.core.Impl.SERVER_ERROR, errorName, errorMessage);
    },

    processRedirect : function(request, context, node) {
        /**
         * <redirect url="url to redirect" />
         */
        var redirectUrl = node.getAttribute("url");
        if (!redirectUrl) {
            var _Impl = this._Impl;

            _Impl.sendError(request, context, myfaces._impl.core.Impl.MALFORMEDXML, myfaces._impl.core.Impl.MALFORMEDXML,this._Lang.getMessage("ERR_RED_URL", null, "_AjaxResponse.processRedirect"));
            return false;
        }
        redirectUrl = this._Lang.trim(redirectUrl);
        if (redirectUrl == "") {
            return false;
        }
        window.location = redirectUrl;
        return true;
    },

    /**
     * main entry point for processing the changes
     * it deals with the &lt;changes&gt; node of the
     * response
     *
     * @param request the xhr request object
     * @param context the context map
     * @param node the changes node to be processed
     */
    processChanges : function(request, context, node) {
        var changes = node.childNodes;

        //note we need to trace the changes which could affect our insert update or delete
        //se that we can realign our ViewStates afterwards
        //the realignment must happen post change processing
        
        for (var i = 0; i < changes.length; i++) {
         
            switch (changes[i].tagName) {

                case this.CMD_UPDATE:
                    if (!this.processUpdate(request, context, changes[i])) {
                        return false;
                    }
                    break;
                case this.CMD_EVAL:
                    this._Lang.globalEval(changes[i].firstChild.data);
                    break;
                case this.CMD_INSERT:
                    if (!this.processInsert(request, context, changes[i])) return false;
                    break;
                case this.CMD_DELETE:
                    if (!this.processDelete(request, context, changes[i])) return false;
                    break;
                case this.CMD_ATTRIBUTES:
                    if (!this.processAttributes(request, context, changes[i])) return false;
                    break;
                case this.CMD_EXTENSION:
                    break;
                default:
                    var _Impl = this._Impl;
                    _Impl.sendError(request, context, myfaces._impl.core.Impl.MALFORMEDXML);
                    return false;
            }
        }

        return true;
    },

    /**
     * First substep process a pending update tag
     *
     * @param request the xhr request object
     * @param context the context map
     * @param node the changes node to be processed
     */
    processUpdate : function(request, context, node) {
        if (node.getAttribute('id') == this.P_VIEWSTATE) {
            //update the submitting forms viewstate to the new value
            // The source form has to be pulled out of the CURRENT document first because the context object
            // may refer to an invalid document if an update of the entire body has occurred before this point.
            var viewStateValue = node.firstChild.nodeValue;

            //TODO save the issuing form id in the context element
             var mfInternal = context._mfInternal,
                    elementId = (mfInternal) ? mfInternal["_mfSourceControlId"] : context.source.id,
                    sourceForm = (mfInternal && mfInternal["_mfSourceFormId"] &&
                                  document.forms[mfInternal["_mfSourceFormId"]]) ?
                           document.forms[mfInternal["_mfSourceFormId"]] : ((elementId)? this._Dom.fuzzyFormDetection(elementId): null);

            this.appliedViewState = viewStateValue;
            //source form could not be determined either over the form identifer or the element
            //we now skip this phase and just add everything we need for the fixup code

            if (!sourceForm) {
                //no source form found is not an error because
                //we might be able to recover one way or the other
                return true;
            }

            this._setVSTForm(sourceForm);
        }
        else {
            // response may contain several blocks
            var cDataBlock = this._Dom.concatCDATABlocks(node);

            switch (node.getAttribute('id')) {
                case this.P_VIEWROOT:


                    cDataBlock = cDataBlock.substring(cDataBlock.indexOf("<html"));

                    var parsedData = this._replaceHead(request, context, cDataBlock);

                    var resultNode = ('undefined' != typeof parsedData &&  null != parsedData) ? this._replaceBody(request, context, cDataBlock, parsedData) : this._replaceBody(request, context, cDataBlock);
                    if (resultNode) {
                        this._pushOperationResult(resultNode);
                    }
                    break;
                case this.P_VIEWHEAD:
                    //we cannot replace the head, almost no browser allows this, some of them throw errors
                    //others simply ignore it or replace it and destroy the dom that way!
                    this._replaceHead(request, context, cDataBlock);

                    break;
                case this.P_VIEWBODY:
                    //we assume the cdata block is our body including the tag
                    var resultNode = this._replaceBody(request, context, cDataBlock);
                    if (resultNode) {
                        this._pushOperationResult(resultNode);
                    }
                    break;

                default:
                    var resultNode = this.replaceHtmlItem(request, context, node.getAttribute('id'), cDataBlock);
                    if (resultNode) {
                        this._pushOperationResult(resultNode);
                    }
                    break;
            }
        }
        return true;
    },

    _pushOperationResult: function(resultNode) {
        var pushSubnode = this._Lang.hitch(this, function(currNode) {
            var parentForm = this._Dom.getParent(currNode, "form");
            if (null != parentForm) {
                this._updateForms.push(parentForm);
            }
            else {
                this._updateElems.push(currNode);
            }
        });
        var isArr = 'undefined' != typeof resultNode.length && 'undefined' == typeof resultNode.nodeType;
        if (isArr && resultNode.length) {
            for (var cnt = 0; cnt < resultNode.length; cnt++) {
                pushSubnode(resultNode[cnt]);
            }
        } else if (!isArr) {
            pushSubnode(resultNode);
        }

    },

    /**
     * replaces a current head theoretically,
     * pratically only the scripts are evaled anew since nothing else
     * can be changed.
     *
     * @param request the current request
     * @param context the ajax context
     * @param newData the data to be processed
     *
     * @return an xml representation of the page for further processing if possible
     */
    _replaceHead: function(request, context, newData) {

        var _Impl = this._Impl;

        var doc = this._Lang.parseXML(newData);

        var newHead = null;
        if (this._Lang.isXMLParseError(doc)) {
            doc = this._Lang.parseXML(newData.replace(/<!\-\-[\s\n]*<!\-\-/g, "<!--").replace(/\/\/-->[\s\n]*\/\/-->/g, "//-->"));
        }

        if (this._Lang.isXMLParseError(doc)) {
            //the standard xml parser failed we retry with the stripper
            var parser = new (this._RT.getGlobalConfig("updateParser", myfaces._impl._util._HtmlStripper))();
            var headData = parser.parse(newData, "head");

            newHead = this._Lang.parseXML("<head>" + headData + "</head>");
            //last and slowest option create a new head element and let the browser
            //do its slow job
            if (this._Lang.isXMLParseError(newHead)) {
                try {
                    newHead = document.createElement("head");
                    newHead.innerHTML = headData;
                } catch (e) {
                    //we give up no further fallbacks
                    _Impl.sendError(request, context, _Impl.MALFORMEDXML, _Impl.MALFORMEDXML, "Error head replacement failed reason:"+e.toString());
                    return null;
                }
            }
        } else {
            //parser worked we go on
            newHead = doc.getElementsByTagName("head")[0];
        }

        this._Dom.runScripts(newHead, true);

        return doc;
    },


    /**
     * special method to handle the body dom manipulation,
     * replacing the entire body does not work fully by simply adding a second body
     * and by creating a range instead we have to work around that by dom creating a second
     * body and then filling it properly!
     *
     * @param {Object} request our request object
     * @param {Object} context (Map) the response context
     * @param {String} newData the markup which replaces the old dom node!
     * @param {Node} parsedData (optional) preparsed XML representation data of the current document
     */
    _replaceBody : function(request, context, newData /*varargs*/) {

        var parser = new (this._RT.getGlobalConfig("updateParser", myfaces._impl._util._HtmlStripper))();
        var oldBody = document.getElementsByTagName("body")[0];
        var placeHolder = document.createElement("div");

        placeHolder.id = "myfaces_bodyplaceholder";

        var bodyParent = oldBody.parentNode;
        this._Dom._removeChildNodes(oldBody);
        oldBody.innerHTML = "";
        var newBody = oldBody;

        newBody.appendChild(placeHolder);

        var bodyData = null;

        var doc = (arguments.length > 3) ? arguments[3] : this._Lang.parseXML(newData);

        if (this._Lang.isXMLParseError(doc)) {
            doc = this._Lang.parseXML(newData.replace(/<!\-\-[\s\n]*<!\-\-/g, "<!--").replace(/\/\/-->[\s\n]*\/\/-->/g, "//-->"));
        }

        if (this._Lang.isXMLParseError(doc)) {
            //the standard xml parser failed we retry with the stripper

            var parser = new (this._RT.getGlobalConfig("updateParser", myfaces._impl._util._HtmlStripper))();


            bodyData = parser.parse(newData, "body");
        } else {
            //parser worked we go on
            var newBodyData = doc.getElementsByTagName("body")[0];
            bodyData = this._Lang.serializeChilds(newBodyData);

            if (!this._RT.browser.isIEMobile || this._RT.browser.isIEMobile >= 7) {
                //TODO check what is failing there
                for (var cnt = 0; cnt < newBodyData.attributes.length; cnt++) {
                    var value = newBodyData.attributes[cnt].value;
                    if (value)
                        this._Dom.setAttribute(newBody, newBodyData.attributes[cnt].name, value);
                }
            }
        }

       
        var returnedElement = this.replaceHtmlItem(request, context, placeHolder, bodyData);

        if (returnedElement) {
            this._pushOperationResult(returnedElement);
        }
        return returnedElement;
    },

    /**
     * Replaces HTML elements through others and handle errors if the occur in the replacement part
     *
     * @param {Object} request (xhrRequest)
     * @param {Object} context (Map)
     * @param {Object} itemIdToReplace (String|Node) - ID of the element to replace
     * @param {String} markup - the new tag
     */
    replaceHtmlItem : function(request, context, itemIdToReplace, markup) {
        try {
            // (itemIdToReplace instanceof Node) is NOT compatible with IE8
            var item = (!this._Lang.isString(itemIdToReplace)) ? itemIdToReplace :
                    this._Dom.byId(itemIdToReplace) /*used to call getElementFromForm*/;
            if (!item) {
                throw Error(this._Lang.getMessage("ERR_ITEM_ID_NOTFOUND", null,"_AjaxResponse.replaceHtmlItem",(itemIdToReplace)? itemIdToReplace.toString():"undefined"));
            }
            return this._Dom.outerHTML(item, markup);

        } catch (e) {
            this._onException(request, context, "myfaces._impl.xhrCore._AjaxResponse", "replaceHTMLItem", e);
        }
        return null;
    }
    ,

    /*insert, three attributes can be present
     * id = insert id
     * before = before id
     * after = after  id
     *
     * the insert id is the id of the node to be inserted
     * the before is the id if set which the component has to be inserted before
     * the after is the id if set which the component has to be inserted after
     **/
    processInsert : function(request, context, node) {
        /*remapping global namespaces for speed and readability reasons*/
        var _Impl = this._Impl;
        var _Dom = this._Dom;
        var _Lang = this._Lang;

        var insertId = node.getAttribute('id');
        var beforeId = node.getAttribute('before');
        var afterId = node.getAttribute('after');

        var isInsert = insertId && this._Lang.trim(insertId) != "";
        var isBefore = beforeId && this._Lang.trim(beforeId) != "";
        var isAfter = afterId && this._Lang.trim(afterId) != "";

        if (!isInsert) {
            _Impl.sendError(request, context, _Impl.MALFORMEDXML, _Impl.MALFORMEDXML,this._Lang.getMessage("ERR_PPR_IDREQ"));
            return false;
        }
        if (!(isBefore || isAfter)) {
            _Impl.sendError(request, context, _Impl.MALFORMEDXML, _Impl.MALFORMEDXML,this._Lang.getMessage("ERR_PPR_INSERTBEFID"));
            return false;
        }
        //either before or after but not two at the same time
        var nodeHolder = null;
        var parentNode = null;


        var cDataBlock = this._Dom.concatCDATABlocks(node);

        var replacementFragment;
        if (isBefore) {
            beforeId = this._Lang.trim(beforeId);
            var beforeNode = document.getElementById(beforeId);
            if (!beforeNode) {
                _Impl.sendError(request, context, _Impl.MALFORMEDXML, _Impl.MALFORMEDXML,this._Lang.getMessage("ERR_PPR_INSERTBEFID_1", null,"_AjaxResponse.processInsert",beforeId));
                return false;
            }
            /**
             *we generate a temp holder
             *so that we can use innerHTML for
             *generating the content upfront
             *before inserting it"
             **/
            nodeHolder = document.createElement("div");
            parentNode = beforeNode.parentNode;
            parentNode.insertBefore(nodeHolder, beforeNode);
            replacementFragment = this.replaceHtmlItem(request, context,
                    nodeHolder, cDataBlock);
            if (replacementFragment) {
                this._pushOperationResult(replacementFragment);
            }

        } else {
            afterId = this._Lang.trim(afterId);
            var afterNode = document.getElementById(afterId);
            if (!afterNode) {
                _Impl.sendError(request, context, _Impl.MALFORMEDXML, _Impl.MALFORMEDXML, this._Lang.getMessage("ERR_PPR_INSERTBEFID_2", null,"_AjaxResponse.processInsert", afterId));
                return false;
            }

            nodeHolder = document.createElement("div");
            parentNode = afterNode.parentNode;

            //TODO nextsibling not working in ieMobile 6.1 we have to change the method
            //of accessing it to something else
            parentNode.insertBefore(nodeHolder, afterNode.nextSibling);

            replacementFragment = this.replaceHtmlItem(request, context,
                    nodeHolder, cDataBlock);

            if (replacementFragment) {
                this._pushOperationResult(replacementFragment);
            }

        }
        return true;
    }
    ,

    processDelete : function(request, context, node) {
        var _Impl = this._Impl;

        var deleteId = node.getAttribute('id');
        if (!deleteId) {
            _Impl.sendError(request, context, _Impl.MALFORMEDXML,
                    _Impl.MALFORMEDXML,this._Lang.getMessage("ERR_PPR_DELID", null,"_AjaxResponse.processDelete"));
            return false;
        }

        var item = this._Dom.byId(deleteId);
        if (!item) {
            throw Error(this._Lang.getMessage("ERR_PPR_UNKNOWNCID", null,"_AjaxResponse.processDelete",deleteId));
        }

        var parentForm = this._Dom.getParent(item, "form");
        if (null != parentForm) {
            this._updateForms.push(parentForm);
        }
        this._Dom.deleteItem(item);

        return true;
    }
    ,

    processAttributes : function(request, context, node) {
        //we now route into our attributes function to bypass
        //IE quirks mode incompatibilities to the biggest possible extent
        //most browsers just have to do a setAttributes but IE
        //behaves as usual not like the official standard
        //myfaces._impl._util.this._Dom.setAttribute(domNode, attribute, value;

        var _Impl = this._Impl;

        //<attributes id="id of element"> <attribute name="attribute name" value="attribute value" />* </attributes>
        var elemId = node.getAttribute('id');
        if (!elemId) {
            _Impl.sendError(request, context, _Impl.MALFORMEDXML
                    , _Impl.MALFORMEDXML, "Error in attributes, id not in xml markup");
            return false;
        }
        var childNodes = node.childNodes;

        if (!childNodes) {
            return false;
        }
        for (var loop2 = 0; loop2 < childNodes.length; loop2++) {
            var attributesNode = childNodes[loop2];

            var attrName = attributesNode.getAttribute("name");
            var attrValue = attributesNode.getAttribute("value");

            if (!attrName) {
                continue;
            }

            attrName = this._Lang.trim(attrName);
            /*no value means reset*/
            //value can be of boolean value hence full check
            if ('undefined' == typeof attrValue || null == attrValue) {
                attrValue = "";
            }

            switch (elemId) {
                case this.P_VIEWROOT:
                    throw new Error(this._Lang.getMessage("ERR_NO_VIEWROOTATTR", null,"_AjaxResponse.processAttributes"));
                    break;

                case this.P_VIEWHEAD:
                    throw new Error(this._Lang.getMessage("ERR_NO_HEADATTR", null,"_AjaxResponse.processAttributes"));
                    break;

                case this.P_VIEWBODY:
                    var element = document.getElementsByTagName("body")[0];
                    this._Dom.setAttribute(element, attrName, attrValue);
                    break;

                default:
                    this._Dom.setAttribute(document.getElementById(elemId), attrName, attrValue);
                    break;
            }

        }
        return true;
    },
    _finalize: function() {
        delete this._onException;
        delete this._onWarning;
        delete this._updateElems;
        // List of forms to be updated if any inner block is updated
        delete this._updateForms;
        delete this.appliedViewState;
    }

});

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * The xhr core adapter
 * which provides the transport mechanisms to the calling
 * objects, and controls the queue behavior, the error handling
 * and partial page submit functionality among other things
 *
 * The idea behind this is to make the ajax request object as barebones
 * as possible and shift the extra functionality like queuing
 * parameter handling etc... to this class so that our transports become more easily
 * pluggable. This should keep the path open to iframe submits and other transport layers
 *
 * the call to the corresponding transport just should be a
 * transport.xhrQueuedPost
 *
 * or transport.xhrPost,transport.xhrGet  etc... in the future
 *
 * Note we have taken a pattern lesson or two from the dojo toolkit and its excellent handling
 * of transports by our patterns here (which is mainly a centralized transport singleton which routes
 * to different transport implementations and the auto passing of parameters into their
 * corresponding protected attributes on class level in the transports themselves)
 *
 */
/** @namespace myfaces._impl.xhrCore._Transports */
myfaces._impl.core._Runtime.extendClass("myfaces._impl.xhrCore._Transports"
        , Object, {

    _PAR_ERRORLEVEL:"errorlevel",
    _PAR_QUEUESIZE:"queuesize",
    _PAR_PPS:"pps",
    _PAR_TIMEOUT:"timeout",
    _PAR_DELAY:"delay",


    /**
     * a singleton queue
     * note the structure of our inheritance
     * is that that _queue is attached to prototype
     * and hence the pointer to the request qeue
     * is shared over all instances
     *
     * if you need to have it per instance for complex objects
     * you have to initialize in the constructor
     *
     * (This is the same limitation dojo class inheritance
     * where our inheritance pattern is derived from has)
     */
    _q: new myfaces._impl.xhrCore._AjaxRequestQueue(),

    _threshold: "ERROR",

    _Lang :  myfaces._impl._util._Lang,
    _RT: myfaces._impl.core._Runtime,


    /**
     * a simple not enqueued xhr post
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    xhrPost : function(source, sourceForm, context, passThrgh) {
        (new (this._getAjaxReqClass(context))(this._getArguments(source, sourceForm, context, passThrgh))).send();
    },

    /**
     * xhr post with enqueuing as defined by the jsf 2.0 specification
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    xhrQueuedPost : function(source, sourceForm, context, passThrgh) {
        this._q.enqueue(
                new (this._getAjaxReqClass(context))(this._getArguments(source, sourceForm, context, passThrgh)));
    },

    /**
     * xhr get without enqueuing
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    xhrGet : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        // we have to live with it
        args.ajaxType = "GET";
        (new (this._getAjaxReqClass(context))(args)).send();
    },

    /**
     * xhr get which takes the existing queue into consideration to by synchronized
     * to previous queued post requests
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    xhrQueuedGet : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        // we have to live with it
        args.ajaxType = "GET";
        this._q.enqueue(
                new (this._getAjaxReqClass(context))(args));
    },


    /**
     * iframe post without queueing
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    multipartPost : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        // we have to live with it

        (new (this._getMultipartReqClass(context))(args)).send();
    },

    /**
     * iframe queued post
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    multipartQueuedPost : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        this._q.enqueue(
                new (this._getMultipartReqClass(context))(args));
    },


    /**
     * iframe get without queueing
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    multipartGet : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        // we have to live with it
        args.ajaxType = "GET";
        (new (this._getMultipartReqClass(context))(args)).send();
    },

    /**
     * iframe queued http get
     *
     * mapped options already have the exec and view properly in place
     * myfaces specifics can be found under mappedOptions.myFaces
     * @param {Node} source the source of this call
     * @param {Node} sourceForm the html form which is the source of this call
     * @param {Object} context (Map) the internal pass through context
     * @param {Object} passThrgh (Map) values to be passed through
     **/
    multipartQueuedGet : function(source, sourceForm, context, passThrgh) {
        var args = this._getArguments(source, sourceForm, context, passThrgh);
        // note in get the timeout is not working delay however is and queue size as well
        // since there are no cross browser ways to resolve a timeout on xhr level
        args.ajaxType = "GET";
        this._q.enqueue(
                new (this._getMultipartReqClass(context))(args));
    },


    /**
     * Spec. 13.3.3
     * Examining the response markup and updating the DOM tree
     * @param {XmlHttpRequest} request - the ajax request
     * @param {XmlHttpRequest} context - the ajax context
     */
    response : function(request, context) {
        var ajaxObj =  new (this._getAjaxReqClass(context))({xhr: request, context: context});

        ajaxObj._response.processResponse(request, context);
    },

    /**
     * creates the arguments map and
     * fetches the config params in a proper way in to
     * deal with them in a flat way (from the nested context way)
     *
     * @param source the source of the request
     * @param sourceForm the sourceform
     * @param context   the context holding all values
     * @param passThrgh the passThrough values to be blended into the response
     */
    _getArguments: function(source, sourceForm, context, passThrgh) {
        var _RT = myfaces._impl.core._Runtime;
        var _getConfig = _RT.getLocalOrGlobalConfig;
        var _Lang = myfaces._impl._util._Lang;

        var ret = {
            "source": source,
            "sourceForm": sourceForm,
            "context": context,
            "passThrough": passThrgh,
            "xhrQueue": this._q,

            //standard done callback
            "onDone": this._Lang.hitch(this, this._stdOnDone),
            //standard success callback
            "onSuccess": this._Lang.hitch(this, this._stdOnSuccess),
            //standard server side error callback
            "onError": this._Lang.hitch(this, this._stdOnError),

            //standard timeout callback
            "onTimeout": this._Lang.hitch(this, this._stdOnTimeout),

            //now to the internal error handlers which perform operations
            //on the queue
            //standard exception handling callback
            "onException": this._Lang.hitch(this, this._stdErrorHandler),
            //standard warning handling callback
            "onWarn": this._Lang.hitch(this, this._stdErrorHandler)
        };

        //we now mix in the config settings which might either be set globally
        //or pushed in under the context myfaces.<contextValue> into the current request 
        this._applyConfig(ret, context, "alarmThreshold", this._PAR_ERRORLEVEL);
        this._applyConfig(ret, context, "queueSize", this._PAR_QUEUESIZE);
        this._applyConfig(ret, context, "timeout", this._PAR_TIMEOUT);
        this._applyConfig(ret, context, "delay", this._PAR_DELAY);

        //now partial page submit needs a different treatment
        //since pps == execute strings
        if (_getConfig(context, this._PAR_PPS, false)
                && _Lang.exists(passThrgh, myfaces._impl.core.Impl.P_EXECUTE)
                && passThrgh[myfaces._impl.core.Impl.P_EXECUTE].length > 0) {
            ret['partialIdsArray'] = passThrgh[myfaces._impl.core.Impl.P_EXECUTE].split(" ");
        }
        return ret;
    },

    /**
     * helper method to apply a config setting to our varargs param list
     *
     * @param destination the destination map to receive the setting
     * @param context the current context
     * @param destParm the destination param of the destination map
     * @param srcParm the source param which is the key to our config setting
     */
    _applyConfig: function(destination, context, destParm, srcParm) {
        var _RT = myfaces._impl.core._Runtime;
        var _getConfig = _RT.getLocalOrGlobalConfig;
        if (_getConfig(context, srcParm, null) != null) {
            destination[destParm] = _getConfig(context, srcParm, null);
        }
    },

    /**
     * standard on done handler which routes to the
     * event sending specified by the spec
     *
     * @param request the xhr request object
     * @param context the context holding all values for further processing
     */
    _stdOnDone: function(request, context) {
        this._loadImpl();

        this._Impl.sendEvent(request, context, this._Impl.COMPLETE);
    },

    /**
     * standard spec compliant success handler
     *
     * @param request the xhr request object
     * @param context the context holding all values for further processing
     */
    _stdOnSuccess: function(request, context) {
        //_onSuccess
        this._loadImpl();
        try {
            this._Impl.response(request, context);

            this._Impl.sendEvent(request, context, this._Impl.SUCCESS);
        } finally {
            this._q.processQueue();
            //ie6 helper cleanup
            delete context.source;
        }
    },

    /**
     * now to the error case handlers which by spec should
     * route to our standard error queue
     *
     * @param request the xhr request object
     * @param context the context holding all values for further processing
     */
    _stdOnError: function(request, context) {
        this._loadImpl();

        //_onError
        var errorText;
        try {
            errorText = "Request failed";
            if (request.status) {
                errorText += "with status " + request.status;
                if (request.statusText) {
                    errorText += " and reason " + this._xhr.statusText;
                }
            }

        } catch (e) {
            errorText = "Request failed with unknown status";
        } finally {
            try {
                this._Impl.sendError(request, context, this._Impl.HTTPERROR,
                        this._Impl.HTTPERROR, errorText);
            } finally {
                this._q.processQueue();
                //ie6 helper cleanup
                delete context.source;
                
            }
        }
        //_onError

    },

    /**
     * standard timeout handler
     * the details on how to handle the timeout are
     * handled by the calling request object
     */
    _stdOnTimeout: function(request, context) {
        this._loadImpl();
        try {
            //we issue an event not an error here before killing the xhr process
            this._Impl.sendEvent(request, context, this._Impl.TIMEOUT_EVENT,
                    this._Impl.TIMEOUT_EVENT);
            //timeout done we process the next in the queue
        } finally {
            //We trigger the next one in the queue
            this._q.processQueue();
        }
        //ready state done should be called automatically
    },

    /**
     * Client error handlers which also in the long run route into our error queue
     * but also are able to deliver more meaningful messages
     * note, in case of an error all subsequent xhr requests are dropped
     * to get a clean state on things
     *
     * @param request the xhr request object
     * @param context the context holding all values for further processing
     * @param sourceClass (String) the issuing class for a more meaningful message
     * @param func the issuing function
     * @param exception the embedded exception
     */
    _stdErrorHandler: function(request, context, sourceClass, func, exception) {
        this._loadImpl();
        var _Lang =  myfaces._impl._util._Lang;
        var exProcessed = _Lang.isExceptionProcessed(exception);
        try {
            //newer browsers do not allow to hold additional values on native objects like exceptions
            //we hence capsule it into the request, which is gced automatically
            //on ie as well, since the stdErrorHandler usually is called between requests
            //this is a valid approach

            if (this._threshold == "ERROR" && !exProcessed) {
                this._Impl.sendError(request, context, this._Impl.CLIENT_ERROR, exception.name,
                        "MyFaces ERROR:" + this._Lang.createErrorMsg(sourceClass, func, exception));
            }
        } finally {
            this._q.cleanup();
            //we forward the exception, just in case so that the client
            //will receive it in any way
            try {
                if(!exProcessed) {
                    _Lang.setExceptionProcessed(exception);
                }
            } catch(e) {

            }
            throw exception;
        }
    },


    _loadImpl: function() {
        if (!this._Impl) {
            this._Impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
        }
        return this._Impl;
    },

    /**
     * centralized transport switching helper
     * for the multipart submit case
     *
     * @param context the context which is passed down
     */
    _getMultipartReqClass: function(context) {
        //if (this._RT.getLocalOrGlobalConfig(context, "transportAutoSelection", false) && this._RT.getXHRLvl() >= 2) {
        //    return myfaces._impl.xhrCore._AjaxRequestLevel2;
        //} else {
            return myfaces._impl.xhrCore._IFrameRequest;
        //}
    },


    _getAjaxReqClass: function(context) {
        return myfaces._impl.xhrCore._AjaxRequest;
    }

});

/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** @namespace myfaces._impl.core.Impl*/
/** @namespace myfaces._impl._util._ListenerQueue */
myfaces._impl.core._Runtime.singletonExtendClass("myfaces._impl.core.Impl", Object, {

    //third option myfaces._impl.xhrCoreAjax which will be the new core impl for now
    _transport      : new (myfaces._impl.core._Runtime.getGlobalConfig("transport", myfaces._impl.xhrCore._Transports))(),

    /**
     * external event listener queue!
     */
    _evtListeners   : new (myfaces._impl.core._Runtime.getGlobalConfig("eventListenerQueue", myfaces._impl._util._ListenerQueue))(),

    /**
     * external error listener queue!
     */
    _errListeners   : new (myfaces._impl.core._Runtime.getGlobalConfig("errorListenerQueue", myfaces._impl._util._ListenerQueue))(),

    /*CONSTANTS*/

    /*internal identifiers for options*/
    IDENT_ALL:  "@all",
    IDENT_NONE: "@none",
    IDENT_THIS: "@this",
    IDENT_FORM: "@form",

    /*
     * [STATIC] constants
     */

    P_PARTIAL_SOURCE:   "javax.faces.source",
    P_VIEWSTATE:        "javax.faces.ViewState",
    P_AJAX:             "javax.faces.partial.ajax",
    P_EXECUTE:          "javax.faces.partial.execute",
    P_RENDER:           "javax.faces.partial.render",
    P_EVT:              "javax.faces.partial.event",

    /* message types */
    ERROR: "error",
    EVENT: "event",

    /* event emitting stages */
    BEGIN:      "begin",
    COMPLETE:   "complete",
    SUCCESS:    "success",

    /*ajax errors spec 14.4.2*/
    HTTPERROR:      "httpError",
    EMPTY_RESPONSE: "emptyResponse",
    MALFORMEDXML:   "malformedXML",
    SERVER_ERROR:   "serverError",
    CLIENT_ERROR:   "clientError",
    TIMEOUT_EVENT:  "timeout",

    _Lang:  myfaces._impl._util._Lang,
    _Dom:   myfaces._impl._util._Dom,

    /*blockfilter for the passthrough filtering, the attributes given here
     * will not be transmitted from the options into the passthrough*/
    _BLOCKFILTER: {onerror: true, onevent: true, render: true, execute: true, myfaces: true},

    /**
     * collect and encode data for a given form element (must be of type form)
     * find the javax.faces.ViewState element and encode its value as well!
     * return a concatenated string of the encoded values!
     *
     * @throws error in case of the given element not being of type form!
     * https://issues.apache.org/jira/browse/MYFACES-2110
     */
    getViewState : function(form) {
        /**
         *  typecheck assert!, we opt for strong typing here
         *  because it makes it easier to detect bugs
         */
        if (form) {
            form = this._Lang.byId(form);
        }

        if (!form
                || !form.nodeName
                || form.nodeName.toLowerCase() != "form") {
            throw new Error(this._Lang.getMessage("ERR_VIEWSTATE"));
        }

        var ajaxUtils = new myfaces._impl.xhrCore._AjaxUtils(0);

        var ret = this._Lang.createFormDataDecorator([]);
		ajaxUtils.encodeSubmittableFields(ret, null, null, null, form, null);
        return ret.makeFinal();
    },

    /**
     * this function has to send the ajax requests
     *
     * following request conditions must be met:
     * <ul>
     *  <li> the request must be sent asynchronously! </li>
     *  <li> the request must be a POST!!! request </li>
     *  <li> the request url must be the form action attribute </li>
     *  <li> all requests must be queued with a client side request queue to ensure the request ordering!</li>
     * </ul>
     *
     * @param {String|Node} elem any dom element no matter being it html or jsf, from which the event is emitted
     * @param {|Event|} event any javascript event supported by that object
     * @param {|Object|} options  map of options being pushed into the ajax cycle
     *
     *
     * TODO refactoring, the passthrgh handling is only for dragging in request parameters
     * we should rewrite this part
     *
     *
     * a) transformArguments out of the function
     * b) passThrough handling with a map copy with a filter map block map
     */
    request : function(elem, event, options) {
     
        /*namespace remap for our local function context we mix the entire function namespace into
         *a local function variable so that we do not have to write the entire namespace
         *all the time
         **/
        var _Lang = this._Lang;
        var _Dom =  this._Dom;
        var getConfig = myfaces._impl.core._Runtime.getLocalOrGlobalConfig;

        /*assert if the onerror is set and once if it is set it must be of type function*/
        _Lang.assertType(options.onerror, "function");
        /*assert if the onevent is set and once if it is set it must be of type function*/
        _Lang.assertType(options.onevent, "function");

        //options not set we define a default one with nothing
        options = options || {};

        /**
         * we cross reference statically hence the mapping here
         * the entire mapping between the functions is stateless
         */
        //null definitely means no event passed down so we skip the ie specific checks
        if ('undefined' == typeof event) {
            event = window.event || null;
        }

        elem = _Lang.byId(elem);
        var elementId = _Dom.nodeIdOrName(elem);

        /*
         * We make a copy of our options because
         * we should not touch the incoming params!
         */
        var passThrgh = _Lang.mixMaps({}, options, true, this._BLOCKFILTER);

        if (event) {
            passThrgh[this.P_EVT] = event.type;
        }

        /**
         * ajax pass through context with the source
         * onevent and onerror
         */
        var context = {
            source: elem,
            onevent: options.onevent,
            onerror: options.onerror,

            //TODO move the myfaces part into the _mfInternal part
            myfaces: options.myfaces
        };

        /**
         * fetch the parent form
         *
         * note we also add an override possibility here
         * so that people can use dummy forms and work
         * with detached objects
         */
        var form = (options.myfaces && options.myfaces.form)?
                _Lang.byId(options.myfaces.form):
                this._getForm(elem, event);

        /**
         * binding contract the javax.faces.source must be set
         */
        passThrgh[this.P_PARTIAL_SOURCE] = elementId;

        /**
         * javax.faces.partial.ajax must be set to true
         */
        passThrgh[this.P_AJAX] = true;

        if (options.execute) {
            /*the options must be a blank delimited list of strings*/
            /*compliance with Mojarra which automatically adds @this to an execute
             * the spec rev 2.0a however states, if none is issued nothing at all should be sent down
             */
            this._transformList(passThrgh, this.P_EXECUTE, options.execute + " @this", form, elementId);
        } else {
            passThrgh[this.P_EXECUTE] = elementId;
        }

        if (options.render) {
            this._transformList(passThrgh, this.P_RENDER, options.render, form, elementId);
        }

        /**
         * multiple transports upcoming jsf 2.1 feature currently allowed
         * default (no value) xhrQueuedPost
         *
         * xhrQueuedPost
         * xhrPost
         * xhrGet
         * xhrQueuedGet
         * iframePost
         * iframeQueuedPost
         *
         */
        var transportType = this._getTransportType(context, passThrgh, form);

        //additional meta information to speed things up, note internal non jsf
        //pass through options are stored under _mfInternal in the context
        context._mfInternal = {};
        var mfInternal = context._mfInternal;

        mfInternal["_mfSourceFormId"] =     form.id;
        mfInternal["_mfSourceControlId"] =  elementId;
        mfInternal["_mfTransportType"] =    transportType;

        //mojarra compatibility, mojarra is sending the form id as well
        //this is not documented behavior but can be determined by running
        //mojarra under blackbox conditions
        //i assume it does the same as our formId_submit=1 so leaving it out
        //wont hurt but for the sake of compatibility we are going to add it
        passThrgh[form.id] = form.id;

        this._transport[transportType](elem, form, context, passThrgh);

    },

    /**
     * fetches the form in an unprecise manner depending
     * on an element or event target
     *
     * @param elem
     * @param event
     */
    _getForm: function(elem, event) {
        var _Dom =  this._Dom;
        var _Lang = this._Lang;
        var form =  _Dom.fuzzyFormDetection(elem);

        if (!form && event) {
            //in case of no form is given we retry over the issuing event
            form = _Dom.fuzzyFormDetection(_Lang.getEventTarget(event));
            if (!form) {
                throw Error(_Lang.getMessage("ERR_FORM"));
            }
        } else if (!form) {
            throw Error(_Lang.getMessage("ERR_FORM"));
        }
        return form;
    },

    /**
     * determines the transport type to be called
     * for the ajax call
     *
     * @param context
     * @param passThrgh
     * @param form
     */
    _getTransportType: function(context, passThrgh, form) {
        /**
         * if execute or render exist
         * we have to pass them down as a blank delimited string representation
         * of an array of ids!
         */
        //for now we turn off the transport auto selection, to enable 2.0 backwards compatibility
        //on protocol level, the file upload only can be turned on if the auto selection is set to true
        var getConfig = myfaces._impl.core._Runtime.getLocalOrGlobalConfig;
        var _Lang     = this._Lang;
        var _Dom      = this._Dom;

        var transportAutoSelection = getConfig(context, "transportAutoSelection", false);
        var isMultipart = (transportAutoSelection && _Dom.getAttribute(form, "enctype") == "multipart/form-data") ?
                _Dom.isMultipartCandidate(passThrgh[this.P_EXECUTE]) :
                false;

        /**
         * multiple transports upcoming jsf 2.1 feature currently allowed
         * default (no value) xhrQueuedPost
         *
         * xhrQueuedPost
         * xhrPost
         * xhrGet
         * xhrQueuedGet
         * iframePost
         * iframeQueuedPost
         *
         */
        var transportType = (!isMultipart) ?
                getConfig(context, "transportType", "xhrQueuedPost") :
                getConfig(context, "transportType", "multipartQueuedPost");
        if (!this._transport[transportType]) {
            //throw new Error("Transport type " + transportType + " does not exist");
            throw new Error(_Lang.getMessage("ERR_TRANSPORT",null, transportType));
        }
        return transportType;

    },

    /**
     * transforms the list to the expected one
     * with the proper none all form and this handling
     * (note we also could use a simple string replace but then
     * we would have had double entries under some circumstances)
     *
     * @param passThrgh
     * @param target
     * @param srcStr
     * @param form
     * @param elementId
     */
    _transformList: function(passThrgh, target, srcStr, form, elementId) {
        var _Lang = this._Lang;
        //this is probably the fastest transformation method
        //it uses an array and an index to position all elements correctly
        //the offset variable is there to prevent 0 which results in a javascript
        //false
        var offset = 1,
                vals  = (srcStr) ? srcStr.split(/\s+/) : [],
                idIdx = (vals.length) ? _Lang.arrToMap(vals, offset) : {},

                //helpers to improve speed and compression
                none    = idIdx[this.IDENT_NONE],
                all     = idIdx[this.IDENT_ALL],
                theThis = idIdx[this.IDENT_THIS],
                theForm = idIdx[this.IDENT_FORM];

        if (none) {
            //in case of none only one value is returned
            passThrgh[target] = this.IDENT_NONE;
            return passThrgh;
        }
        if (all) {
            //in case of all only one value is returned
            passThrgh[target] = this.IDENT_ALL;
            return passThrgh;
        }

        if (theForm) {
            //the form is replaced with the proper id but the other
            //values are not touched
            vals[theForm - offset] = form.id;
        }
        if (theThis && !idIdx[elementId]) {
            //in case of this, the element id is set
            vals[theThis - offset] = elementId;
        }

        //the final list must be blank separated
        passThrgh[target] = vals.join(" ");
        return passThrgh;
    },

    addOnError : function(/*function*/errorListener) {
        /*error handling already done in the assert of the queue*/
        this._errListeners.enqueue(errorListener);
    },

    addOnEvent : function(/*function*/eventListener) {
        /*error handling already done in the assert of the queue*/
        this._evtListeners.enqueue(eventListener);
    },



    /**
     * implementation triggering the error chain
     *
     * @param {Object} request the request object which comes from the xhr cycle
     * @param {Object} context (Map) the context object being pushed over the xhr cycle keeping additional metadata
     * @param {String} name the error name
     * @param {String} serverErrorName the server error name in case of a server error
     * @param {String} serverErrorMessage the server error message in case of a server error
     *
     *  handles the errors, in case of an onError exists within the context the onError is called as local error handler
     *  the registered error handlers in the queue receiv an error message to be dealt with
     *  and if the projectStage is at development an alert box is displayed
     *
     *  note: we have additional functionality here, via the global config myfaces.config.defaultErrorOutput a function can be provided
     *  which changes the default output behavior from alert to something else
     *
     *
     */
    sendError : function sendError(/*Object*/request, /*Object*/ context, /*String*/ name, /*String*/ serverErrorName, /*String*/ serverErrorMessage) {
        var _Lang = myfaces._impl._util._Lang;

        var eventData = {};
        //we keep this in a closure because we might reuse it for our serverErrorMessage
        var malFormedMessage = function() {
            return (name && name === myfaces._impl.core.Impl.MALFORMEDXML) ? _Lang.getMessage("ERR_MALFORMEDXML") : "";
        };

        eventData.type = this.ERROR;

        eventData.status            = name;
        eventData.serverErrorName   = serverErrorName;
        eventData.serverErrorMessage =  serverErrorMessage;

        try {
            eventData.source        = context.source;
            eventData.responseCode  = request.status;
            eventData.responseText  = request.responseText;
            eventData.responseXML   = request.responseXML;
        } catch (e) {
            // silently ignore: user can find out by examining the event data
        }

        /**/
        if (context["onerror"]) {
            context.onerror(eventData);
        }

        /*now we serve the queue as well*/
        this._errListeners.broadcastEvent(eventData);

        if (jsf.getProjectStage() === "Development" && this._errListeners.length() == 0) {
            var defaultErrorOutput = myfaces._impl.core._Runtime.getGlobalConfig("defaultErrorOutput", alert);
            var finalMessage = [];

            finalMessage.push((name) ? name : "");
            finalMessage.push((serverErrorName) ? serverErrorName : "");
            finalMessage.push((serverErrorMessage) ? serverErrorMessage : "");
            finalMessage.push(malFormedMessage());

            defaultErrorOutput(finalMessage.join("-") + _Lang.getMessage("MSG_DEV_MODE"));
        }
    },

    /**
     * sends an event
     */
    sendEvent : function sendEvent(/*Object*/request, /*Object*/ context, /*event name*/ name) {
        var _Lang = myfaces._impl._util._Lang;
        var eventData = {};
        eventData.type = this.EVENT;

        eventData.status = name;
        eventData.source = context.source;

        if (name !== this.BEGIN) {

            try {
                //we bypass a problem with ie here, ie throws an exception if no status is given on the xhr object instead of just passing a value
                var getValue = function(value, key) {
                    try {
                        return value[key]
                    } catch (e) {
                        return "unkown";
                    }
                };

                eventData.responseCode = getValue(request, "status");
                eventData.responseText = getValue(request, "responseText");
                eventData.responseXML  = getValue(request, "responseXML");

            } catch (e) {
                var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
                impl.sendError(request, context, this.CLIENT_ERROR, "ErrorRetrievingResponse",
                        _Lang.getMessage("ERR_CONSTRUCT", e.toString()));
                     
                //client errors are not swallowed
                throw e;
            }

        }

        /**/
        if (context.onevent) {
            /*calling null to preserve the original scope*/
            context.onevent.call(null, eventData);
        }

        /*now we serve the queue as well*/
        this._evtListeners.broadcastEvent(eventData);
    },

    /**
     * processes the ajax response if the ajax request completes successfully
     * @param {Object} request (xhrRequest) the ajax request!
     * @param {Object} context (Map) context map keeping context data not being passed down over
     * the request boundary but kept on the client
     */
    response : function(request, context) {
        this._transport.response(request, context);
    },

    /**
     * @return the project stage also emitted by the server:
     * it cannot be cached and must be delivered over the server
     * The value for it comes from the request parameter of the jsf.js script called "stage".
     */
    getProjectStage : function() {
        /* run through all script tags and try to find the one that includes jsf.js */
        var scriptTags = document.getElementsByTagName("script");
        var getConfig  = myfaces._impl.core._Runtime.getGlobalConfig;
        
        for (var i = 0; i < scriptTags.length; i++) {
            if (scriptTags[i].src.search(/\/javax\.faces\.resource\/jsf\.js.*ln=javax\.faces/) != -1) {
                var result = scriptTags[i].src.match(/stage=([^&;]*)/);
                if (result) {
                    // we found stage=XXX
                    // return only valid values of ProjectStage
                    if (   result[1] == "Production"
                        || result[1] == "Development"
                        || result[1] == "SystemTest"
                        || result[1] == "UnitTest") {
                        return result[1];
                    }
                }
                else {
                    //we found the script, but there was no stage parameter -- Production
                    //(we also add an override here for testing purposes, the default, however is Production)
                    return getConfig("projectStage", "Production");
                    //return "Production";
                }
            }
        }
        /* we could not find anything valid --> return the default value */
        return getConfig("projectStage", "Production");
    },

    /**
     * implementation of the external chain function
     * moved into the impl
     *
     *  @param {Object} source the source which also becomes
     * the scope for the calling function (unspecified side behavior)
     * the spec states here that the source can be any arbitrary code block.
     * Which means it either is a javascript function directly passed or a code block
     * which has to be evaluated separately.
     *
     * After revisiting the code additional testing against components showed that
     * the this parameter is only targeted at the component triggering the eval
     * (event) if a string code block is passed. This is behavior we have to resemble
     * in our function here as well, I guess.
     *
     * @param {Event} event the event object being passed down into the the chain as event origin
     *   the spec is contradicting here, it on one hand defines event, and on the other
     *   it says it is optional, after asking, it meant that event must be passed down
     *   but can be undefined
     */
    chain : function(source, event) {
        var len   = arguments.length;
        var _Lang = this._Lang;

        //the spec is contradicting here, it on one hand defines event, and on the other
        //it says it is optional, I have cleared this up now
        //the spec meant the param must be passed down, but can be 'undefined'
        if (len < 2) {
            throw new Error(_Lang.getMessage("ERR_EV_OR_UNKNOWN"));
        } else if (len < 3) {
            if ('function' == typeof event || this._Lang.isString(event)) {

                throw new Error(_Lang.getMessage("ERR_EVT_PASS"));
            }
            //nothing to be done here, move along
            return true;
        }
        //now we fetch from what is given from the parameter list
        //we cannot work with splice here in any performant way so we do it the hard way
        //arguments only are give if not set to undefined even null values!

        //assertions source either null or set as dom element:

        if ('undefined' == typeof source) {
            throw new Error(_Lang.getMessage("ERR_SOURCE_DEF_NULL"));
            //allowed chain datatypes
        } else if ('function' == typeof source) {
            throw new Error(_Lang.getMessage("ERR_SOURCE_FUNC"));
        }
        if (this._Lang.isString(source)) {
            throw new Error(_Lang.getMessage("ERR_SOURCE_NOSTR"));
        }

        //assertion if event is a function or a string we already are in our function elements
        //since event either is undefined, null or a valid event object

        if ('function' == typeof event || this._Lang.isString(event)) {
            throw new Error(_Lang.getMessage("ERR_EV_OR_UNKNOWN"));
        }

        for (var cnt = 2; cnt < len; cnt++) {
            //we do not change the scope of the incoming functions
            //but we reuse the argument array capabilities of apply
            var ret;

            if ('function' == typeof arguments[cnt]) {
                ret = arguments[cnt].call(source, event);
            } else {
                //either a function or a string can be passed in case of a string we have to wrap it into another function
                ret = new Function("event", arguments[cnt]).call(source, event);
            }
            //now if one function returns false in between we stop the execution of the cycle
            //here, note we do a strong comparison here to avoid constructs like 'false' or null triggering
            if (ret === false /*undefined check implicitly done here by using a strong compare*/) {
                return false;
            }
        }
        return true;

    }
});    

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

/**
 *MyFaces core javascripting libraries
 *
 *  Those are the central public API functions in the JSF2
 *  Ajax API! They handle the entire form submit and ajax send
 *  and resolve cycle!
 */

/**
 * reserve the root namespace
 */
if ('undefined' != typeof OpenAjax && ('undefined' == typeof jsf || null == typeof jsf)) {
    OpenAjax.hub.registerLibrary("jsf", "www.sun.com", "1.0", null);
}
//just in case openajax has failed (testing environment)
//under normal circumstances this should not happen
//either check for window.jsf == true or do it the verbose way
if (!window.jsf) {
    window.jsf = new function() {
        /*
         * specified by the spec symbols/jsf.html#.specversion
         * as specified left two digits major release number
         * middle two digits minor spec release number
         * right two digits bug release number
         */
        this.specversion = 200000;
        /**
         * specified by the spec symbols/jsf.html#.implversion
         * a number increased with every implementation version
         * and reset by moving to a new spec release number
         *
         * Due to the constraints that we cannot put
         * non jsf.&lt;namespace&gt; references outside of functions in the api
         * we have to set the version here instead of the impl.
         */
        this.implversion = 2;

        /**
         * @return the current project state emitted by the server side method:
         * javax.faces.application.Application.getProjectStage()
         */
        this.getProjectStage = function() {
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.getProjectStage();
        };

        /**
         * collect and encode data for a given form element (must be of type form)
         * find the javax.faces.ViewState element and encode its value as well!
         * return a concatenated string of the encoded values!
         *
         * @throws an exception in case of the given element not being of type form!
         * https://issues.apache.org/jira/browse/MYFACES-2110
         */
        this.getViewState = function(formElement) {
            /*we are not allowed to add the impl on a global scope so we have to inline the code*/
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.getViewState(formElement);
        };
    };
}

/**
 * just to make sure no questions arise, I simply prefer here a weak
 * typeless comparison just in case some frameworks try to interfere
 * by overriding null or fiddeling around with undefined or typeof in some ways
 * it is safer in this case than the standard way of doing a strong comparison
 **/
if (!jsf.ajax) {
    jsf.ajax = new function() {


        /**
         * this function has to send the ajax requests
         *
         * following request conditions must be met:
         * <ul>
         *  <li> the request must be sent asynchronously! </li>
         *  <li> the request must be a POST!!! request </li>
         *  <li> the request url must be the form action attribute </li>
         *  <li> all requests must be queued with a client side request queue to ensure the request ordering!</li>
         * </ul>
         *
         * @param {String|Node} element: any dom element no matter being it html or jsf, from which the event is emitted
         * @param {|EVENT|} event: any javascript event supported by that object
         * @param {Map||} options : map of options being pushed into the ajax cycle
         */
        this.request = function(element, event, options) {
            if (!options) {
                options = {};
            }
            /*we are not allowed to add the impl on a global scope so we have to inline the code*/
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.request(element, event, options);
        };

        this.addOnError = function(/*function*/errorListener) {
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.addOnError(errorListener);
        };

        this.addOnEvent = function(/*function*/eventListener) {
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.addOnEvent(eventListener);
        };

        /**
         * processes the ajax response if the ajax request completes successfully
         * @param request the ajax request!
         * @param context the ajax context!
         */
        this.response = function(/*xhr request object*/request, context) {
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.response(request, context);
        };
    }
}

if (!jsf.util) {
    jsf.util = new function() {

        /**
         * varargs function which executes a chain of code (functions or any other code)
         *
         * if any of the code returns false, the execution
         * is terminated prematurely skipping the rest of the code!
         *
         * @param {DomNode} source, the callee object
         * @param {Event} event, the event object of the callee event triggering this function
         *
         */
        this.chain = function(source, event) {
            var impl = myfaces._impl.core._Runtime.getGlobalConfig("jsfAjaxImpl", myfaces._impl.core.Impl);
            return impl.chain.apply(impl, arguments);
        };
    }
}



